/*!
\file
\brief Application Estimator

The general task of estimation is defined on the following scheme:
\dot
digraph estimation{
	node [shape=box];
	{rank="same"; "Data Source"; "Bayesian Model"}
	"Data Source" -> "Bayesian Model" [label="data"];
	"Bayesian Model" -> "Result Logger" [label="estimated\n statistics"];
	"Data Source" -> "Result Logger" [label="Simulated\n data"];
}
\enddot

Here,
\li Data Source is an object (class DS) providing sequential data, \f$ [d_1, d_2, \ldots d_t] \f$.
\li Bayesian Model is an object (class BM) performing Bayesian filtering,
\li Result Logger is an object (class logger) dedicated to storing important data from the experiment.

\section  cmd Command-line usage
Execute command:
\code
$> estimator config_file.cfg
\endcode

Full description of the experiment is in the file config_file.cfg which is expected to have the following structure:
\code
system = {type = "DS_offspring", ...};      // definition of a data source
estimator = {type = "BM_offspring", ...};   // definition of an estimator
logger = {type = "logger_type",...};        // definition of a logger
experiment = {ndat = 11000; };              // definition of number of data records
\endcode

The above description must be specialized to specific classes. See, \subpage arx_ui how to do it for estimation of an ARX model.

\section ex Matlab usage
Execute command:
\code
>> estimator
\endcode
and follow the help there.

 */

#include <estim/arx_ext.h>
#include <stat/emix.h>
#include <base/datasources.h>
#include <base/loggers.h>
#include <estim/particles.h>
#include <estim/kalman.h>

//#include "mex_datasource.h"

using namespace bdm;

#ifdef MEX
#include <itpp/itmex.h>
#include <mex/mex_BM.h>
#include <mex/mex_logger.h>
#include <mex/mex_datasource.h>
#include <mex/mex_function.h>

void mexFunction ( int n_output, mxArray *output[], int n_input, const mxArray *input[] ) {
	// Check the number of inputs and output arguments
	if ( n_input<2 ) mexErrMsgTxt ( "Usage:\n"
		                                "[Res,estimators,Res2]=estimator(system, estimators, experiment, logger)\n"
		                                "  system     = struct('class','datasource',...);  % Estimated system\n"
		                                "  estimators = {struct('class','estimator',...),  % Estimators\n"
		                                "                struct('class','estimator',...),...} \n"
		                                "  === optional ==="
		                                "  experiment = struct('ndat',10);                 % number of data in experiment, full length of finite datasources, 10 otherwise \n"
		                                "  logger     = struct('class','mexlogger');       % How to store results, default=mexlog, i.e. matlab structure\n\n"
		                                "Output:\n"
		                                "  Res          Matlab structure with logged results, \n"  
		                                "  estimators   Array of estimators updated with data \n"
		                                "  Res2         When logfull log_level is on, this structure is filled with structures of posterior densities\n\n"
		                                "see documentation of classes datasource, BM, and mexlogger and their offsprings in BDM." );

	RV::clear_all();
	//CONFIG
	UImxArray Cfg;
	try {
		Cfg.addGroup ( input[0],"system" );
		Cfg.addList ( input[1],"estimators" );
		if ( n_input>2 ) {
			Cfg.addGroup ( input[2],"experiment" );
		}
		if ( n_input>3 ) {
			Cfg.addGroup ( input[3],"logger" );
		}
	} catch ( SettingException e ) {
		it_error ( "error: "+string ( e.getPath() ) );
	}

	//DBG
	//Cfg.writeFile ( "estimator.cfg" )

#else
int main ( int argc, char* argv[] ) {
	const char *fname;
	if ( argc>1 ) {
		fname = argv[1];
	} else {
		fname="estimator.cfg";
	}
	UIFile Cfg ( fname );
#endif

	shared_ptr<DS> Ds = UI::build<DS> ( Cfg, "system" );
	Array<shared_ptr<BM> > Es;
	UI::get ( Es,Cfg, "estimators" );
	int Ndat=10;
	if ( Cfg.exists ( "experiment" ) ) {
		if (  UI::get(Ndat, Cfg.getRoot()["experiment"],"ndat" ) ) {
			bdm_assert ( Ndat<=Ds->max_length(), "Data source has less data then required" );
		};
	} else {
		if ( Ds->max_length() < std::numeric_limits< int >::max() ) {
			Ndat=Ds->max_length();
		}
		;// else Ndat=10;
	}
	shared_ptr<logger> L = UI::build<logger> ( Cfg, "logger",UI::optional );
	if ( !L ) {
#ifdef MEX
		//mex logger has only from_setting constructor - we  have to fill it...
		L=new mexlog ( Ndat );
#else
		L=new stdlog();
#endif
	}

	Ds->log_register ( *L, "DS" );
	string Ename;
	Setting &S=Cfg;
	for ( int i=0; i<Es.length(); i++ ) {
		if (!UI::get ( Ename, S["estimators"][i], "name" ,UI::optional)){
			Ename="Est"+num2str ( i );
		}
		if (!Es(i)->posterior().log_level[epdf::logmean]){
			const_cast<epdf&>(Es (i) ->posterior()).log_level[epdf::logmean] =true;
		}		
		Es ( i )->log_register ( *L,Ename ); // estimate
	}
	L->init();

	vec dt=zeros ( Ds->_drv()._dsize() );   //data variable
	Array<datalink*> Dls ( Es.length() );
	Array<datalink*> Dlsc ( Es.length() );
	Array<datalink_buffered*> Dls_buf (0);
	for ( int i=0; i<Es.length(); i++ ) {
		//check if they are properly named
		bdm_assert(Es(i)->_yrv()._dsize() == Es(i)->dimensiony(), "Estimator["+num2str(i)+"] does not name yrv properly."
		"size(yrv)="+num2str(Es(i)->_yrv()._dsize() ) + ", declared dimension of y="+num2str(Es(i)->dimensiony()));
		bdm_assert(Es(i)->_rvc()._dsize() == Es(i)->dimensionc(), "Estimator["+num2str(i)+"] does not name rvc properly."
		"size(rvc)="+num2str(Es(i)->_rvc()._dsize() ) + ", declared dimension of rvc="+num2str(Es(i)->dimensionc()));
		//connect actual data
		Dls ( i ) = new datalink ( Es ( i )->_yrv(),Ds->_drv() ); //datalink between a datasource and estimator
		//connect data in condition
		if (Es ( i )->_rvc().mint()<0){ 
			//delayed values are required
			
			//create delayed dl
			int ith_buf=Dls_buf.size();
			Dls_buf.set_size( ith_buf + 1, true);
			Dls_buf(ith_buf) = new datalink_buffered(); 
			//add dl to list of buffered DS
			Dlsc(i) = Dls_buf(ith_buf);
			Dlsc(i)->set_connection ( Es ( i )->_rvc(),Ds->_drv() ); //datalink between a datasource and estimator
			
			bdm_assert_debug(Dlsc(i)->_downsize() == Es ( i )->_rvc()._dsize(), "Data required by Est[" + num2str(i) + "], " + 
			Es(i)->_rvc().to_string() + ", are not available in DS drv:" + Ds->_drv().to_string(););
		} else {
			Dlsc ( i ) = new datalink ( Es ( i )->_rvc(),Ds->_drv() ); //datalink between a datasource and estimator
		}
	}

	for ( int tK=0;tK<Ndat;tK++ ) {
		Ds->getdata ( dt );					// read data
		Ds->log_write ( );

		for ( int i=0; i<Es.length(); i++ ) {
			if (tK + Es ( i )->_rvc().mint() > 0 ) {
				Es ( i )->bayes ( Dls ( i )->pushdown ( dt ), Dlsc(i) ->pushdown(dt) );		// update estimates
			}
			Es ( i )->log_write ();
		}
		
		L->step();
		Ds->step();							// simulator step
		//update buffered fdat links
		for (int i=0; i<Dls_buf.length(); i++){
			Dls_buf(i)->store_data(dt);
		}
			
	}

	L->finalize();
	// ------------------ End of routine -----------------------------

#ifdef MEX
	mexlog* mL=dynamic_cast<mexlog*> ( L.get() );

	if ( mL ) { // user wants output!!
		if ( n_output<1 ) mexErrMsgTxt ( "Wrong number of output variables!" );
		output[0] = mL->toCell();
		if (n_output>1){ // write estimators
			UImxArray Ests;
			UI::save(Es, Ests,"estimators");
			output[1]=UImxArray::create_mxArray(Ests);
		}
		if (n_output>2) {
			mL->_setting_conf().setAutoConvert(true);
			output[2]= UImxArray::create_mxArray(mL->_setting_conf().getRoot());
		}
	}
#endif
	for (int i=0;i<Dls.length(); i++){delete Dls(i); delete Dlsc(i);}
	UIFile F;
	UI::save(Es, F.getRoot(),"estimators");
	F.writeFile("estim_out.cfg");
}
