/*!
  \file
  \brief Robust Bayesian auto-regression model
  \author Jan Sindelar.
*/

#ifndef ROBUST_H
#define ROBUST_H

#include <stat/exp_family.h>
#include <limits>
#include <vector>
#include <algorithm>
	
using namespace bdm;
using namespace std;

const double max_range = numeric_limits<double>::max()/10e-5;

class polyhedron;
class vertex;

/// A class describing a single polyhedron of the split complex. From a collection of such classes a Hasse diagram
/// of the structure in the exponent of a Laplace-Inverse-Gamma density will be created.
class polyhedron
{
	/// A property having a value of 1 usually, with higher value only if the polyhedron arises as a coincidence of
	/// more than just the necessary number of conditions. For example if a newly created line passes through an already
	/// existing point, the points multiplicity will rise by 1.
	int multiplicity;	

public:
	/// A list of polyhedrons parents within the Hasse diagram.
	vector<polyhedron*> parents;

	/// A list of polyhedrons children withing the Hasse diagram.
	vector<polyhedron*> children;

	/// All the vertices of the given polyhedron
	vector<vertex*> vertices;

	/// A list used for storing children that lie in the positive region related to a certain condition
	vector<polyhedron*> positivechildren;

	/// A list used for storing children that lie in the negative region related to a certain condition
	vector<polyhedron*> negativechildren;

	/// Children intersecting the condition
	vector<polyhedron*> neutralchildren;

	/// List of triangulation polyhedrons of the polyhedron given by their relative vertices. 
	vector<vector<vertex*>> triangulations;

	/// A list of relative addresses serving for Hasse diagram construction.
	vector<int> kids_rel_addresses;

	/// Default constructor
	polyhedron()
	{
		multiplicity = 1;	
	}
	
	/// Setter for raising multiplicity
	void RaiseMultiplicity()
	{
		multiplicity++;
	}

	/// Setter for lowering multiplicity
	void LowerMultiplicity()
	{
		multiplicity--;
	}
	
	/// An obligatory operator, when the class is used within a C++ STL structure like a vector
	int operator==(polyhedron polyhedron2)
	{
		return true;
	}

	/// An obligatory operator, when the class is used within a C++ STL structure like a vector
	int operator<(polyhedron polyhedron2)
	{
		return false;
	}
};

/// A
class vertex : public polyhedron
{
	vector<double> coordinates;

public:

	vertex();

	vertex(vector<double> coordinates)
	{
		this->coordinates = coordinates;
	}

	void push_coordinate(double coordinate)
	{
		coordinates.push_back(coordinate);
	}

	vector<double> get_coordinates()
	{
		vector<double> returned_vec;

		copy(this->coordinates.begin(),this->coordinates.end(),returned_vec.begin());

		return returned_vec;
	}
};

class toprow : public polyhedron
{
	vector<double> condition;

public:

	toprow();

	toprow(vector<double> condition)
	{
		this->condition = condition;
	}

};




//! Conditional(e) Multicriteria-Laplace-Inverse-Gamma distribution density
class emlig : public eEF
{

	vector<vector<polyhedron*>> statistic;
	
public:	

	emlig(int number_of_parameters)
	{
		create_statistic(number_of_parameters);
	}

	emlig(vector<vector<polyhedron*>> statistic)
	{
		this->statistic = statistic;
	}



protected:

    void create_statistic(int number_of_parameters)
	{
		vector<double> origin_coord;	

		vertex *origin = new vertex(origin_coord);

		origin->vertices.push_back(origin);

		vector<polyhedron*> origin_vec;

		origin_vec.push_back(origin);

		statistic.push_back(origin_vec);

		for(int i=0;i<number_of_parameters;i++)
		{
			vector<double> origin_coord1 = origin->get_coordinates();
			vector<double> origin_coord2 = origin->get_coordinates();

			origin->push_coordinate(0);

			origin_coord1.push_back(max_range);
			origin_coord2.push_back(-max_range);

			vertex *new_point1 = new vertex(origin_coord1);
			vertex *new_point2 = new vertex(origin_coord2);	

			vector<vector<polyhedron*>> new_statistic1;
			vector<vector<polyhedron*>> new_statistic2;

			
			for(int j=0;j<statistic.size();j++)
			{
				int element_number = 0;

				for(vector<polyhedron*>::iterator horiz_ref = statistic[j].begin();horiz_ref<statistic[j].end();horiz_ref++)
				{	
					if(!(*horiz_ref)->parents.empty())
					{
						for(vector<polyhedron*>::iterator parent_ref = (*horiz_ref)->parents.begin();parent_ref < (*horiz_ref)->parents.end();parent_ref++)
						{
							(*parent_ref)->kids_rel_addresses.push_back(element_number);							
						}						
					}

					vector<double> vec1(i+2,0);
					vector<double> vec2(i+2,0);

					toprow *current_copy1 = new toprow(vec1);
					toprow *current_copy2 = new toprow(vec2);					

					for(vector<vertex*>::iterator vert_ref = (*horiz_ref)->vertices.begin();vert_ref<(*horiz_ref)->vertices.end();vert_ref++)
					{
						current_copy1->vertices.push_back(*vert_ref);
						current_copy2->vertices.push_back(*vert_ref);						
					}
					

					current_copy1->vertices.push_back(new_point1);
					current_copy2->vertices.push_back(new_point2);

					current_copy1->triangulations.push_back(current_copy1->vertices);
					current_copy2->triangulations.push_back(current_copy2->vertices);

					

					if(!(*horiz_ref)->kids_rel_addresses.empty())
					{
						for(vector<int>::iterator kid_ref = (*horiz_ref)->kids_rel_addresses.begin();kid_ref<(*horiz_ref)->kids_rel_addresses.end();kid_ref++)
						{						
							current_copy1->children.push_back(new_statistic1[i][(*kid_ref)]);
							current_copy2->children.push_back(new_statistic2[i][(*kid_ref)]);

							new_statistic1[i][(*kid_ref)]->parents.push_back(current_copy1);
							new_statistic2[i][(*kid_ref)]->parents.push_back(current_copy2);
						}						

						(*horiz_ref)->kids_rel_addresses.clear();
					}
					else
					{
						current_copy1->children.push_back(new_point1);
						current_copy2->children.push_back(new_point2);

						new_point1->parents.push_back(current_copy1);
						new_point2->parents.push_back(current_copy2);
					}

					current_copy1->children.push_back(*horiz_ref);
					current_copy2->children.push_back(*horiz_ref);

					new_statistic1[i+1].push_back(current_copy1);
					new_statistic2[i+1].push_back(current_copy2);
					
					element_number++;
				}				
			}				
			
		}
	}


	
	
};

//! Robust Bayesian AR model for Multicriteria-Laplace-Inverse-Gamma density
class RARX : public BMEF{
};


#endif //TRAGE_H
