
#include "kalman.h"

namespace bdm {

using std::endl;



void KalmanFull::bayes ( const vec &yt, const vec &cond ) {
    bdm_assert_debug ( yt.length() == ( dimy ), "KalmanFull::bayes wrong size of dt, " +
                       num2str(yt.length()) + ", expected size is " + num2str(dimy) );
    bdm_assert_debug ( cond.length() == ( dimc ), "KalmanFull::bayes wrong size of cond, " +
                       num2str(cond.length()) + ", expected size is " + num2str(dimc) );

    const vec &u = cond; // in this case cond=ut
    const vec &y = yt;

    vec& mu = est._mu();
    mat &P = est._R();
    mat& _Ry = fy._R();
    vec& yp = fy._mu();
    //Time update
    mu = A * mu + B * u;
    P  = A * P * A.transpose() + ( mat ) Q;

    //Data update
    _Ry = C * P * C.transpose() + ( mat ) R;
    _K = P * C.transpose() * inv ( _Ry );
    P -= _K * C * P; // P = P -KCP;
    yp = C * mu + D * u;
    mu += _K * ( y - yp );

    if ( evalll ) {
        ll = fy.evallog ( y );
    }
};



/////////////////////////////// EKFS
EKFfull::EKFfull ( ) : KalmanFull () {};

void EKFfull::set_parameters ( const shared_ptr<diffbifn> &pfxu0, const shared_ptr<diffbifn> &phxu0, const mat Q0, const mat R0 ) {
    pfxu = pfxu0;
    phxu = phxu0;

    set_dim ( pfxu0->_dimx() );
    dimy = phxu0->dimension();
    dimc = pfxu0->_dimu();
    est.set_parameters ( zeros ( dimension() ), eye ( dimension() ) );

    A.set_size ( dimension(), dimension() );
    C.set_size ( dimy, dimension() );
    //initialize matrices A C, later, these will be only updated!
    pfxu->dfdx_cond ( est._mu(), zeros ( dimc ), A, true );
    B.clear();
    phxu->dfdx_cond ( est._mu(), zeros ( dimc ), C, true );
    D.clear();

    R = R0;
    Q = Q0;
}

void EKFfull::bayes ( const vec &yt, const vec &cond ) {
    bdm_assert_debug ( yt.length() == ( dimy ), "EKFull::bayes wrong size of dt" );
    bdm_assert_debug ( cond.length() == ( dimc ), "EKFull::bayes wrong size of dt" );

    const vec &u = cond;
    const vec &y = yt; //lazy to change it
    vec &mu = est._mu();
    mat &P = est._R();
    mat& _Ry = fy._R();
    vec& yp = fy._mu();

    pfxu->dfdx_cond ( mu, zeros ( dimc ), A, true );
    phxu->dfdx_cond ( mu, zeros ( dimc ), C, true );

    //Time update
    mu = pfxu->eval ( mu, u );// A*mu + B*u;
    P  = A * P * A.transpose() + ( mat ) Q;

    //Data update
    _Ry = C * P * C.transpose() + ( mat ) R;
    _K = P * C.transpose() * inv ( _Ry );
    P -= _K * C * P; // P = P -KCP;
    yp = phxu->eval ( mu, u );
    mu += _K * ( y - yp );

    if ( BM::evalll ) {
        ll = fy.evallog ( y );
    }
};



void KalmanCh::set_parameters ( const mat &A0, const mat &B0, const mat &C0, const mat &D0, const chmat &Q0, const chmat &R0 ) {

    ( ( StateSpace<chmat>* ) this )->set_parameters ( A0, B0, C0, D0, Q0, R0 );

    _K = zeros ( dimension(), dimy );
}

void KalmanCh::initialize() {
    preA = zeros ( dimy + dimension() + dimension(), dimy + dimension() );
//	preA.clear();
    preA.set_submatrix ( 0, 0, R._Ch() );
    preA.set_submatrix ( dimy + dimension(), dimy, Q._Ch() );
}

void KalmanCh::bayes ( const vec &yt, const vec &cond ) {
    bdm_assert_debug ( yt.length() == dimy, "yt mismatch" );
    bdm_assert_debug ( cond.length() == dimc, "yt mismatch" );

    const vec &u = cond;
    const vec &y = yt;
    vec pom ( dimy );

    chmat &_P = est._R();
    vec &_mu = est._mu();
    mat _K ( dimension(), dimy );
    chmat &_Ry = fy._R();
    vec &_yp = fy._mu();
    //TODO get rid of Q in qr()!
    //	mat Q;

    //R and Q are already set in set_parameters()
    preA.set_submatrix ( dimy, 0, ( _P._Ch() ) *C.T() );
    //Fixme can be more efficient if .T() is not used
    preA.set_submatrix ( dimy, dimy, ( _P._Ch() ) *A.T() );

    if ( !qr ( preA, postA ) ) {
        bdm_warning ( "QR in KalmanCh unstable!" );
    }

    ( _Ry._Ch() ) = postA ( 0, dimy - 1, 0, dimy - 1 );
    _K = inv ( A ) * ( postA ( 0, dimy - 1 , dimy, dimy + dimension() - 1 ) ).T();
    ( _P._Ch() ) = postA ( dimy, dimy + dimension() - 1, dimy, dimy + dimension() - 1 );

    _mu = A * ( _mu ) + B * u;
    _yp = C * _mu - D * u;

    backward_substitution ( _Ry._Ch(), ( y - _yp ), pom );
    _mu += ( _K ) * pom;

    /*		cout << "P:" <<_P.to_mat() <<endl;
    		cout << "Ry:" <<_Ry.to_mat() <<endl;
    		cout << "_K:" <<_K <<endl;*/

    if ( evalll == true ) { //likelihood of observation y
        ll = fy.evallog ( y );
    }
}

void StateCanonical::connect_mlnorm ( const mlnorm<fsqmat> &ml ) {
    //get ids of yrv
    const RV &yrv = ml._rv();
    //need to determine u_t - it is all in _rvc that is not in ml._rv()
    RV rgr0 = ml._rvc().remove_time();
    RV urv = rgr0.subt ( yrv );

    //We can do only 1d now... :(
    bdm_assert ( yrv._dsize() == 1, "Only for SISO so far..." );

    // create names for
    RV xrv; //empty
    RV Crv; //empty
    int td = ml._rvc().mint();
    // assuming strictly proper function!!!
    for ( int t = -1; t >= td; t-- ) {
        xrv.add ( yrv.copy_t ( t ) );
        Crv.add ( urv.copy_t ( t ) );
    }

    // get mapp
    th2A.set_connection ( xrv, ml._rvc() );
    th2C.set_connection ( Crv, ml._rvc() );
    th2D.set_connection ( urv, ml._rvc() );

    //set matrix sizes
    this->A = zeros ( xrv._dsize(), xrv._dsize() );
    for ( int j = 1; j < xrv._dsize(); j++ ) {
        A ( j, j - 1 ) = 1.0;    // off diagonal
    }
    this->B = zeros ( xrv._dsize(), 1 );
    this->B ( 0 ) = 1.0;
    this->C = zeros ( 1, xrv._dsize() );
    this->D = zeros ( 1, urv._dsize() );
    this->Q = zeros ( xrv._dsize(), xrv._dsize() );
    // R is set by update

    //set cache
    this->A1row = zeros ( xrv._dsize() );
    this->C1row = zeros ( xrv._dsize() );
    this->D1row = zeros ( urv._dsize() );

    update_from ( ml );
    validate();
};

void StateCanonical::update_from ( const mlnorm<fsqmat> &ml ) {

    vec theta = ml._A().get_row ( 0 ); // this

    th2A.filldown ( theta, A1row );
    th2C.filldown ( theta, C1row );
    th2D.filldown ( theta, D1row );

    R = ml._R();

    A.set_row ( 0, A1row );
    C.set_row ( 0, C1row + D1row ( 0 ) *A1row );
    D.set_row ( 0, D1row );
}

void StateFromARX::connect_mlnorm ( const mlnorm<chmat> &ml, RV &xrv, RV &urv ) {
    //get ids of yrv
    const RV &yrv = ml._rv();
    //need to determine u_t - it is all in _rvc that is not in ml._rv()
    const RV &rgr = ml._rvc();
    RV rgr0 = rgr.remove_time();
    urv = rgr0.subt ( yrv );

    // create names for state variables
    xrv = yrv;

    int y_multiplicity = -rgr.mint ( yrv );
    int y_block_size = yrv.length() * ( y_multiplicity ); // current yt + all delayed yts
    for ( int m = 0; m < y_multiplicity - 1; m++ ) { // ========= -1 is important see arx2statespace_notes
        xrv.add ( yrv.copy_t ( -m - 1 ) ); //add delayed yt
    }
    //! temporary RV for connection to ml.rvc, since notation of xrv and ml.rvc does not match
    RV xrv_ml = xrv.copy_t ( -1 );

    // add regressors
    ivec u_block_sizes ( urv.length() ); // no_blocks = yt + unique rgr
    for ( int r = 0; r < urv.length(); r++ ) {
        RV R = urv.subselect ( vec_1 ( r ) ); //non-delayed element of rgr
        int r_size = urv.size ( r );
        int r_multiplicity = -rgr.mint ( R );
        u_block_sizes ( r ) = r_size * r_multiplicity ;
        for ( int m = 0; m < r_multiplicity; m++ ) {
            xrv.add ( R.copy_t ( -m - 1 ) ); //add delayed yt
            xrv_ml.add ( R.copy_t ( -m - 1 ) ); //add delayed yt
        }
    }
    // add constant
    if ( any ( ml._mu_const() != 0.0 ) ) {
        have_constant = true;
        xrv.add ( RV ( "bdm_reserved_constant_one", 1 ) );
    } else {
        have_constant = false;
    }


    // get mapp
    th2A.set_connection ( xrv_ml, ml._rvc() );
    th2B.set_connection ( urv, ml._rvc() );

    //set matrix sizes
    this->A = zeros ( xrv._dsize(), xrv._dsize() );
    //create y block
    diagonal_part ( this->A, yrv._dsize(), 0, y_block_size - yrv._dsize() );

    this->B = zeros ( xrv._dsize(), urv._dsize() );
    //add diagonals for rgr
    int active_x = y_block_size;
    int active_Bcol = 0;
    for ( int r = 0; r < urv.length(); r++ ) {
        if (u_block_sizes(r)>0) {
            diagonal_part ( this->A, active_x + urv.size ( r ), active_x, u_block_sizes ( r ) - urv.size ( r ) );
            this->B.set_submatrix ( active_x, active_Bcol, eye ( urv.size ( r ) ) );
            active_Bcol+=u_block_sizes(r);
        }
        active_x += u_block_sizes ( r );
    }
    this->C = zeros ( yrv._dsize(), xrv._dsize() );
    this->C.set_submatrix ( 0, 0, eye ( yrv._dsize() ) );
    this->D = zeros ( yrv._dsize(), urv._dsize() );
    this->R.setCh ( zeros ( yrv._dsize(), yrv._dsize() ) );
    this->Q.setCh ( zeros ( xrv._dsize(), xrv._dsize() ) );
    // Q is set by update

    update_from ( ml );
    validate();
}

void StateFromARX::update_from ( const mlnorm<chmat> &ml ) {
    vec Arow = zeros ( A.cols() );
    vec Brow = zeros ( B.cols() );
    //  ROW- WISE EVALUATION =====
    for ( int i = 0; i < ml._rv()._dsize(); i++ ) {

        vec theta = ml._A().get_row ( i );

        th2A.filldown ( theta, Arow );
        if ( have_constant ) {
            // constant is always at the end no need for datalink
            Arow ( A.cols() - 1 ) = ml._mu_const() ( i );
        }
        this->A.set_row ( i, Arow );

        th2B.filldown ( theta, Brow );
        this->B.set_row ( i, Brow );
    }
    this->Q._Ch().set_submatrix ( 0, 0, ml.__R()._Ch() );

}


void EKFCh::set_parameters ( const shared_ptr<diffbifn> &pfxu0, const shared_ptr<diffbifn> &phxu0, const chmat Q0, const chmat R0 ) {
    pfxu = pfxu0;
    phxu = phxu0;

    set_dim ( pfxu0->_dimx() );
    dimy = phxu0->dimension();
    dimc = pfxu0->_dimu();

    vec &_mu = est._mu();
    // if mu is not set, set it to zeros, just for constant terms of A and C
    if ( _mu.length() != dimension() ) _mu = zeros ( dimension() );
    A = zeros ( dimension(), dimension() );
    C = zeros ( dimy, dimension() );
    preA = zeros ( dimy + dimension() + dimension(), dimy + dimension() );

    //initialize matrices A C, later, these will be only updated!
    pfxu->dfdx_cond ( _mu, zeros ( dimc ), A, true );
//	pfxu->dfdu_cond ( *_mu,zeros ( dimu ),B,true );
    B.clear();
    phxu->dfdx_cond ( _mu, zeros ( dimc ), C, true );
//	phxu->dfdu_cond ( *_mu,zeros ( dimu ),D,true );
    D.clear();

    R = R0;
    Q = Q0;

    // Cholesky special!
    preA.clear();
    preA.set_submatrix ( 0, 0, R._Ch() );
    preA.set_submatrix ( dimy + dimension(), dimy, Q._Ch() );
}


void EKFCh::bayes ( const vec &yt, const vec &cond ) {

    vec pom ( dimy );
    const vec &u = cond;
    const vec &y = yt;
    vec &_mu = est._mu();
    chmat &_P = est._R();
    chmat &_Ry = fy._R();
    vec &_yp = fy._mu();

    pfxu->dfdx_cond ( _mu, u, A, false ); //update A by a derivative of fx
    phxu->dfdx_cond ( _mu, u, C, false ); //update A by a derivative of fx

    //R and Q are already set in set_parameters()
    preA.set_submatrix ( dimy, 0, ( _P._Ch() ) *C.T() );
    //Fixme can be more efficient if .T() is not used
    preA.set_submatrix ( dimy, dimy, ( _P._Ch() ) *A.T() );

// 	mat Sttm = _P->to_mat();
//	cout << preA <<endl;
//	cout << "_mu:" << _mu <<endl;

    if ( !qr ( preA, postA ) ) {
        bdm_warning ( "QR in EKFCh unstable!" );
    }


    ( _Ry._Ch() ) = postA ( 0, dimy - 1, 0, dimy - 1 );
    _K = inv ( A ) * ( postA ( 0, dimy - 1 , dimy, dimy + dimension() - 1 ) ).T();
    ( _P._Ch() ) = postA ( dimy, dimy + dimension() - 1, dimy, dimy + dimension() - 1 );

// 	mat iRY = inv(_Ry->to_mat());
// 	mat Stt = Sttm - Sttm * C.T() * iRY * C * Sttm;
// 	mat _K2 = Stt*C.T()*inv(R.to_mat());

    // prediction
    _mu = pfxu->eval ( _mu , u );
    _yp = phxu->eval ( _mu, u );

    /*	vec mu2 = *_mu + ( _K2 ) * ( y-*_yp );*/

    //correction //= initial value is already prediction!
    backward_substitution ( _Ry._Ch(), ( y - _yp ), pom );
    _mu += ( _K ) * pom ;

    /*	_K = (_P->to_mat())*C.transpose() * ( _iRy->to_mat() );
    	*_mu = pfxu->eval ( *_mu ,u ) + ( _K )* ( y-*_yp );*/

// 		cout << "P:" <<_P.to_mat() <<endl;
// 		cout << "Ry:" <<_Ry.to_mat() <<endl;
// 	cout << "_mu:" <<_mu <<endl;
// 	cout << "dt:" <<dt <<endl;

    if ( evalll == true ) { //likelihood of observation y
        ll = fy.evallog ( y );
    }
}

void EKFCh::from_setting ( const Setting &set ) {
    BM::from_setting ( set );
    shared_ptr<diffbifn> IM = UI::build<diffbifn> ( set, "IM", UI::compulsory );
    shared_ptr<diffbifn> OM = UI::build<diffbifn> ( set, "OM", UI::compulsory );

    //statistics
    int dim = IM->dimension();
    vec mu0;
    if ( !UI::get ( mu0, set, "mu0" ) )
        mu0 = zeros ( dim );

    mat P0;
    vec dP0;
    if ( UI::get ( dP0, set, "dP0" ) )
        P0 = diag ( dP0 );
    else if ( !UI::get ( P0, set, "P0" ) )
        P0 = eye ( dim );

    set_statistics ( mu0, P0 );

    //parameters
    vec dQ, dR;
    UI::get ( dQ, set, "dQ", UI::compulsory );
    UI::get ( dR, set, "dR", UI::compulsory );
    set_parameters ( IM, OM, diag ( dQ ), diag ( dR ) );
}

void MultiModel::from_setting ( const Setting &set ) {
    Array<EKFCh*> A;
    UI::get ( A, set, "models", UI::compulsory );

    set_parameters ( A );
    set_yrv ( A ( 0 )->_yrv() );
    //set_rv(A(0)->_rv());
}

void EKF_UD::set_parameters ( const shared_ptr<diffbifn> &pfxu0, const shared_ptr<diffbifn> &phxu0, const mat Q0, const vec R0 ) {
	pfxu = pfxu0;
	phxu = phxu0;
	
	set_dim ( pfxu0->_dimx() );
	dimy = phxu0->dimension();
	dimc = pfxu0->_dimu();
	
	vec &_mu = est._mu();
	// if mu is not set, set it to zeros, just for constant terms of A and C
	if ( _mu.length() != dimension() ) _mu = zeros ( dimension() );
	A = zeros ( dimension(), dimension() );
	C = zeros ( dimy, dimension() );
	
	//initialize matrices A C, later, these will be only updated!
	pfxu->dfdx_cond ( _mu, zeros ( dimc ), A, true );
	//	pfxu->dfdu_cond ( *_mu,zeros ( dimu ),B,true );
	phxu->dfdx_cond ( _mu, zeros ( dimc ), C, true );
	//	phxu->dfdu_cond ( *_mu,zeros ( dimu ),D,true );
	
	R = R0;
	Q = Q0;
	
	// 
}


void EKF_UD::bayes ( const vec &yt, const vec &cond ) {
	//preparatory
	vec &_mu=est._mu();
	const vec &u=cond;
	int dim = dimension();
	
	U = est._R()._L().T();
	D =  est._R()._D();
	
	////////////
	
	pfxu->dfdx_cond ( _mu, u, A, false ); //update A by a derivative of fx
	phxu->dfdx_cond ( _mu, u, C, false ); //update A by a derivative of fx

	mat PhiU = A*U;
	
	vec Din = D;
	int i,j,k;
	double sigma;
	mat G = eye(dim);
	//////// thorton
	
	//move mean;
	_mu = pfxu->eval(_mu,u);
	
	for (i=dim-1; i>=0;i--){
		sigma = 0.0; 
		for (j=0; j<dim; j++) {
			sigma += PhiU(i,j)*PhiU(i,j) *Din(j); 
			sigma += G(i,j)*G(i,j) * Q(j,j); 
		}
		D(i) = sigma; 
		for (j=0;j<i;j++){ 
			sigma = 0.0; 
			for (k=0;k<dim;k++){ 
				sigma += PhiU(i,k)*Din(k)*PhiU(j,k); 
			}
			for (k=0;k<dim;k++){ 
				sigma += G(i,k)*Q(k,k)*G(j,k); 
			}
			//
			U(j,i) = sigma/D(i); 
			for (k=0;k<dim;k++){ 
				PhiU(j,k) = PhiU(j,k) - U(j,i)*PhiU(i,k); 
			}
			for (k=0;k<dim;k++){ 
				G(j,k) = G(j,k) - U(j,i)*G(i,k); 
			}
		}
	}
	
	if ( log_level[logU] ){
		// transformed U
		mat tU;
		mat P= U*diag(D)*U.T();
		
		vec xref(5);
		xref(0)= 30.0*1.4142;
		xref(1)= 30.0*1.4142;
		xref(2)= 6.283185*200.;
		xref(3) = 3.141593;
		xref(4) = 34.0;
		
		mat T = diag(1.0/(xref));
		mat Pf = T*P*T;
		
		ldmat Pld(Pf);
		
		//vec tmp=vec(U._data(),dimension()*dimension());
		vec tmp=vec(Pld._L()._data(),dimension()*dimension());
		log_level.store(logU,round(((int)1<<14)*tmp));
		log_level.store(logD,Pld._D());
	}
	if ( log_level[logG] ){
		vec tmp=vec(G._data(),dimension()*dimension());
		log_level.store(logG,tmp);
	}
	//cout << "Ut: " << U << endl;
	//cout << "Dt: " << D << endl;
	// bierman
	
	double dz,alpha,gamma,beta,lambda;
	vec a;
	vec b;
	vec yp = phxu->eval(_mu,u);
	for (int iy=0; iy<dimy; iy++){
		a     = U.T()*C.get_row(iy);    // a is not modified, but 
		b     = elem_mult(D,a);    			 // b is modified to become unscaled Kalman gain. 
		dz    = yt(iy) - yp(iy); 

		alpha = R(iy); 
		gamma = 1/alpha; 
		for (j=0;j<dim;j++){
			beta   = alpha; 
			alpha  = alpha + a(j)*b(j); 
			lambda = -a(j)*gamma; 
			gamma  = 1.0/alpha; 
			D(j) = beta*gamma*D(j); 

// 			cout << "a: " << alpha << "g: " << gamma << endl;
			for (i=0;i<j;i++){
				beta   = U(i,j); 
				U(i,j) = beta + b(i)*lambda; 
				b(i)   = b(i) + b(j)*beta; 
			}
		}
		double dzs = gamma*dz;  // apply scaling to innovations 
		_mu   = _mu + dzs*b; // multiply by unscaled Kalman gain 
		
		//cout << "Ub: " << U << endl;
		//cout << "Db: " << D << endl <<endl;
		
	}
		
/////
est._R().__L()=U.T();
est._R().__D()=D;

	if ( evalll == true ) { //likelihood of observation y
	}
}

void EKF_UD::from_setting ( const Setting &set ) {
	BM::from_setting ( set );
	shared_ptr<diffbifn> IM = UI::build<diffbifn> ( set, "IM", UI::compulsory );
	shared_ptr<diffbifn> OM = UI::build<diffbifn> ( set, "OM", UI::compulsory );
	
	//statistics
	int dim = IM->dimension();
	vec mu0;
	if ( !UI::get ( mu0, set, "mu0" ) )
		mu0 = zeros ( dim );
	
	mat P0;
	vec dP0;
	if ( UI::get ( dP0, set, "dP0" ) )
		P0 = diag ( dP0 );
	else if ( !UI::get ( P0, set, "P0" ) )
		P0 = eye ( dim );
	
	est._mu()=mu0;
	est._R()=ldmat(P0);
	
	//parameters
	vec dQ, dR;
	UI::get ( dQ, set, "dQ", UI::compulsory );
	UI::get ( dR, set, "dR", UI::compulsory );
	set_parameters ( IM, OM, diag ( dQ ), dR  );
	
	UI::get(log_level, set, "log_level", UI::optional);
}

}
