#ifndef UI_H
#define UI_H

#include <sstream>
#include <iostream>
#include <stdio.h>
#include <string>
#include <map>
#include <utility>
#include <xercesc/dom/DOM.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/util/XMLString.hpp>
#include <iostream>
#include <xercesc/framework/LocalFileFormatTarget.hpp>
#include <xercesc/framework/LocalFileInputSource.hpp>
#include <xercesc/dom/DOMWriter.hpp>
#include <xercesc/parsers/XercesDOMParser.hpp>

#ifdef XERCES_CPP_NAMESPACE_USE
XERCES_CPP_NAMESPACE_USE
#endif

using std::string;
using namespace std;


/*!
@brief Xerces interface class

This class is used to interact with the Xerces library.
*/
class GlobalXercesConnector
{
public:
	//! DOMImplementation is a base class for the all DOM oparations
	DOMImplementation *pImplementation;
	//! This DOMWriter is used to export internal data into xml files
	DOMWriter *pSerializer;
	//! This DOMWriter is used to import external data from xml files
	XercesDOMParser *pParser;


	//!default constructor
	GlobalXercesConnector();
	//!destructor
	~GlobalXercesConnector();


	//! function transcodes Xerces' XMLCh-based strings into C++ strings
	static string XMLCh2str( const XMLCh* const  XMLCh_str );

	class Comparator
	{
	public:
		//! operator compares two XMLCh strings and wheather the first is alphabethically higher
		bool operator()( const XMLCh* const a , const XMLCh* const b) const;
	};
};

//! The only global instance of the GlobalXercesConnector class
extern const GlobalXercesConnector XMLConnector;


/*!
@brief User Info base class

This class is used to store information about parameters of an object. It support loading and saving of the information and, potentially, interaction with the user.
*/

class Attribute 
{
private:
	//! Name which identifies the attribute among others related to the same XML-tag, 
	//! must be unique within its context!
	XMLCh* const transcodedName;

public:
	//! Type definition of mapping which transforms names to the related attributes 
	typedef map<const XMLCh* const, Attribute* const, GlobalXercesConnector::Comparator> MappedAttributes;

	//! The only constructor which fills the transcodedName attribute
	Attribute( const string name );
	//! Destructor
	~Attribute();

	//! Set accessor to this attribute converting input string into a properly-typed value 
	virtual void Set( const string str ) = 0;
	//! Get accessor converting stored value into a string
	virtual const string Get() = 0;	
	//! This method is the key method to connect the attribute to the related UI element
	void Attach( MappedAttributes &externalAttributes);
	//! This method add and DOMAttribute node to the DOMElement passed as an argument
	void FillAttribute(DOMElement &element);
};

/*!
@brief Class encapsulating all the necessary stuff to work with the double attribute 
*/
class DoubleAttribute: public Attribute
{
public:
	double value;
	DoubleAttribute( const string name);
	void Set( const string str );
	const string Get();
};


/*!
@brief Class encapsulating all the necessary stuff to work with an int attribute 
*/
class IntAttribute : public Attribute
{
public:
	int value;
	IntAttribute( const string name );
	void Set( const string str );
	const string Get();
};

/*!
@brief Class encapsulating all the necessary stuff to work with a string attribute 
*/
class StringAttribute : public Attribute
{
public:
	string value;
	StringAttribute( const string name );
	void Set( const string str );
	const string Get();
};

/*!
@brief UserInfoCore class is for internal purposes only. Use UserInfo<T> instead.

The raison d'etre of this class is to allow pointers to (the main part of) 
UserInfo<T> objects even for different generic types.
*/
class UserInfoCore
{
public:
	//! Fills internal attributes and descendant elements according DOMElement
	virtual void ParseElement(DOMElement *element) = 0;

	//! Fills DOMElement according internal attributes and descendant elements
	virtual void FillElement(DOMElement &element) = 0;
};

/*!
@brief The main user info template class

You should derive this class whenever you need new UserInfo.
*/
template<typename T> class UserInfo : public UserInfoCore
{
private:
	//! Name which identifies the element among others related to the same XML-tag, 
	//! must be unique within its context!
	XMLCh* transcodedName;
	
	//! Type definition of mapping which transforms names to the related elements
	typedef map<const XMLCh* const, UserInfoCore* const, GlobalXercesConnector::Comparator> MappedElements;

protected:
	//! MappiLength of the output vector
	MappedElements elements;
	//! Length of the output vector
	Attribute::MappedAttributes attributes;

public:
	//! Explanation for an user is the only obligatory attribute
	StringAttribute help;

	//! The only constructor which fills both the transcodedName and the help attribute
	UserInfo<T>( 
		const string name, 
		const string help_msg = "" ) 
		: help("help"),		
		transcodedName( XMLString::transcode( name.c_str() ) )
	{	
		XMLString::upperCase( transcodedName );
		help.Attach( attributes );
		help.value = help_msg;
	};
	//! Destructor
	~UserInfo<T>()
	{
		XMLString::release( (XMLCh**)&transcodedName );
	}

	//! returns object of templated type filled with data stored in this UserInfo instance
	virtual T* build(void) = 0;

	//! Save UserInfo to the file (typically with an XML extension)
	void Save ( char* fileName )
	{
		XMLCh* transcodedFileName = XMLString::transcode( fileName );
		LocalFileFormatTarget outputTarget( transcodedFileName );
		XMLString::release( &transcodedFileName );

		DOMDocument* pDoc = XMLConnector.pImplementation->createDocument( L"M3K USER INFO", L"ROOT", NULL );
		DOMElement *pRoot = pDoc->getDocumentElement();

		FillElement( *pRoot );

		XMLConnector.pSerializer->writeNode( &outputTarget, *pDoc);
		delete pDoc;
	}	
	//! Load UserInfo from the file (typically with an XML extension)
	void Load ( char* fileName ) 
	{
		bool bFailed = false;

		try
		{
			XMLCh* transcodedFileName = XMLString::transcode( fileName ) ;
			const LocalFileInputSource inputSource( transcodedFileName );		
			XMLString::release( &transcodedFileName );

			XMLConnector.pParser->parse( inputSource );
			bFailed = ( XMLConnector.pParser->getErrorCount() != 0 );
		}
		catch (...)
		{
			bFailed = true;
		}

		if( !bFailed)
		{
			DOMDocument *pDoc = XMLConnector.pParser->getDocument();		
			DOMElement *pRoot = pDoc->getDocumentElement();
			bFailed = true;
			for( DOMNode* node = pRoot->getFirstChild(); node != NULL; node = node->getNextSibling() )
				if( node->getNodeType() == DOMNode::ELEMENT_NODE )
				{
					ParseElement( (DOMElement*) node );
					bFailed = false;
					break;
				}
		}

		if( bFailed )
		{
			// exception?!
		}		
	}
	//! This method is the key method to connect the element to its parent UI element
	void Attach( MappedElements &externalElements)
	{
		pair<const XMLCh* const, UserInfoCore* const> newPair( transcodedName, this );
		externalElements.insert(newPair);
	}
	
	void ParseElement(DOMElement *element) 
	{
		DOMNodeList* nodeList = element->getChildNodes();
		if( nodeList )
			for( int i = 0; i < nodeList->getLength(); i++ )
			{
				DOMNode* node = nodeList->item(i);
				if( node->getNodeType() == DOMNode::ELEMENT_NODE )
				{
					DOMElement* childElement = (DOMElement*) node;
					MappedElements::const_iterator iter = elements.find( childElement->getTagName() );

					if( iter != elements.end())
						iter->second->ParseElement( childElement );
				}
			}
			
		DOMNamedNodeMap* nodeMap = element->getAttributes();
		if( nodeMap )
			for( int i = 0; i < nodeMap->getLength(); i++ )
			{
				DOMNode* node = nodeMap->item(i);
				if( node->getNodeType() == DOMNode::ATTRIBUTE_NODE )
				{
					DOMAttr* attribute = (DOMAttr*) node;
					Attribute::MappedAttributes::const_iterator iter = attributes.find( attribute->getName() );

					if( iter != attributes.end())
					{
						string attributeValue = GlobalXercesConnector::XMLCh2str( attribute->getValue() );
						iter->second->Set( attributeValue );
					}
				}
			}
	}
	
	void FillElement(DOMElement &element)
	{
		DOMDocument* pDoc = element.getOwnerDocument();

		DOMElement* pHead = pDoc->createElement( transcodedName );		
		element.appendChild( pHead );

		for( MappedElements::const_iterator iter = elements.begin(); iter != elements.end(); iter++)
			iter->second->FillElement( *pHead );

		for( Attribute::MappedAttributes::iterator iter = attributes.begin(); iter != attributes.end(); iter++)
			iter->second->FillAttribute( *pHead );
	}
};

#endif // #ifndef UI_H