#ifndef PMSM_CTR_H
#define PMSM_CTR_H

#include <design/ctrlbase.h>
#include "pmsm.h"

/*! \defgroup PMSM 
@{
*/

using namespace bdm;


/*! PI Controller*/
class PI_ctrl: public Controller{
public:
	//!integral state
	double S;
	double Pd;
	double Pi;
	double minOut;
	double maxOut;
	
	PI_ctrl(double Pd0, double Pi0, double minOut0, double maxOut0): S(0.0), Pd(Pd0), Pi(Pi0),minOut(minOut0),maxOut(maxOut0){}
	virtual vec ctrlaction ( const vec &cond ) {
			double tmp;
			tmp = Pd*cond(0)+S;
			
			S += Pi*cond(0);
			
			if (tmp>maxOut) tmp=maxOut;
			if (tmp<minOut) tmp=minOut;
			return vec_1(tmp);
	}
};

/*! \brief Root class for controllers of PMSM
 * 
 * It defines the interface for controllers that connect to the pmsmDSctrl. 
 * The adapt() function calls bayes() of an internal BM. Note that it is assumed that the estimator is compatible with PMSM system.
 * I.e. it must have rv={ia, ib} and rvc={ua,ub}, any other BM will not work.
 * 
 */
class PMSMCtrl: public Controller{
protected:
	//! Est where rv and rcv are not used!
	shared_ptr<BM> Est;
	//! switch to use or not to use the internal estimator
	bool estim;
	
	// internal quantities filled by PMSMCtrl::ctrlaction()
	double isa;
	double isb;
	double ome;
	double the;
	double Ww;
	
public:

	PMSMCtrl():Controller()	{
		rv = RV("{ua ub }");
		rvc = RV("{o_ia o_ib o_ua o_ub o_om o_th Ww }");
	}
	
    void adapt(const itpp::vec& data){
		if (estim){
			vec y=data.get(0,1);
			vec u=data.get(2,3);
			Est->bayes(y,u);
		}
	}
	
	
    virtual vec ctrlaction(const itpp::vec& cond) {
		
		if (estim){
			vec x_est=Est->posterior().mean();
			isa=x_est(0);
			isb=x_est(1);
			ome=x_est(2);
			the=x_est(3);
		} else {
			isa=cond(0);//x_est(0);
			isb=cond(1);//x_est(1);
			ome=cond(4);//x[2];//x_est(2);
			the=cond(5);//x_est(3);
		}
		Ww=cond(6);
		
		return empty_vec; // dummy return
	};
    void from_setting(const libconfig::Setting& set){
		Controller::from_setting(set);
		Est=UI::build<BM>(set,"estim",UI::optional);
		estim = Est; 
	}
	void log_register ( logger &L, const string &prefix ) {
		Controller::log_register(L,prefix);
		if (estim) Est->log_register(L,prefix);
	}
	void log_write() const{
		if (estim) Est->log_write();
	}
};
//UIREGISTER(PMSMCtrl); -- abstract 

class PMSM_PICtrl: public PMSMCtrl{
public:
	PI_ctrl Cwq;
	PI_ctrl Cuq;
	PI_ctrl Cud;
		
	PMSM_PICtrl():PMSMCtrl(),
	Cwq(3.0, 3.0*0.000125/0.1, -1200, 1200), 
	Cuq(20.0, 20.0*0.000125/0.005, -1200, 1200),
	Cud(20.0, 20.0*0.000125/0.005, -1200, 1200) {}
		
	
	virtual vec ctrlaction(const itpp::vec& cond) {
		PMSMCtrl::ctrlaction(cond); // fills isa,isb,om,the
		
		double Isd = isa*cos(the)+isb*sin(the);
		double Isq = isb*cos(the)-isa*sin(the);
		
		double Iqw=(Cwq.ctrlaction(vec_1(Ww-ome)))(0); // conversion from vec
		
		double ud = (Cud.ctrlaction(vec_1(-Isd)))(0);
		double uq = (Cuq.ctrlaction(vec_1(Iqw-Isq)))(0);
		
		const double Ls0=0.003465; // inductance
		const double Fmag0= 0.1989; // Magnetic??
		
		ud-=Ls0*ome*Iqw; // har
		uq+=Fmag0*ome;
		
		double Um = sqrt(ud*ud+uq*uq);
		double beta = atan(uq/ud);
		if (ud<0) beta += M_PI;
		beta +=the;
		
		vec uab(2);
		uab(0)=Um*cos(beta);		   // usx = usa
		uab(1)=Um*sin(beta);    // usy
		
		return uab;
	};
};
UIREGISTER(PMSM_PICtrl);

/*!@}*/
#endif //PMSM_CTR_H
