/*!
  \file
  \brief Robust Bayesian auto-regression model
  \author Jan Sindelar.
*/

#ifndef ROBUST_H
#define ROBUST_H

//#include <stat/exp_family.h>
#include <itpp/itbase.h>
#include <map>
#include <limits>
#include <vector>
#include <list>
#include <set>
#include <algorithm>
	
//using namespace bdm;
using namespace std;
using namespace itpp;

const double max_range = 10;//numeric_limits<double>::max()/10e-10;

/// An enumeration of possible actions performed on the polyhedrons. We can merge them or split them.
enum actions {MERGE, SPLIT};

// Forward declaration of polyhedron, vertex and emlig
class polyhedron;
class vertex;
class emlig;

/*
class t_simplex
{
public:
	set<vertex*> minima;

	set<vertex*> simplex;

	t_simplex(vertex* origin_vertex)
	{
		simplex.insert(origin_vertex);
		minima.insert(origin_vertex);
	}
};*/

/// A class representing a single condition that can be added to the emlig. A condition represents data entries in a statistical model.
class condition
{	
public:
	/// Value of the condition representing the data
	vec value;	

	/// Mulitplicity of the given condition may represent multiple occurences of same data entry.
	int multiplicity;

	/// Default constructor of condition class takes the value of data entry and creates a condition with multiplicity 1 (first occurence of the data).
	condition(vec value)
	{
		this->value = value;
		multiplicity = 1;
	}
};


/// A class describing a single polyhedron of the split complex. From a collection of such classes a Hasse diagram
/// of the structure in the exponent of a Laplace-Inverse-Gamma density will be created.
class polyhedron
{
	/// A property having a value of 1 usually, with higher value only if the polyhedron arises as a coincidence of
	/// more than just the necessary number of conditions. For example if a newly created line passes through an already
	/// existing point, the points multiplicity will rise by 1.
	int multiplicity;	

	/// A property representing the position of the polyhedron related to current condition with relation to which we
	/// are splitting the parameter space (new data has arrived). This property is setup within a classification procedure and
	/// is only valid while the new condition is being added. It has to be reset when new condition is added and new classification
	/// has to be performed.
	int split_state;

	/// A property representing the position of the polyhedron related to current condition with relation to which we
	/// are merging the parameter space (data is being deleted usually due to a moving window model which is more adaptive and 
	/// steps in for the forgetting in a classical Gaussian AR model). This property is setup within a classification procedure and
	/// is only valid while the new condition is being removed. It has to be reset when new condition is removed and new classification
	/// has to be performed.
	int merge_state;

			

public:
	/// A pointer to the multi-Laplace inverse gamma distribution this polyhedron belongs to.
	emlig* my_emlig;

	/// A list of polyhedrons parents within the Hasse diagram.
	list<polyhedron*> parents;

	/// A list of polyhedrons children withing the Hasse diagram.
	list<polyhedron*> children;

	/// All the vertices of the given polyhedron
	set<vertex*> vertices;

	/// The conditions that gave birth to the polyhedron. If some of them is removed, the polyhedron ceases to exist.
	set<condition*> parentconditions;

	/// A list used for storing children that lie in the positive region related to a certain condition
	list<polyhedron*> positivechildren;

	/// A list used for storing children that lie in the negative region related to a certain condition
	list<polyhedron*> negativechildren;

	/// Children intersecting the condition
	list<polyhedron*> neutralchildren;

	/// A set of grandchildren of the polyhedron that when new condition is added lie exactly on the condition hyperplane. These grandchildren
	/// behave differently from other grandchildren, when the polyhedron is split. New grandchild is not necessarily created on the crossection of
	/// the polyhedron and new condition.
	set<polyhedron*> totallyneutralgrandchildren;

	/// A set of children of the polyhedron that when new condition is added lie exactly on the condition hyperplane. These children
	/// behave differently from other children, when the polyhedron is split. New child is not necessarily created on the crossection of
	/// the polyhedron and new condition.
	set<polyhedron*> totallyneutralchildren;

	/// Reverse relation to the totallyneutralgrandchildren set is needed for merging of already existing polyhedrons to keep 
	/// totallyneutralgrandchildren list up to date.
	set<polyhedron*> grandparents;

	/// Vertices of the polyhedron classified as positive related to an added condition. When the polyhderon is split by the new condition,
	/// these vertices will belong to the positive part of the splitted polyhedron.
	set<vertex*> positiveneutralvertices;

	/// Vertices of the polyhedron classified as negative related to an added condition. When the polyhderon is split by the new condition,
	/// these vertices will belong to the negative part of the splitted polyhedron.
	set<vertex*> negativeneutralvertices;

	/// A bool specifying if the polyhedron lies exactly on the newly added condition or not.
	bool totally_neutral;

	/// When two polyhedrons are merged, there always exists a child lying on the former border of the polyhedrons. This child manages the merge
	/// of the two polyhedrons. This property gives us the address of the mediator child.
	polyhedron* mergechild;

	/// If the polyhedron serves as a mergechild for two of its parents, we need to have the address of the parents to access them. This 
	/// is the pointer to the positive parent being merged.
	polyhedron* positiveparent;

	/// If the polyhedron serves as a mergechild for two of its parents, we need to have the address of the parents to access them. This 
	/// is the pointer to the negative parent being merged.
	polyhedron* negativeparent;	

	/// Adressing withing the statistic. Next_poly is a pointer to the next polyhedron in the statistic on the same level (if this is a point,
	/// next_poly will be a point etc.).
	polyhedron* next_poly;

	/// Adressing withing the statistic. Prev_poly is a pointer to the previous polyhedron in the statistic on the same level (if this is a point,
	/// next_poly will be a point etc.).
	polyhedron* prev_poly;

	/// A property counting the number of messages obtained from children within a classification procedure of position of the polyhedron related
	/// an added/removed condition. If the message counter reaches the number of children, we know the polyhedrons' position has been fully classified.
	int message_counter;

	/// List of triangulation polyhedrons of the polyhedron given by their relative vertices. 
	map<double,set<vertex*>> triangulation;

	/// A list of relative addresses serving for Hasse diagram construction.
	list<int> kids_rel_addresses;

	/// Default constructor
	polyhedron()
	{
		multiplicity = 1;

		message_counter = 0;

		totally_neutral = NULL;

		mergechild = NULL;		
	}
	
	/// Setter for raising multiplicity
	void raise_multiplicity()
	{
		multiplicity++;
	}

	/// Setter for lowering multiplicity
	void lower_multiplicity()
	{
		multiplicity--;
	}

	int get_multiplicity()
	{
		return multiplicity;
	}
	
	/// An obligatory operator, when the class is used within a C++ STL structure like a vector
	int operator==(polyhedron polyhedron2)
	{
		return true;
	}

	/// An obligatory operator, when the class is used within a C++ STL structure like a vector
	int operator<(polyhedron polyhedron2)
	{
		return false;
	}

	
	/// A setter of state of current polyhedron relative to the action specified in the argument. The three possible states of the
	/// polyhedron are -1 - NEGATIVE, 0 - NEUTRAL, 1 - POSITIVE. Neutral state means that either the state has been reset or the polyhedron is
	/// ready to be split/merged.
	int set_state(double state_indicator, actions action)
	{
		switch(action)
		{
			case MERGE:
				merge_state = (int)sign(state_indicator);
				return merge_state;			
			case SPLIT:
				split_state = (int)sign(state_indicator);
				return split_state;		
		}
	}

	/// A getter of state of current polyhedron relative to the action specified in the argument. The three possible states of the
	/// polyhedron are -1 - NEGATIVE, 0 - NEUTRAL, 1 - POSITIVE. Neutral state means that either the state has been reset or the polyhedron is
	/// ready to be split/merged.
	int get_state(actions action)
	{
		switch(action)
		{
			case MERGE:
				return merge_state;			
			break;
			case SPLIT:
				return split_state;
			break;
		}
	}

	/// Method for obtaining the number of children of given polyhedron.
	int number_of_children()
	{
		return children.size();
	}

	/// A method for triangulation of given polyhedron.
	void triangulate(bool should_integrate);	
};


/// A class for representing 0-dimensional polyhedron - a vertex. It will be located in the bottom row of the Hasse
/// diagram representing a complex of polyhedrons. It has its coordinates in the parameter space.
class vertex : public polyhedron
{
	/// A dynamic array representing coordinates of the vertex
	vec coordinates;

public:
	/// A property specifying the value of the density (ted nevim, jestli je to jakoby log nebo ne) above the vertex.
	double function_value;

	/// Default constructor
	vertex();

	/// Constructor of a vertex from a set of coordinates
	vertex(vec coordinates)
	{
		this->coordinates   = coordinates;

		vertices.insert(this);

		set<vertex*> vert_simplex;

		vert_simplex.insert(this);

		triangulation.insert(pair<double,set<vertex*>>(0,vert_simplex));
	}

	/// A method that widens the set of coordinates of given vertex. It is used when a complex in a parameter
	/// space of certain dimension is established, but the dimension is not known when the vertex is created.
	void push_coordinate(double coordinate)
	{
		coordinates  = concat(coordinates,coordinate);		
	}

	/// A method obtaining the set of coordinates of a vertex. These coordinates are not obtained as a pointer 
	/// (not given by reference), but a new copy is created (they are given by value).
	vec get_coordinates()
	{
		return coordinates;
	}
		
};


/// A class representing a polyhedron in a top row of the complex. Such polyhedron has a condition that differen   tiates
/// it from polyhedrons in other rows. 
class toprow : public polyhedron
{
	
public:
	double probability;

	vertex* minimal_vertex;

	/// A condition used for determining the function of a Laplace-Inverse-Gamma density resulting from Bayesian estimation
	vec condition_sum;

	int condition_order;

	/// Default constructor
	toprow(){};

	/// Constructor creating a toprow from the condition
	toprow(condition *condition, int condition_order)
	{
		this->condition_sum   = condition->value;
		this->condition_order = condition_order;
	}

	toprow(vec condition_sum, int condition_order)
	{
		this->condition_sum   = condition_sum;
		this->condition_order = condition_order;
	}

	double integrate_simplex(set<vertex*> simplex, char c);

};






class c_statistic
{

public:
	polyhedron* end_poly;
	polyhedron* start_poly;

	vector<polyhedron*> rows;

	vector<polyhedron*> row_ends;

	c_statistic()
	{
		end_poly   = new polyhedron();
		start_poly = new polyhedron();
	};

	void append_polyhedron(int row, polyhedron* appended_start, polyhedron* appended_end)
	{
		if(row>((int)rows.size())-1)
		{
			if(row>rows.size())
			{
				throw new exception("You are trying to append a polyhedron whose children are not in the statistic yet!");
				return;
			}

			rows.push_back(end_poly);
			row_ends.push_back(end_poly);
		}

		// POSSIBLE FAILURE: the function is not checking if start and end are connected

		if(rows[row] != end_poly)
		{
			appended_start->prev_poly = row_ends[row];
			row_ends[row]->next_poly = appended_start;			
						
		}
		else if((row>0 && rows[row-1]!=end_poly)||row==0)
		{
			appended_start->prev_poly = start_poly;
			rows[row]= appended_start;			
		}
		else
		{
			throw new exception("Wrong polyhedron insertion into statistic: missing intermediary polyhedron!");
		}

		appended_end->next_poly = end_poly;
		row_ends[row] = appended_end;
	}

	void append_polyhedron(int row, polyhedron* appended_poly)
	{
		append_polyhedron(row,appended_poly,appended_poly);
	}

	void insert_polyhedron(int row, polyhedron* inserted_poly, polyhedron* following_poly)
	{		
		if(following_poly != end_poly)
		{
			inserted_poly->next_poly = following_poly;
			inserted_poly->prev_poly = following_poly->prev_poly;

			if(following_poly->prev_poly == start_poly)
			{
				rows[row] = inserted_poly;
			}
			else
			{				
				inserted_poly->prev_poly->next_poly = inserted_poly;								
			}

			following_poly->prev_poly = inserted_poly;
		}
		else
		{
			this->append_polyhedron(row, inserted_poly);
		}		
	
	}


	

	void delete_polyhedron(int row, polyhedron* deleted_poly)
	{
		if(deleted_poly->prev_poly != start_poly)
		{
			deleted_poly->prev_poly->next_poly = deleted_poly->next_poly;
		}
		else
		{
			rows[row] = deleted_poly->next_poly;
		}

		if(deleted_poly->next_poly!=end_poly)
		{
			deleted_poly->next_poly->prev_poly = deleted_poly->prev_poly;
		}
		else
		{
			row_ends[row] = deleted_poly->prev_poly;
		}

		

		deleted_poly->next_poly = NULL;
		deleted_poly->prev_poly = NULL;					
	}

	int size()
	{
		return rows.size();
	}

	polyhedron* get_end()
	{
		return end_poly;
	}

	polyhedron* get_start()
	{
		return start_poly;
	}

	int row_size(int row)
	{
		if(this->size()>row && row>=0)
		{
			int row_size = 0;
			
			for(polyhedron* row_poly = rows[row]; row_poly!=end_poly; row_poly=row_poly->next_poly)
			{
				row_size++;
			}

			return row_size;
		}
		else
		{
			throw new exception("There is no row to obtain size from!");
		}
	}
};


class my_ivec : public ivec
{
public:
	my_ivec():ivec(){};

	my_ivec(ivec origin):ivec()
	{
		this->ins(0,origin);
	}

	bool operator>(const my_ivec &second) const
	{
		return max(*this)>max(second);
		
		/*
		int size1 = this->size();
		int size2 = second.size();		
		 
		int counter1 = 0;
		while(0==0)
		{
			if((*this)[counter1]==0)
			{
				size1--;
			}
			
			if((*this)[counter1]!=0)
				break;

			counter1++;
		}

		int counter2 = 0;
		while(0==0)
		{
			if(second[counter2]==0)
			{
				size2--;
			}
			
			if(second[counter2]!=0)
				break;

			counter2++;
		}

		if(size1!=size2)
		{
			return size1>size2;
		}
		else
		{
			for(int i = 0;i<size1;i++)
			{
				if((*this)[counter1+i]!=second[counter2+i])
				{
					return (*this)[counter1+i]>second[counter2+i];
				}
			}

			return false;
		}*/
	}

	
	bool operator==(const my_ivec &second) const
	{
		return max(*this)==max(second);
		
		/*
		int size1 = this->size();
		int size2 = second.size();		
		 
		int counter = 0;
		while(0==0)
		{
			if((*this)[counter]==0)
		{
			size1--;
		}
			
		if((*this)[counter]!=0)
			break;

		counter++;
		}

		counter = 0;
		while(0==0)
		{
			if(second[counter]==0)
			{
				size2--;
			}
			
			if(second[counter]!=0)
				break;

			counter++;
		}

		if(size1!=size2)
		{
			return false;
		}
		else
		{
			for(int i=0;i<size1;i++)
			{
				if((*this)[size()-1-i]!=second[second.size()-1-i])
				{
					return false;
				}
			}

			return true;
		}*/
	}

	bool operator<(const my_ivec &second) const
	{
		return !(((*this)>second)||((*this)==second));
	}

	bool operator!=(const my_ivec &second) const
	{
		return !((*this)==second);
	}

	bool operator<=(const my_ivec &second) const
	{
		return !((*this)>second);
	}

	bool operator>=(const my_ivec &second) const
	{
		return !((*this)<second);
	}

	my_ivec right(my_ivec original)
	{
		
	}
};







//! Conditional(e) Multicriteria-Laplace-Inverse-Gamma distribution density
class emlig // : eEF
{

	/// A statistic in a form of a Hasse diagram representing a complex of convex polyhedrons obtained as a result
	/// of data update from Bayesian estimation or set by the user if this emlig is a prior density
	

	vector<list<polyhedron*>> for_splitting;
		
	vector<list<polyhedron*>> for_merging;

	list<condition*> conditions;

	double normalization_factor;

	

	void alter_toprow_conditions(condition *condition, bool should_be_added)
	{
		for(polyhedron* horiz_ref = statistic.rows[statistic.size()-1];horiz_ref!=statistic.get_end();horiz_ref=horiz_ref->next_poly)
		{
			set<vertex*>::iterator vertex_ref = horiz_ref->vertices.begin();

			do
			{
				vertex_ref++;
			}
			while((*vertex_ref)->parentconditions.find(condition)==(*vertex_ref)->parentconditions.end());

			double product = (*vertex_ref)->get_coordinates()*condition->value;

			if(should_be_added)
			{
				((toprow*) horiz_ref)->condition_order++;

				if(product>0)
				{
					((toprow*) horiz_ref)->condition_sum += condition->value;
				}
				else
				{
					((toprow*) horiz_ref)->condition_sum -= condition->value;
				}
			}
			else
			{ 
				((toprow*) horiz_ref)->condition_order--;

				if(product<0)			
				{
					((toprow*) horiz_ref)->condition_sum += condition->value;
				}
				else
				{
					((toprow*) horiz_ref)->condition_sum -= condition->value;
				}
			}				
		}
	}



	void send_state_message(polyhedron* sender, condition *toadd, condition *toremove, int level)
	{			

		bool shouldmerge    = (toremove != NULL);
		bool shouldsplit    = (toadd != NULL);
		
		if(shouldsplit||shouldmerge)
		{
			for(list<polyhedron*>::iterator parent_iterator = sender->parents.begin();parent_iterator!=sender->parents.end();parent_iterator++)
			{
				polyhedron* current_parent = *parent_iterator;

				current_parent->message_counter++;

				bool is_last  = (current_parent->message_counter == current_parent->number_of_children());
				bool is_first = (current_parent->message_counter == 1);

				if(shouldmerge)
				{
					int child_state  = sender->get_state(MERGE);
					int parent_state = current_parent->get_state(MERGE);

					if(parent_state == 0||is_first)
					{
						parent_state = current_parent->set_state(child_state, MERGE);						
					}					

					if(child_state == 0)
					{
						if(current_parent->mergechild == NULL)
						{
							current_parent->mergechild = sender;
						}							
					}					

					if(is_last)
					{						
						if(parent_state == 1)
						{
							((toprow*)current_parent)->condition_sum-=toremove->value;
							((toprow*)current_parent)->condition_order--;
						}

						if(parent_state == -1)
						{
							((toprow*)current_parent)->condition_sum+=toremove->value;
							((toprow*)current_parent)->condition_order--;
						}
						
						if(current_parent->mergechild != NULL)
						{
							if(current_parent->mergechild->get_multiplicity()==1)
							{
								if(parent_state > 0)
								{							
									current_parent->mergechild->positiveparent = current_parent;							
								}

								if(parent_state < 0)
								{							
									current_parent->mergechild->negativeparent = current_parent;							
								}
							}							
						}						
						else
						{
							//current_parent->set_state(0,MERGE);	

							if(level == number_of_parameters - 1)
							{
								toprow* cur_par_toprow = ((toprow*)current_parent);
								cur_par_toprow->probability = 0.0;
								
								map<double,set<vertex*>> new_triangulation;

								for(map<double,set<vertex*>>::iterator t_ref = current_parent->triangulation.begin();t_ref!=current_parent->triangulation.end();t_ref++)
								{
									double cur_prob = cur_par_toprow->integrate_simplex((*t_ref).second,'C');
									
									cur_par_toprow->probability += cur_prob;

									new_triangulation.insert(pair<double,set<vertex*>>(cur_prob,(*t_ref).second));
								}

								current_parent->triangulation.clear();
								current_parent->triangulation.insert(new_triangulation.begin(),new_triangulation.end());
							}
						}

						if(parent_state == 0)
						{
							for_merging[level+1].push_back(current_parent);
							//current_parent->parentconditions.erase(toremove);
						}						

												
					}					
				}

				if(shouldsplit)
				{
					current_parent->totallyneutralgrandchildren.insert(sender->totallyneutralchildren.begin(),sender->totallyneutralchildren.end());

					for(set<polyhedron*>::iterator tot_child_ref = sender->totallyneutralchildren.begin();tot_child_ref!=sender->totallyneutralchildren.end();tot_child_ref++)
					{
						(*tot_child_ref)->grandparents.insert(current_parent);
					}

					switch(sender->get_state(SPLIT))
					{
					case 1:
						current_parent->positivechildren.push_back(sender);
						current_parent->positiveneutralvertices.insert(sender->vertices.begin(),sender->vertices.end());
					break;
					case 0:
						current_parent->neutralchildren.push_back(sender);
						current_parent->positiveneutralvertices.insert(sender->positiveneutralvertices.begin(),sender->positiveneutralvertices.end());
						current_parent->negativeneutralvertices.insert(sender->negativeneutralvertices.begin(),sender->negativeneutralvertices.end());

						if(current_parent->totally_neutral == NULL)
						{
							current_parent->totally_neutral = sender->totally_neutral;
						}
						else
						{
							current_parent->totally_neutral = current_parent->totally_neutral && sender->totally_neutral;
						}

						if(sender->totally_neutral)
						{
							current_parent->totallyneutralchildren.insert(sender);
						}
							
					break;
					case -1:
						current_parent->negativechildren.push_back(sender);
						current_parent->negativeneutralvertices.insert(sender->vertices.begin(),sender->vertices.end());
					break;
					}

					if(is_last)
					{
						
						/// \TODO Nechapu druhou podminku, zda se mi ze je to spatne.. Nemela by byt jen prvni? Nebo se jedna o nastaveni totalni neutrality?
						if((current_parent->negativechildren.size()>0&&current_parent->positivechildren.size()>0)||
													(current_parent->neutralchildren.size()>0&&current_parent->totally_neutral==false))
						{
							for_splitting[level+1].push_back(current_parent);						
								
							current_parent->set_state(0, SPLIT);
						}
						else
						{
							if(current_parent->negativechildren.size()>0)
							{
								current_parent->set_state(-1, SPLIT);

								((toprow*)current_parent)->condition_sum-=toadd->value;
									
							}
							else if(current_parent->positivechildren.size()>0)
							{
								current_parent->set_state(1, SPLIT);

								((toprow*)current_parent)->condition_sum+=toadd->value;									
							}
							else
							{
								current_parent->raise_multiplicity();								
							}

							((toprow*)current_parent)->condition_order++;

							if(level == number_of_parameters - 1)
							{
								toprow* cur_par_toprow = ((toprow*)current_parent);
								cur_par_toprow->probability = 0.0;
									
								map<double,set<vertex*>> new_triangulation;
								
								for(map<double,set<vertex*>>::iterator t_ref = current_parent->triangulation.begin();t_ref!=current_parent->triangulation.end();t_ref++)
								{
									double cur_prob = cur_par_toprow->integrate_simplex((*t_ref).second,'C');
									
									cur_par_toprow->probability += cur_prob;

									new_triangulation.insert(pair<double,set<vertex*>>(cur_prob,(*t_ref).second));
								}

								current_parent->triangulation.clear();
								current_parent->triangulation.insert(new_triangulation.begin(),new_triangulation.end());
							}

							if(current_parent->mergechild == NULL)
							{
								current_parent->positivechildren.clear();
								current_parent->negativechildren.clear();
								current_parent->neutralchildren.clear();
								current_parent->totallyneutralchildren.clear();
								current_parent->totallyneutralgrandchildren.clear();
								// current_parent->grandparents.clear();
								current_parent->positiveneutralvertices.clear();
								current_parent->negativeneutralvertices.clear();
								current_parent->totally_neutral = NULL;
								current_parent->kids_rel_addresses.clear();
							}							
						}
					}
				}

				if(is_last)
				{
					current_parent->mergechild = NULL;
					current_parent->message_counter = 0;

					send_state_message(current_parent,toadd,toremove,level+1);
				}
			
			}
			
		}		
	}
	
public:	
	c_statistic statistic;

	vertex* minimal_vertex;

	double likelihood_value;

	vector<multiset<my_ivec>> correction_factors;

	int number_of_parameters;

	/// A default constructor creates an emlig with predefined statistic representing only the range of the given
	/// parametric space, where the number of parameters of the needed model is given as a parameter to the constructor.
	emlig(int number_of_parameters)
	{	
		this->number_of_parameters = number_of_parameters;

		create_statistic(number_of_parameters);

		likelihood_value = numeric_limits<double>::max();
	}

	/// A constructor for creating an emlig when the user wants to create the statistic by himself. The creation of a
	/// statistic is needed outside the constructor. Used for a user defined prior distribution on the parameters.
	emlig(c_statistic statistic)
	{
		this->statistic = statistic;	

		likelihood_value = numeric_limits<double>::max();
	}

	void step_me(int marker)
	{
		
		for(int i = 0;i<statistic.size();i++)
		{
			int zero = 0;
			int one  = 0;
			int two  = 0;

			for(polyhedron* horiz_ref = statistic.rows[i];horiz_ref!=statistic.get_end();horiz_ref=horiz_ref->next_poly)
			{
				
				/*
				if(i==statistic.size()-1)
				{
					//cout << ((toprow*)horiz_ref)->condition_sum << "   " << ((toprow*)horiz_ref)->probability << endl;
					cout << "Order:" << ((toprow*)horiz_ref)->condition_order << endl;
				}
				if(i==0)
				{
					cout << ((vertex*)horiz_ref)->get_coordinates() << endl;
				}
				*/

				char* string = "Checkpoint";

				if((*horiz_ref).parentconditions.size()==0)
				{
					zero++;
				}
				else if((*horiz_ref).parentconditions.size()==1)
				{
					one++;					
				}
				else
				{
					two++;
				}
				
			}
		}
		

		/*
		list<vec> table_entries;
		for(polyhedron* horiz_ref = statistic.rows[statistic.size()-1];horiz_ref!=statistic.row_ends[statistic.size()-1];horiz_ref=horiz_ref->next_poly)
		{
			toprow *current_toprow = (toprow*)(horiz_ref);
			for(list<set<vertex*>>::iterator tri_ref = current_toprow->triangulation.begin();tri_ref!=current_toprow->triangulation.end();tri_ref++)
			{
				for(set<vertex*>::iterator vert_ref = (*tri_ref).begin();vert_ref!=(*tri_ref).end();vert_ref++)
				{
					vec table_entry = vec();
					
					table_entry.ins(0,(*vert_ref)->get_coordinates()*current_toprow->condition.get(1,current_toprow->condition.size()-1)-current_toprow->condition.get(0,0));
					
					table_entry.ins(0,(*vert_ref)->get_coordinates());

					table_entries.push_back(table_entry);
				}
			}			
		}

		unique(table_entries.begin(),table_entries.end());

				
		
		for(list<vec>::iterator entry_ref = table_entries.begin();entry_ref!=table_entries.end();entry_ref++)
		{
			ofstream myfile;
			myfile.open("robust_data.txt", ios::out | ios::app);
			if (myfile.is_open())
			{
				for(int i = 0;i<(*entry_ref).size();i++)
				{
					myfile << (*entry_ref)[i] << ";";
				}
				myfile << endl;
			
				myfile.close();
			}
			else
			{
				cout << "File problem." << endl;
			}
		}
		*/
		

		return;
	}

	int statistic_rowsize(int row)
	{
		return statistic.row_size(row);
	}

	void add_condition(vec toadd)
	{
		vec null_vector = "";

		add_and_remove_condition(toadd, null_vector);
	}


	void remove_condition(vec toremove)
	{		
		vec null_vector = "";

		add_and_remove_condition(null_vector, toremove);
	
	}

	void add_and_remove_condition(vec toadd, vec toremove)
	{
		step_me(0);
		
		likelihood_value = numeric_limits<double>::max();

		bool should_remove = (toremove.size() != 0);
		bool should_add    = (toadd.size() != 0);

		for_splitting.clear();
		for_merging.clear();

		for(int i = 0;i<statistic.size();i++)
		{
			list<polyhedron*> empty_split;
			list<polyhedron*> empty_merge;

			for_splitting.push_back(empty_split);
			for_merging.push_back(empty_merge);
		}

		list<condition*>::iterator toremove_ref = conditions.end();
		bool condition_should_be_added = should_add;

		for(list<condition*>::iterator ref = conditions.begin();ref!=conditions.end();ref++)
		{
			if(should_remove)
			{
				if((*ref)->value == toremove)
				{
					if((*ref)->multiplicity>1)
					{
						(*ref)->multiplicity--;

						alter_toprow_conditions(*ref,false);

						should_remove = false;
					}
					else
					{
						toremove_ref = ref;							
					}
				}
			}

			if(should_add)
			{
				if((*ref)->value == toadd)
				{
					(*ref)->multiplicity++;

					alter_toprow_conditions(*ref,true);

					should_add = false;

					condition_should_be_added = false;
				}				
			}
		}	

		condition* condition_to_remove = NULL;

		if(toremove_ref!=conditions.end())
		{
			condition_to_remove = *toremove_ref;
			conditions.erase(toremove_ref);			
		}

		condition* condition_to_add = NULL;

		if(condition_should_be_added)
		{
			condition* new_condition = new condition(toadd);
			
			conditions.push_back(new_condition);
			condition_to_add = new_condition;
		}		
		
		for(polyhedron* horizontal_position = statistic.rows[0];horizontal_position!=statistic.get_end();horizontal_position=horizontal_position->next_poly)
		{		
			vertex* current_vertex = (vertex*)horizontal_position;
			
			if(should_add||should_remove)
			{
				vec appended_coords = current_vertex->get_coordinates();
				appended_coords.ins(0,-1.0);				

				if(should_add)
				{
					double local_condition = 0;// = toadd*(appended_coords.first/=appended_coords.second);

					local_condition = appended_coords*toadd;

					current_vertex->set_state(local_condition,SPLIT);

					/// \TODO There should be a rounding error tolerance used here to insure we are not having too many points because of rounding error.
					if(local_condition == 0)
					{
						current_vertex->totally_neutral = true;

						current_vertex->raise_multiplicity();

						current_vertex->negativeneutralvertices.insert(current_vertex);
						current_vertex->positiveneutralvertices.insert(current_vertex);
					}					
				}
			
				if(should_remove)
				{					
					set<condition*>::iterator cond_ref;
					
					for(cond_ref = current_vertex->parentconditions.begin();cond_ref!=current_vertex->parentconditions.end();cond_ref++)
					{
						if(*cond_ref == condition_to_remove)
						{
							break;
						}
					}

					if(cond_ref!=current_vertex->parentconditions.end())
					{
						current_vertex->parentconditions.erase(cond_ref);
						current_vertex->set_state(0,MERGE);
						for_merging[0].push_back(current_vertex);
					}
					else
					{
						double local_condition = toremove*appended_coords;
						current_vertex->set_state(local_condition,MERGE);
					}
				}				
			}

			send_state_message(current_vertex, condition_to_add, condition_to_remove, 0);		
			
		}

		
		
		if(should_remove)
		{
			for(int i = 0;i<for_merging.size();i++)
			{
				for(list<polyhedron*>::iterator merge_ref = for_merging[i].begin();merge_ref!=for_merging[i].end();merge_ref++)
				{
					cout << (*merge_ref)->get_state(MERGE) << ",";
				}

				cout << endl;
			}

			set<vertex*> vertices_to_be_reduced;			
			
			int k = 1;

			for(vector<list<polyhedron*>>::iterator vert_ref = for_merging.begin();vert_ref<for_merging.end();vert_ref++)
			{
				for(list<polyhedron*>::reverse_iterator merge_ref = vert_ref->rbegin();merge_ref!=vert_ref->rend();merge_ref++)
				{
					if((*merge_ref)->get_multiplicity()>1)
					{
						if(k==1)
						{
							vertices_to_be_reduced.insert((vertex*)(*merge_ref));
						}
						else
						{
							(*merge_ref)->lower_multiplicity();
						}
					}
					else
					{
						toprow* current_positive = (toprow*)(*merge_ref)->positiveparent;
						toprow* current_negative = (toprow*)(*merge_ref)->negativeparent;

						//current_positive->condition_sum -= toremove;
						//current_positive->condition_order--;

						current_positive->parentconditions.erase(condition_to_remove);
						
						current_positive->children.insert(current_positive->children.end(),current_negative->children.begin(),current_negative->children.end());
						current_positive->children.remove(*merge_ref);

						for(list<polyhedron*>::iterator child_ref = current_negative->children.begin();child_ref!=current_negative->children.end();child_ref++)
						{
							(*child_ref)->parents.remove(current_negative);
							(*child_ref)->parents.push_back(current_positive);													
						}

						// current_positive->parents.insert(current_positive->parents.begin(),current_negative->parents.begin(),current_negative->parents.end());
						// unique(current_positive->parents.begin(),current_positive->parents.end());

						for(list<polyhedron*>::iterator parent_ref = current_negative->parents.begin();parent_ref!=current_negative->parents.end();parent_ref++)
						{
							(*parent_ref)->children.remove(current_negative);

							switch(current_negative->get_state(SPLIT))
							{
							case -1:
								(*parent_ref)->negativechildren.remove(current_negative);
								break;
							case 0:
								(*parent_ref)->neutralchildren.remove(current_negative);								
								break;
							case 1:
								(*parent_ref)->positivechildren.remove(current_negative);
								break;
							}
							//(*parent_ref)->children.push_back(current_positive);
						}

						if(current_positive->get_state(SPLIT)!=0&&current_negative->get_state(SPLIT)==0)
						{
							for(list<polyhedron*>::iterator parent_ref = current_positive->parents.begin();parent_ref!=current_positive->parents.end();parent_ref++)
							{
								if(current_positive->get_state(SPLIT)==1)
								{
									(*parent_ref)->positivechildren.remove(current_positive);
								}
								else
								{
									(*parent_ref)->negativechildren.remove(current_positive);
								}

								(*parent_ref)->neutralchildren.push_back(current_positive);
							}

							current_positive->set_state(0,SPLIT);
							for_splitting[k].push_back(current_positive);
						}
						
						if((current_positive->get_state(SPLIT)==0&&!current_positive->totally_neutral)||(current_negative->get_state(SPLIT)==0&&!current_negative->totally_neutral))
						{
							current_positive->negativechildren.insert(current_positive->negativechildren.end(),current_negative->negativechildren.begin(),current_negative->negativechildren.end());						
							
							current_positive->positivechildren.insert(current_positive->positivechildren.end(),current_negative->positivechildren.begin(),current_negative->positivechildren.end());
													
							current_positive->neutralchildren.insert(current_positive->neutralchildren.end(),current_negative->neutralchildren.begin(),current_negative->neutralchildren.end());
						
							switch((*merge_ref)->get_state(SPLIT))
							{
							case -1:
								current_positive->negativechildren.remove(*merge_ref);
								break;
							case 0:
								current_positive->neutralchildren.remove(*merge_ref);
								break;
							case 1:
								current_positive->positivechildren.remove(*merge_ref);
								break;
							}

							current_positive->totallyneutralchildren.insert(current_negative->totallyneutralchildren.begin(),current_negative->totallyneutralchildren.end());
							
							current_positive->totallyneutralchildren.erase(*merge_ref);

							current_positive->totallyneutralgrandchildren.insert(current_negative->totallyneutralgrandchildren.begin(),current_negative->totallyneutralgrandchildren.end());

							current_positive->negativeneutralvertices.insert(current_negative->negativeneutralvertices.begin(),current_negative->negativeneutralvertices.end());
							current_positive->positiveneutralvertices.insert(current_negative->positiveneutralvertices.begin(),current_negative->positiveneutralvertices.end());
						}
						else
						{
							if(!current_positive->totally_neutral)
							{
								current_positive->positivechildren.clear();
								current_positive->negativechildren.clear();
								current_positive->neutralchildren.clear();
								current_positive->totallyneutralchildren.clear();
								current_positive->totallyneutralgrandchildren.clear();								
								current_positive->positiveneutralvertices.clear();
								current_positive->negativeneutralvertices.clear();
								current_positive->totally_neutral = NULL;
								current_positive->kids_rel_addresses.clear();								
							}
						
						}

												
						
						current_positive->vertices.insert(current_negative->vertices.begin(),current_negative->vertices.end());
						
						
						for(set<vertex*>::iterator vert_ref = (*merge_ref)->vertices.begin();vert_ref!=(*merge_ref)->vertices.end();vert_ref++)
						{
							if((*vert_ref)->get_multiplicity()==1)
							{
								current_positive->vertices.erase(*vert_ref);
								
								if((current_positive->get_state(SPLIT)==0&&!current_positive->totally_neutral)||(current_negative->get_state(SPLIT)==0&&!current_negative->totally_neutral))
								{
									current_positive->negativeneutralvertices.erase(*vert_ref);
									current_positive->positiveneutralvertices.erase(*vert_ref);
								}
							}
						}
						
						if(current_negative->get_state(SPLIT)==0&&!current_negative->totally_neutral)
						{
							for_splitting[k].remove(current_negative);	
						}

						if(current_positive->totally_neutral)
						{
							if(!current_negative->totally_neutral)
							{
								for(set<polyhedron*>::iterator grand_ref = current_positive->grandparents.begin();grand_ref!=current_positive->grandparents.end();grand_ref++)
								{
									(*grand_ref)->totallyneutralgrandchildren.erase(current_positive);
								}								
							}
							else
							{
								for(set<polyhedron*>::iterator grand_ref = current_negative->grandparents.begin();grand_ref!=current_negative->grandparents.end();grand_ref++)
								{
									(*grand_ref)->totallyneutralgrandchildren.erase(current_negative);
									(*grand_ref)->totallyneutralgrandchildren.insert(current_positive);
								}								
							}
						}
						else
						{
							if(current_negative->totally_neutral)
							{
								for(set<polyhedron*>::iterator grand_ref = current_negative->grandparents.begin();grand_ref!=current_negative->grandparents.end();grand_ref++)
								{
									(*grand_ref)->totallyneutralgrandchildren.erase(current_negative);									
								}
							}							
						}

						current_positive->grandparents.clear();

						
						
						current_positive->totally_neutral = (current_positive->totally_neutral && current_negative->totally_neutral);

						current_positive->triangulate(k==for_splitting.size()-1);
						
						statistic.delete_polyhedron(k,current_negative);

						delete current_negative;

						for(list<polyhedron*>::iterator child_ref = (*merge_ref)->children.begin();child_ref!=(*merge_ref)->children.end();child_ref++)
						{
							(*child_ref)->parents.remove(*merge_ref);
						}

						/*
						for(list<polyhedron*>::iterator parent_ref = (*merge_ref)->parents.begin();parent_ref!=(*merge_ref)->parents.end();parent_ref++)
						{
							(*parent_ref)->positivechildren.remove(*merge_ref);
							(*parent_ref)->negativechildren.remove(*merge_ref);
							(*parent_ref)->neutralchildren.remove(*merge_ref);
							(*parent_ref)->children.remove(*merge_ref);
						}
						*/

						for(set<polyhedron*>::iterator grand_ch_ref = (*merge_ref)->totallyneutralgrandchildren.begin();grand_ch_ref!=(*merge_ref)->totallyneutralgrandchildren.end();grand_ch_ref++)
						{
							(*grand_ch_ref)->grandparents.erase(*merge_ref);
						}

						
						for(set<polyhedron*>::iterator grand_p_ref = (*merge_ref)->grandparents.begin();grand_p_ref!=(*merge_ref)->grandparents.end();grand_p_ref++)
						{
							(*grand_p_ref)->totallyneutralgrandchildren.erase(*merge_ref);
						}
						
						for_splitting[k-1].remove(*merge_ref);

						statistic.delete_polyhedron(k-1,*merge_ref);

						if(k==1)
						{
							vertices_to_be_reduced.insert((vertex*)(*merge_ref));
						}
						else
						{
							delete *merge_ref;
						}
					}
				}			
			
				k++;

			}

			for(set<vertex*>::iterator vert_ref = vertices_to_be_reduced.begin();vert_ref!=vertices_to_be_reduced.end();vert_ref++)
			{
				if((*vert_ref)->get_multiplicity()>1)
				{
					(*vert_ref)->lower_multiplicity();
				}
				else
				{
					delete *vert_ref;
				}
			}

			delete condition_to_remove;
		}
		
		vector<int> sizevector;
		for(int s = 0;s<statistic.size();s++)
		{
			sizevector.push_back(statistic.row_size(s));
			cout << statistic.row_size(s) << ", ";
		}

		cout << endl;

		if(should_add)
		{
			int k = 1;

			vector<list<polyhedron*>>::iterator beginning_ref = ++for_splitting.begin();

			for(vector<list<polyhedron*>>::iterator vert_ref = beginning_ref;vert_ref<for_splitting.end();vert_ref++)
			{			

				for(list<polyhedron*>::reverse_iterator split_ref = vert_ref->rbegin();split_ref != vert_ref->rend();split_ref++)
				{
					polyhedron* new_totally_neutral_child;

					polyhedron* current_polyhedron = (*split_ref);
					
					if(vert_ref == beginning_ref)
					{
						vec coordinates1 = ((vertex*)(*(current_polyhedron->children.begin())))->get_coordinates();						
						vec coordinates2 = ((vertex*)(*(++current_polyhedron->children.begin())))->get_coordinates();
						
						vec extended_coord2 = coordinates2;
						extended_coord2.ins(0,-1.0);						

						double t = (-toadd*extended_coord2)/(toadd(1,toadd.size()-1)*(coordinates1-coordinates2));						

						vec new_coordinates = (1-t)*coordinates2+t*coordinates1;						

						// cout << "c1:" << coordinates1 << endl << "c2:" << coordinates2 << endl << "nc:" << new_coordinates << endl;

						vertex* neutral_vertex = new vertex(new_coordinates);						

						new_totally_neutral_child = neutral_vertex;
					}
					else
					{
						toprow* neutral_toprow = new toprow();
						
						neutral_toprow->condition_sum   = ((toprow*)current_polyhedron)->condition_sum; // tohle tu bylo driv: zeros(number_of_parameters+1);
						neutral_toprow->condition_order = ((toprow*)current_polyhedron)->condition_order+1;

						new_totally_neutral_child = neutral_toprow;
					}

					new_totally_neutral_child->parentconditions.insert(current_polyhedron->parentconditions.begin(),current_polyhedron->parentconditions.end());
					new_totally_neutral_child->parentconditions.insert(condition_to_add);

					new_totally_neutral_child->my_emlig = this;
					
					new_totally_neutral_child->children.insert(new_totally_neutral_child->children.end(),
														current_polyhedron->totallyneutralgrandchildren.begin(),
																current_polyhedron->totallyneutralgrandchildren.end());

					

					// cout << ((toprow*)current_polyhedron)->condition << endl << toadd << endl;

					toprow* positive_poly = new toprow(((toprow*)current_polyhedron)->condition_sum+toadd, ((toprow*)current_polyhedron)->condition_order+1);
					toprow* negative_poly = new toprow(((toprow*)current_polyhedron)->condition_sum-toadd, ((toprow*)current_polyhedron)->condition_order+1);

					positive_poly->my_emlig = this;
					negative_poly->my_emlig = this;

					positive_poly->parentconditions.insert(current_polyhedron->parentconditions.begin(),current_polyhedron->parentconditions.end());
					negative_poly->parentconditions.insert(current_polyhedron->parentconditions.begin(),current_polyhedron->parentconditions.end());

					for(set<polyhedron*>::iterator grand_ref = current_polyhedron->totallyneutralgrandchildren.begin(); grand_ref != current_polyhedron->totallyneutralgrandchildren.end();grand_ref++)
					{
						(*grand_ref)->parents.push_back(new_totally_neutral_child);
						
						// tohle tu nebylo. ma to tu byt?
						//positive_poly->totallyneutralgrandchildren.insert(*grand_ref);
						//negative_poly->totallyneutralgrandchildren.insert(*grand_ref);

						//(*grand_ref)->grandparents.insert(positive_poly);
						//(*grand_ref)->grandparents.insert(negative_poly);

						new_totally_neutral_child->vertices.insert((*grand_ref)->vertices.begin(),(*grand_ref)->vertices.end());
					}

					positive_poly->children.push_back(new_totally_neutral_child);
					negative_poly->children.push_back(new_totally_neutral_child);
					

					for(list<polyhedron*>::iterator parent_ref = current_polyhedron->parents.begin();parent_ref!=current_polyhedron->parents.end();parent_ref++)
					{
						(*parent_ref)->totallyneutralgrandchildren.insert(new_totally_neutral_child);
						// new_totally_neutral_child->grandparents.insert(*parent_ref);

						(*parent_ref)->neutralchildren.remove(current_polyhedron);
						(*parent_ref)->children.remove(current_polyhedron);

						(*parent_ref)->children.push_back(positive_poly);
						(*parent_ref)->children.push_back(negative_poly);
						(*parent_ref)->positivechildren.push_back(positive_poly);
						(*parent_ref)->negativechildren.push_back(negative_poly);
					}

					positive_poly->parents.insert(positive_poly->parents.end(),
												current_polyhedron->parents.begin(),
														current_polyhedron->parents.end());

					negative_poly->parents.insert(negative_poly->parents.end(),
												current_polyhedron->parents.begin(),
														current_polyhedron->parents.end());

					

					new_totally_neutral_child->parents.push_back(positive_poly);
					new_totally_neutral_child->parents.push_back(negative_poly);

					for(list<polyhedron*>::iterator child_ref = current_polyhedron->positivechildren.begin();child_ref!=current_polyhedron->positivechildren.end();child_ref++)
					{
						(*child_ref)->parents.remove(current_polyhedron);
						(*child_ref)->parents.push_back(positive_poly);						
					}					

					positive_poly->children.insert(positive_poly->children.end(),
												current_polyhedron->positivechildren.begin(),
															current_polyhedron->positivechildren.end());

					for(list<polyhedron*>::iterator child_ref = current_polyhedron->negativechildren.begin();child_ref!=current_polyhedron->negativechildren.end();child_ref++)
					{
						(*child_ref)->parents.remove(current_polyhedron);
						(*child_ref)->parents.push_back(negative_poly);
					}

					negative_poly->children.insert(negative_poly->children.end(),
												current_polyhedron->negativechildren.begin(),
															current_polyhedron->negativechildren.end());

					positive_poly->vertices.insert(current_polyhedron->positiveneutralvertices.begin(),current_polyhedron->positiveneutralvertices.end());
					positive_poly->vertices.insert(new_totally_neutral_child->vertices.begin(),new_totally_neutral_child->vertices.end());

					negative_poly->vertices.insert(current_polyhedron->negativeneutralvertices.begin(),current_polyhedron->negativeneutralvertices.end());
					negative_poly->vertices.insert(new_totally_neutral_child->vertices.begin(),new_totally_neutral_child->vertices.end());
								
					new_totally_neutral_child->triangulate(false);

					positive_poly->triangulate(k==for_splitting.size()-1);
					negative_poly->triangulate(k==for_splitting.size()-1);
					
					statistic.append_polyhedron(k-1, new_totally_neutral_child);					
					
					statistic.insert_polyhedron(k, positive_poly, current_polyhedron);
					statistic.insert_polyhedron(k, negative_poly, current_polyhedron);					

					statistic.delete_polyhedron(k, current_polyhedron);

					delete current_polyhedron;
				}

				k++;
			}
		}

		
		sizevector.clear();
		for(int s = 0;s<statistic.size();s++)
		{
			sizevector.push_back(statistic.row_size(s));
			cout << statistic.row_size(s) << ", ";
		}
		
		cout << endl;

		/*
		for(polyhedron* topr_ref = statistic.rows[statistic.size()-1];topr_ref!=statistic.row_ends[statistic.size()-1]->next_poly;topr_ref=topr_ref->next_poly)
		{
			cout << ((toprow*)topr_ref)->condition << endl;
		}
		*/

		// step_me(0);

	}

	void set_correction_factors(int order)
		{
			for(int remaining_order = correction_factors.size();remaining_order<order;remaining_order++)
			{
				multiset<my_ivec> factor_templates;
				multiset<my_ivec> final_factors;				

				my_ivec orig_template = my_ivec();				

				for(int i = 1;i<number_of_parameters-remaining_order+1;i++)
				{					
					bool in_cycle = false;
					for(int j = 0;j<=remaining_order;j++)					{
						
						multiset<my_ivec>::iterator fac_ref = factor_templates.begin();

						do
						{
							my_ivec current_template;
							if(!in_cycle)
							{
								current_template = orig_template;
								in_cycle = true;
							}
							else
							{
								current_template = (*fac_ref);
								fac_ref++;
							}							
							
							current_template.ins(current_template.size(),i);

							// cout << "template:" << current_template << endl;
							
							if(current_template.size()==remaining_order+1)
							{
								final_factors.insert(current_template);
							}
							else
							{
								factor_templates.insert(current_template);
							}
						}
						while(fac_ref!=factor_templates.end());
					}
				}	

				correction_factors.push_back(final_factors);			

			}
		}



	mat sample_mat(int n)
	{
		mat sample_mat;
		map<double,toprow*> ordered_toprows;

		/// \TODO tady je to spatne, tady nesmi byt conditions.size(), viz RARX.bayes()
		if(conditions.size()-2-number_of_parameters>=0)
		{
			double sum = 0;

			for(polyhedron* top_ref = statistic.rows[number_of_parameters-1];top_ref!=statistic.end_poly;top_ref=top_ref->next_poly)
			{
				toprow* current_top = (toprow*)top_ref;

				sum+=current_top->probability;
				ordered_toprows.insert(pair<double,toprow*>(sum,current_top));
			}
		}
		else
		{
			throw new exception("You are trying to sample from density that is not determined (parameters can't be integrated out)!");
		}

		while(sample_mat.cols()<n)
		{
			double rnumber = randu();

			toprow* sampled_toprow = ordered_toprows.upper_bound(rnumber)->second;

			rnumber = randu();

			map<double,set<vertex*>>::iterator tri_ref = sampled_toprow->triangulation.begin()
			double sum = 0;

			for(tri_ref = sampled_toprow->triangulation.begin();tri_ref!=sampled_toprow->triangulation.end();tri_ref++)
			{
				sum += (*tri_ref).first;

				if(sum/sampled_toprow->probability >= rnumber)
					break;
			}


		}

		return sample_mat;
	}

protected:

	/// A method for creating plain default statistic representing only the range of the parameter space.
    void create_statistic(int number_of_parameters)
	{
		/*
		for(int i = 0;i<number_of_parameters;i++)
		{
			vec condition_vec = zeros(number_of_parameters+1);
			condition_vec[i+1]  = 1;

			condition* new_condition = new condition(condition_vec);
			
			conditions.push_back(new_condition);
		}
		*/

		// An empty vector of coordinates.
		vec origin_coord;	

		// We create an origin - this point will have all the coordinates zero, but now it has an empty vector of coords.
		vertex *origin = new vertex(origin_coord);

		origin->my_emlig = this;
		
		/*
		// As a statistic, we have to create a vector of vectors of polyhedron pointers. It will then represent the Hasse
		// diagram. First we create a vector of polyhedrons..
		list<polyhedron*> origin_vec;

		// ..we fill it with the origin..
		origin_vec.push_back(origin);

		// ..and we fill the statistic with the created vector.
		statistic.push_back(origin_vec);
		*/

		statistic = *(new c_statistic());		
		
		statistic.append_polyhedron(0, origin);

		// Now we have a statistic for a zero dimensional space. Regarding to how many dimensional space we need to 
		// describe, we have to widen the descriptional default statistic. We use an iterative procedure as follows:
		for(int i=0;i<number_of_parameters;i++)
		{
			// We first will create two new vertices. These will be the borders of the parameter space in the dimension
			// of newly added parameter. Therefore they will have all coordinates except the last one zero. We get the 
			// right amount of zero cooridnates by reading them from the origin
			vec origin_coord = origin->get_coordinates();						

			// And we incorporate the nonzero coordinates into the new cooordinate vectors
			vec origin_coord1 = concat(origin_coord,-max_range); 
			vec origin_coord2 = concat(origin_coord,max_range);				 
					

			// Now we create the points
			vertex* new_point1 = new vertex(origin_coord1);
			vertex* new_point2 = new vertex(origin_coord2);

			new_point1->my_emlig = this;
			new_point2->my_emlig = this;
			
			//*********************************************************************************************************
			// The algorithm for recursive build of a new Hasse diagram representing the space structure from the old
			// diagram works so that you create two copies of the old Hasse diagram, you shift them up one level (points
			// will be segments, segments will be areas etc.) and you connect each one of the original copied polyhedrons
			// with its offspring by a parent-child relation. Also each of the segments in the first (second) copy is
			// connected to the first (second) newly created vertex by a parent-child relation.
			//*********************************************************************************************************


			/*
			// Create the vectors of vectors of pointers to polyhedrons to hold the copies of the old Hasse diagram
			vector<vector<polyhedron*>> new_statistic1;
			vector<vector<polyhedron*>> new_statistic2;
			*/

			c_statistic* new_statistic1 = new c_statistic();
			c_statistic* new_statistic2 = new c_statistic();

			
			// Copy the statistic by rows			
			for(int j=0;j<statistic.size();j++)
			{
				

				// an element counter
				int element_number = 0;

				/*
				vector<polyhedron*> supportnew_1;
				vector<polyhedron*> supportnew_2;

				new_statistic1.push_back(supportnew_1);
				new_statistic2.push_back(supportnew_2);
				*/

				// for each polyhedron in the given row
				for(polyhedron* horiz_ref = statistic.rows[j];horiz_ref!=statistic.get_end();horiz_ref=horiz_ref->next_poly)
				{	
					// Append an extra zero coordinate to each of the vertices for the new dimension
					// If vert_ref is at the first index => we loop through vertices
					if(j == 0)
					{
						// cast the polyhedron pointer to a vertex pointer and push a zero to its vector of coordinates
						((vertex*) horiz_ref)->push_coordinate(0);
					}
					/*
					else
					{
						((toprow*) (*horiz_ref))->condition.ins(0,0);
					}*/

					// if it has parents
					if(!horiz_ref->parents.empty())
					{
						// save the relative address of this child in a vector kids_rel_addresses of all its parents.
						// This information will later be used for copying the whole Hasse diagram with each of the
						// relations contained within.
						for(list<polyhedron*>::iterator parent_ref = horiz_ref->parents.begin();parent_ref != horiz_ref->parents.end();parent_ref++)
						{
							(*parent_ref)->kids_rel_addresses.push_back(element_number);							
						}						
					}

					// **************************************************************************************************
					// Here we begin creating a new polyhedron, which will be a copy of the old one. Each such polyhedron
					// will be created as a toprow, but this information will be later forgotten and only the polyhedrons
					// in the top row of the Hasse diagram will be considered toprow for later use.
					// **************************************************************************************************

					// First we create vectors specifying a toprow condition. In the case of a preconstructed statistic
					// this condition will be a vector of zeros. There are two vectors, because we need two copies of 
					// the original Hasse diagram.
					vec vec1(number_of_parameters+1);
					vec1.zeros();

					vec vec2(number_of_parameters+1);
					vec2.zeros();

					// We create a new toprow with the previously specified condition.
					toprow* current_copy1 = new toprow(vec1, 0);
					toprow* current_copy2 = new toprow(vec2, 0);

					current_copy1->my_emlig = this;
					current_copy2->my_emlig = this;

					// The vertices of the copies will be inherited, because there will be a parent/child relation
					// between each polyhedron and its offspring (comming from the copy) and a parent has all the
					// vertices of its child plus more.
					for(set<vertex*>::iterator vertex_ref = horiz_ref->vertices.begin();vertex_ref!=horiz_ref->vertices.end();vertex_ref++)
					{
						current_copy1->vertices.insert(*vertex_ref);
						current_copy2->vertices.insert(*vertex_ref);						
					}
					
					// The only new vertex of the offspring should be the newly created point.
					current_copy1->vertices.insert(new_point1);
					current_copy2->vertices.insert(new_point2);					
					
					// This method guarantees that each polyhedron is already triangulated, therefore its triangulation
					// is only one set of vertices and it is the set of all its vertices.
					set<vertex*> t_simplex1;
					set<vertex*> t_simplex2;

					t_simplex1.insert(current_copy1->vertices.begin(),current_copy1->vertices.end());
					t_simplex2.insert(current_copy2->vertices.begin(),current_copy2->vertices.end());
					
					current_copy1->triangulation.insert(pair<double,set<vertex*>>(0,t_simplex1));
					current_copy2->triangulation.insert(pair<double,set<vertex*>>(0,t_simplex2));					
					
					// Now we have copied the polyhedron and we have to copy all of its relations. Because we are copying
					// in the Hasse diagram from bottom up, we always have to copy the parent/child relations to all the
					// kids and when we do that and know the child, in the child we will remember the parent we came from.
					// This way all the parents/children relations are saved in both the parent and the child.
					if(!horiz_ref->kids_rel_addresses.empty())
					{
						for(list<int>::iterator kid_ref = horiz_ref->kids_rel_addresses.begin();kid_ref!=horiz_ref->kids_rel_addresses.end();kid_ref++)
						{	
							polyhedron* new_kid1 = new_statistic1->rows[j-1];
							polyhedron* new_kid2 = new_statistic2->rows[j-1];

							// THIS IS NOT EFFECTIVE: It could be improved by having the list indexed for new_statistic, but
							// not indexed for statistic. Hopefully this will not cause a big slowdown - happens only offline.
							if(*kid_ref)
							{
								for(int k = 1;k<=(*kid_ref);k++)
								{
									new_kid1=new_kid1->next_poly;
									new_kid2=new_kid2->next_poly;
								}
							}
							
							// find the child and save the relation to the parent
							current_copy1->children.push_back(new_kid1);
							current_copy2->children.push_back(new_kid2);

							// in the child save the parents' address
							new_kid1->parents.push_back(current_copy1);
							new_kid2->parents.push_back(current_copy2);
						}						

						// Here we clear the parents kids_rel_addresses vector for later use (when we need to widen the
						// Hasse diagram again)
						horiz_ref->kids_rel_addresses.clear();
					}
					// If there were no children previously, we are copying a polyhedron that has been a vertex before.
					// In this case it is a segment now and it will have a relation to its mother (copywise) and to the
					// newly created point. Here we create the connection to the new point, again from both sides.
					else
					{
						// Add the address of the new point in the former vertex
						current_copy1->children.push_back(new_point1);
						current_copy2->children.push_back(new_point2);

						// Add the address of the former vertex in the new point
						new_point1->parents.push_back(current_copy1);
						new_point2->parents.push_back(current_copy2);
					}

					// Save the mother in its offspring
					current_copy1->children.push_back(horiz_ref);
					current_copy2->children.push_back(horiz_ref);

					// Save the offspring in its mother
					horiz_ref->parents.push_back(current_copy1);
					horiz_ref->parents.push_back(current_copy2);	
								
					
					// Add the copies into the relevant statistic. The statistic will later be appended to the previous
					// Hasse diagram
					new_statistic1->append_polyhedron(j,current_copy1);
					new_statistic2->append_polyhedron(j,current_copy2);
					
					// Raise the count in the vector of polyhedrons
					element_number++;			
					
				}
				
			}

			/*
			statistic.begin()->push_back(new_point1);
			statistic.begin()->push_back(new_point2);
			*/

			statistic.append_polyhedron(0, new_point1);
			statistic.append_polyhedron(0, new_point2);

			// Merge the new statistics into the old one. This will either be the final statistic or we will
			// reenter the widening loop. 
			for(int j=0;j<new_statistic1->size();j++)
			{
				/*
				if(j+1==statistic.size())
				{
					list<polyhedron*> support;
					statistic.push_back(support);
				}
				
				(statistic.begin()+j+1)->insert((statistic.begin()+j+1)->end(),new_statistic1[j].begin(),new_statistic1[j].end());
				(statistic.begin()+j+1)->insert((statistic.begin()+j+1)->end(),new_statistic2[j].begin(),new_statistic2[j].end());
				*/
				statistic.append_polyhedron(j+1,new_statistic1->rows[j],new_statistic1->row_ends[j]);
				statistic.append_polyhedron(j+1,new_statistic2->rows[j],new_statistic2->row_ends[j]);
			}			
		}

		/*
		vector<list<toprow*>> toprow_statistic;
		int line_count = 0;

		for(vector<list<polyhedron*>>::iterator polyhedron_ref = ++statistic.begin(); polyhedron_ref!=statistic.end();polyhedron_ref++)
		{
			list<toprow*> support_list;
			toprow_statistic.push_back(support_list);						

			for(list<polyhedron*>::iterator polyhedron_ref2 = polyhedron_ref->begin(); polyhedron_ref2 != polyhedron_ref->end(); polyhedron_ref2++)
			{
				toprow* support_top = (toprow*)(*polyhedron_ref2);

				toprow_statistic[line_count].push_back(support_top);
			}

			line_count++;
		}*/

		/*
		vector<int> sizevector;
		for(int s = 0;s<statistic.size();s++)
		{
			sizevector.push_back(statistic.row_size(s));
		}
		*/
		
	}


	
	
};



//! Robust Bayesian AR model for Multicriteria-Laplace-Inverse-Gamma density
class RARX //: public BM
{
private:

	

	int window_size;	

	list<vec> conditions;

public:
	emlig* posterior;

	RARX(int number_of_parameters, const int window_size)//:BM()
	{
		posterior = new emlig(number_of_parameters);

		this->window_size = window_size;		
	};

	void bayes(const itpp::vec &yt, const itpp::vec &cond = "")
	{
		conditions.push_back(yt);		

		//posterior->step_me(0);
		
		/// \TODO tohle je spatne, tady musi byt jiny vypocet poctu podminek, kdyby nejaka byla multiplicitni, tak tohle bude spatne
		if(conditions.size()>window_size && window_size!=0)
		{			
			posterior->add_and_remove_condition(yt,conditions.front());
			conditions.pop_front();

			//posterior->step_me(1);
		}
		else
		{
			posterior->add_condition(yt);
		}
				
	}

};



#endif //TRAGE_H
