/*!
  \file
  \brief Bayesian Filtering for generalized autoregressive (ARX) model
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef AR_H
#define AR_H

#include <itpp/itbase.h>
#include "../stat/libFN.h"
#include "../stat/libEF.h"

using namespace itpp;

/*!
* \brief Linear Autoregressive model with Gaussian noise

Regression of the following kind:
\f[
y_t = \theta_1 \psi_1 + \theta_2 + \psi_2 +\ldots + \theta_n \psi_n + r e_t
\f]
where unknown parameters \c rv are \f$[\theta r]\f$, regression vector \f$\psi=\psi(y_{1:t},u_{1:t})\f$ is a known function of past outputs and exogeneous variables \f$u_t\f$. Distrubances \f$e_t\f$ are supposed to be normally distributed:
\f[
e_t \sim \mathcal{N}(0,1).
\f]

Extension for time-variant parameters \f$\theta_t,r_t\f$ may be achived using exponential forgetting (Kulhavy and Zarrop, 1993). In such a case, the forgetting factor \c frg \f$\in <0,1>\f$ should be given in the constructor. Time-invariant parameters are estimated for \c frg = 1.
*/
class ARX: public BM {
protected:
	//! Posterior estimate of \f$\theta,r\f$ in the form of Normal-inverse Wishart density
	egiw est;
	//! cached value of est.V
	ldmat &V;
	//! cached value of est.nu
	double &nu;
	//! forgetting factor 
	double frg;
	//! cached value of lognc() in the previous step (used in evaluation of \c ll )
	double last_lognc;
	//! total likelihood
	double tll;
public:
	//! Full constructor
	ARX (RV &rv, mat &V0, double &nu0, double frg0=1.0) : BM(rv),est(rv,V0,nu0), V(est._V()), nu(est._nu()), frg(frg0){last_lognc=est.lognc();tll=0.0;};
	void set_parameters(mat &V0, double &nu0){est._V()=V0;est._nu()=nu0;last_lognc=est.lognc();tll=0.0;}
	void get_parameters(mat &V0, double &nu0){V0=est._V().to_mat(); nu0=est._nu();}
	//! Here \f$dt = [y_t psi_t] \f$.
	void bayes ( const vec &dt );
	epdf& _epdf() {return est;}
	//! Brute force structure estimation.\return indeces of accepted regressors.
	ivec structure_est(egiw Eg0);
	//!access function
	double _tll(){return tll;}
};


#endif // AR_H


