/*!
  \file
  \brief Robust Bayesian auto-regression model
  \author Jan Sindelar.
*/

#ifndef ROBUST_H
#define ROBUST_H

#include <stat/exp_family.h>
#include <itpp/itbase.h>
#include <itpp/base/random.h>
#include <map>
#include <limits>
#include <vector>
#include <list>
#include <set>
#include <algorithm>
	
using namespace bdm;
using namespace std;
using namespace itpp;

static Exponential_RNG ExpRNG;

const double max_range = 5;//numeric_limits<double>::max()/10e-10;

/// An enumeration of possible actions performed on the polyhedrons. We can merge them or split them.
enum actions {MERGE, SPLIT};

// Forward declaration of polyhedron, vertex and emlig
class polyhedron;
class vertex;
class emlig;

/*
class t_simplex
{
public:
	set<vertex*> minima;

	set<vertex*> simplex;

	t_simplex(vertex* origin_vertex)
	{
		simplex.insert(origin_vertex);
		minima.insert(origin_vertex);
	}
};*/

/// A class representing a single condition that can be added to the emlig. A condition represents data entries in a statistical model.
class condition
{	
public:
	/// Value of the condition representing the data
	vec value;	

	/// Mulitplicity of the given condition may represent multiple occurences of same data entry.
	int multiplicity;

	/// Default constructor of condition class takes the value of data entry and creates a condition with multiplicity 1 (first occurence of the data).
	condition(vec value)
	{
		this->value = value;
		multiplicity = 1;
	}
};

class simplex
{
	

public:

	set<vertex*> vertices;

	double probability;

	double volume;

	vector<multimap<double,double>> positive_gamma_parameters;

	vector<multimap<double,double>> negative_gamma_parameters;

	double positive_gamma_sum;

	double negative_gamma_sum;

	double min_beta;
	

	simplex(set<vertex*> vertices)
	{
		this->vertices.insert(vertices.begin(),vertices.end());
		probability = 0;
	}

	simplex(vertex* vertex)
	{
		this->vertices.insert(vertex);
		probability = 0;
	}

	void clear_gammas()
	{
		positive_gamma_parameters.clear();
		negative_gamma_parameters.clear();		
		
		positive_gamma_sum = 0;
		negative_gamma_sum = 0;

		min_beta = numeric_limits<double>::max();
	}

	void insert_gamma(int order, double weight, double beta)
	{
		if(weight>=0)
		{
			while(positive_gamma_parameters.size()<order+1)
			{
				multimap<double,double> map;
				positive_gamma_parameters.push_back(map);
			}

			positive_gamma_sum += weight;

			positive_gamma_parameters[order].insert(pair<double,double>(weight,beta));		
		}
		else
		{
			while(negative_gamma_parameters.size()<order+1)
			{
				multimap<double,double> map;
				negative_gamma_parameters.push_back(map);
			}

			negative_gamma_sum -= weight;

			negative_gamma_parameters[order].insert(pair<double,double>(-weight,beta));
		}

		if(beta < min_beta)
		{
			min_beta = beta;
		}
	}
};


/// A class describing a single polyhedron of the split complex. From a collection of such classes a Hasse diagram
/// of the structure in the exponent of a Laplace-Inverse-Gamma density will be created.
class polyhedron
{
	/// A property having a value of 1 usually, with higher value only if the polyhedron arises as a coincidence of
	/// more than just the necessary number of conditions. For example if a newly created line passes through an already
	/// existing point, the points multiplicity will rise by 1.
	int multiplicity;	

	/// A property representing the position of the polyhedron related to current condition with relation to which we
	/// are splitting the parameter space (new data has arrived). This property is setup within a classification procedure and
	/// is only valid while the new condition is being added. It has to be reset when new condition is added and new classification
	/// has to be performed.
	int split_state;

	/// A property representing the position of the polyhedron related to current condition with relation to which we
	/// are merging the parameter space (data is being deleted usually due to a moving window model which is more adaptive and 
	/// steps in for the forgetting in a classical Gaussian AR model). This property is setup within a classification procedure and
	/// is only valid while the new condition is being removed. It has to be reset when new condition is removed and new classification
	/// has to be performed.
	int merge_state;

			

public:
	/// A pointer to the multi-Laplace inverse gamma distribution this polyhedron belongs to.
	emlig* my_emlig;

	/// A list of polyhedrons parents within the Hasse diagram.
	list<polyhedron*> parents;

	/// A list of polyhedrons children withing the Hasse diagram.
	list<polyhedron*> children;

	/// All the vertices of the given polyhedron
	set<vertex*> vertices;

	/// The conditions that gave birth to the polyhedron. If some of them is removed, the polyhedron ceases to exist.
	set<condition*> parentconditions;

	/// A list used for storing children that lie in the positive region related to a certain condition
	list<polyhedron*> positivechildren;

	/// A list used for storing children that lie in the negative region related to a certain condition
	list<polyhedron*> negativechildren;

	/// Children intersecting the condition
	list<polyhedron*> neutralchildren;

	/// A set of grandchildren of the polyhedron that when new condition is added lie exactly on the condition hyperplane. These grandchildren
	/// behave differently from other grandchildren, when the polyhedron is split. New grandchild is not necessarily created on the crossection of
	/// the polyhedron and new condition.
	set<polyhedron*> totallyneutralgrandchildren;

	/// A set of children of the polyhedron that when new condition is added lie exactly on the condition hyperplane. These children
	/// behave differently from other children, when the polyhedron is split. New child is not necessarily created on the crossection of
	/// the polyhedron and new condition.
	set<polyhedron*> totallyneutralchildren;

	/// Reverse relation to the totallyneutralgrandchildren set is needed for merging of already existing polyhedrons to keep 
	/// totallyneutralgrandchildren list up to date.
	set<polyhedron*> grandparents;

	/// Vertices of the polyhedron classified as positive related to an added condition. When the polyhderon is split by the new condition,
	/// these vertices will belong to the positive part of the splitted polyhedron.
	set<vertex*> positiveneutralvertices;

	/// Vertices of the polyhedron classified as negative related to an added condition. When the polyhderon is split by the new condition,
	/// these vertices will belong to the negative part of the splitted polyhedron.
	set<vertex*> negativeneutralvertices;

	/// A bool specifying if the polyhedron lies exactly on the newly added condition or not.
	bool totally_neutral;

	/// When two polyhedrons are merged, there always exists a child lying on the former border of the polyhedrons. This child manages the merge
	/// of the two polyhedrons. This property gives us the address of the mediator child.
	polyhedron* mergechild;

	/// If the polyhedron serves as a mergechild for two of its parents, we need to have the address of the parents to access them. This 
	/// is the pointer to the positive parent being merged.
	polyhedron* positiveparent;

	/// If the polyhedron serves as a mergechild for two of its parents, we need to have the address of the parents to access them. This 
	/// is the pointer to the negative parent being merged.
	polyhedron* negativeparent;	

	/// Adressing withing the statistic. Next_poly is a pointer to the next polyhedron in the statistic on the same level (if this is a point,
	/// next_poly will be a point etc.).
	polyhedron* next_poly;

	/// Adressing withing the statistic. Prev_poly is a pointer to the previous polyhedron in the statistic on the same level (if this is a point,
	/// next_poly will be a point etc.).
	polyhedron* prev_poly;

	/// A property counting the number of messages obtained from children within a classification procedure of position of the polyhedron related
	/// an added/removed condition. If the message counter reaches the number of children, we know the polyhedrons' position has been fully classified.
	int message_counter;

	/// List of triangulation polyhedrons of the polyhedron given by their relative vertices. 
	set<simplex*> triangulation;

	/// A list of relative addresses serving for Hasse diagram construction.
	list<int> kids_rel_addresses;

	/// Default constructor
	polyhedron()
	{
		multiplicity = 1;

		message_counter = 0;

		totally_neutral = NULL;

		mergechild = NULL;		
	}
	
	/// Setter for raising multiplicity
	void raise_multiplicity()
	{
		multiplicity++;
	}

	/// Setter for lowering multiplicity
	void lower_multiplicity()
	{
		multiplicity--;
	}

	int get_multiplicity()
	{
		return multiplicity;
	}
	
	/// An obligatory operator, when the class is used within a C++ STL structure like a vector
	int operator==(polyhedron polyhedron2)
	{
		return true;
	}

	/// An obligatory operator, when the class is used within a C++ STL structure like a vector
	int operator<(polyhedron polyhedron2)
	{
		return false;
	}

	
	/// A setter of state of current polyhedron relative to the action specified in the argument. The three possible states of the
	/// polyhedron are -1 - NEGATIVE, 0 - NEUTRAL, 1 - POSITIVE. Neutral state means that either the state has been reset or the polyhedron is
	/// ready to be split/merged.
	int set_state(double state_indicator, actions action)
	{
		switch(action)
		{
			case MERGE:
				merge_state = (int)sign(state_indicator);
				return merge_state;			
			case SPLIT:
				split_state = (int)sign(state_indicator);
				return split_state;		
		}
	}

	/// A getter of state of current polyhedron relative to the action specified in the argument. The three possible states of the
	/// polyhedron are -1 - NEGATIVE, 0 - NEUTRAL, 1 - POSITIVE. Neutral state means that either the state has been reset or the polyhedron is
	/// ready to be split/merged.
	int get_state(actions action)
	{
		switch(action)
		{
			case MERGE:
				return merge_state;			
			break;
			case SPLIT:
				return split_state;
			break;
		}
	}

	/// Method for obtaining the number of children of given polyhedron.
	int number_of_children()
	{
		return children.size();
	}

	/// A method for triangulation of given polyhedron.
	double triangulate(bool should_integrate);	
};


/// A class for representing 0-dimensional polyhedron - a vertex. It will be located in the bottom row of the Hasse
/// diagram representing a complex of polyhedrons. It has its coordinates in the parameter space.
class vertex : public polyhedron
{
	/// A dynamic array representing coordinates of the vertex
	vec coordinates;

public:
	/// A property specifying the value of the density (ted nevim, jestli je to jakoby log nebo ne) above the vertex.
	double function_value;

	/// Default constructor
	vertex();

	/// Constructor of a vertex from a set of coordinates
	vertex(vec coordinates)
	{
		this->coordinates   = coordinates;

		vertices.insert(this);

		simplex* vert_simplex = new simplex(vertices);		

		triangulation.insert(vert_simplex);
	}

	/// A method that widens the set of coordinates of given vertex. It is used when a complex in a parameter
	/// space of certain dimension is established, but the dimension is not known when the vertex is created.
	void push_coordinate(double coordinate)
	{
		coordinates  = concat(coordinates,coordinate);		
	}

	/// A method obtaining the set of coordinates of a vertex. These coordinates are not obtained as a pointer 
	/// (not given by reference), but a new copy is created (they are given by value).
	vec get_coordinates()
	{
		return coordinates;
	}
		
};


/// A class representing a polyhedron in a top row of the complex. Such polyhedron has a condition that differen   tiates
/// it from polyhedrons in other rows. 
class toprow : public polyhedron
{
	
public:
	double probability;

	vertex* minimal_vertex;

	/// A condition used for determining the function of a Laplace-Inverse-Gamma density resulting from Bayesian estimation
	vec condition_sum;

	int condition_order;

	/// Default constructor
	toprow(){};

	/// Constructor creating a toprow from the condition
	toprow(condition *condition, int condition_order)
	{
		this->condition_sum   = condition->value;
		this->condition_order = condition_order;
	}

	toprow(vec condition_sum, int condition_order)
	{
		this->condition_sum   = condition_sum;
		this->condition_order = condition_order;
	}

	double integrate_simplex(simplex* simplex, char c);

};







class c_statistic
{

public:
	polyhedron* end_poly;
	polyhedron* start_poly;

	vector<polyhedron*> rows;

	vector<polyhedron*> row_ends;

	c_statistic()
	{
		end_poly   = new polyhedron();
		start_poly = new polyhedron();
	};

	~c_statistic()
	{
		delete end_poly;
		delete start_poly;
	}

	void append_polyhedron(int row, polyhedron* appended_start, polyhedron* appended_end)
	{
		if(row>((int)rows.size())-1)
		{
			if(row>rows.size())
			{
				throw new exception("You are trying to append a polyhedron whose children are not in the statistic yet!");
				return;
			}

			rows.push_back(end_poly);
			row_ends.push_back(end_poly);
		}

		// POSSIBLE FAILURE: the function is not checking if start and end are connected

		if(rows[row] != end_poly)
		{
			appended_start->prev_poly = row_ends[row];
			row_ends[row]->next_poly = appended_start;			
						
		}
		else if((row>0 && rows[row-1]!=end_poly)||row==0)
		{
			appended_start->prev_poly = start_poly;
			rows[row]= appended_start;			
		}
		else
		{
			throw new exception("Wrong polyhedron insertion into statistic: missing intermediary polyhedron!");
		}

		appended_end->next_poly = end_poly;
		row_ends[row] = appended_end;
	}

	void append_polyhedron(int row, polyhedron* appended_poly)
	{
		append_polyhedron(row,appended_poly,appended_poly);
	}

	void insert_polyhedron(int row, polyhedron* inserted_poly, polyhedron* following_poly)
	{		
		if(following_poly != end_poly)
		{
			inserted_poly->next_poly = following_poly;
			inserted_poly->prev_poly = following_poly->prev_poly;

			if(following_poly->prev_poly == start_poly)
			{
				rows[row] = inserted_poly;
			}
			else
			{				
				inserted_poly->prev_poly->next_poly = inserted_poly;								
			}

			following_poly->prev_poly = inserted_poly;
		}
		else
		{
			this->append_polyhedron(row, inserted_poly);
		}		
	
	}


	

	void delete_polyhedron(int row, polyhedron* deleted_poly)
	{
		if(deleted_poly->prev_poly != start_poly)
		{
			deleted_poly->prev_poly->next_poly = deleted_poly->next_poly;
		}
		else
		{
			rows[row] = deleted_poly->next_poly;
		}

		if(deleted_poly->next_poly!=end_poly)
		{
			deleted_poly->next_poly->prev_poly = deleted_poly->prev_poly;
		}
		else
		{
			row_ends[row] = deleted_poly->prev_poly;
		}

		

		deleted_poly->next_poly = NULL;
		deleted_poly->prev_poly = NULL;					
	}

	int size()
	{
		return rows.size();
	}

	polyhedron* get_end()
	{
		return end_poly;
	}

	polyhedron* get_start()
	{
		return start_poly;
	}

	int row_size(int row)
	{
		if(this->size()>row && row>=0)
		{
			int row_size = 0;
			
			for(polyhedron* row_poly = rows[row]; row_poly!=end_poly; row_poly=row_poly->next_poly)
			{
				row_size++;
			}

			return row_size;
		}
		else
		{
			throw new exception("There is no row to obtain size from!");
		}
	}
};


class my_ivec : public ivec
{
public:
	my_ivec():ivec(){};

	my_ivec(ivec origin):ivec()
	{
		this->ins(0,origin);
	}

	bool operator>(const my_ivec &second) const
	{
		return max(*this)>max(second);		
	}
	
	bool operator==(const my_ivec &second) const
	{
		return max(*this)==max(second);	
	}

	bool operator<(const my_ivec &second) const
	{
		return !(((*this)>second)||((*this)==second));
	}

	bool operator!=(const my_ivec &second) const
	{
		return !((*this)==second);
	}

	bool operator<=(const my_ivec &second) const
	{
		return !((*this)>second);
	}

	bool operator>=(const my_ivec &second) const
	{
		return !((*this)<second);
	}

	my_ivec right(my_ivec original)
	{
		
	}
};







//! Conditional(e) Multicriteria-Laplace-Inverse-Gamma distribution density
class emlig // : eEF
{

	/// A statistic in a form of a Hasse diagram representing a complex of convex polyhedrons obtained as a result
	/// of data update from Bayesian estimation or set by the user if this emlig is a prior density
	

	vector<list<polyhedron*>> for_splitting;
		
	vector<list<polyhedron*>> for_merging;

	list<condition*> conditions;

	double normalization_factor;

	int condition_order;

	double last_log_nc;

	

	void alter_toprow_conditions(condition *condition, bool should_be_added)
	{
		for(polyhedron* horiz_ref = statistic.rows[statistic.size()-1];horiz_ref!=statistic.get_end();horiz_ref=horiz_ref->next_poly)
		{
			set<vertex*>::iterator vertex_ref = horiz_ref->vertices.begin();

			do
			{
				vertex_ref++;

				if(vertex_ref==horiz_ref->vertices.end())
				{
					return;
				}
			}
			while((*vertex_ref)->parentconditions.find(condition)!=(*vertex_ref)->parentconditions.end());

			
			
			vec appended_coords = (*vertex_ref)->get_coordinates();
			appended_coords.ins(0,-1.0);
			
			double product = appended_coords*condition->value;

			if(should_be_added)
			{
				((toprow*) horiz_ref)->condition_order++;

				if(product>0)
				{
					((toprow*) horiz_ref)->condition_sum += condition->value;
				}
				else
				{
					((toprow*) horiz_ref)->condition_sum -= condition->value;
				}
			}
			else
			{ 
				((toprow*) horiz_ref)->condition_order--;

				if(product<0)			
				{
					((toprow*) horiz_ref)->condition_sum += condition->value;
				}
				else
				{
					((toprow*) horiz_ref)->condition_sum -= condition->value;
				}
			}				
		}
	}


	/// A method for recursive classification of polyhedrons with respect to SPLITting and MERGEing conditions.
	void send_state_message(polyhedron* sender, condition *toadd, condition *toremove, int level)
	{			

		// We translate existence of toremove and toadd conditions to booleans for ease of manipulation
		bool shouldmerge    = (toremove != NULL);
		bool shouldsplit    = (toadd != NULL);
		
		// If such operation is desired, in the following cycle we send a message about polyhedrons classification
		// to all its parents. We loop through the parents and report the child sending its message.
		if(shouldsplit||shouldmerge)
		{
			for(list<polyhedron*>::iterator parent_iterator = sender->parents.begin();parent_iterator!=sender->parents.end();parent_iterator++)
			{
				// We set an individual pointer to the value at parent_iterator for ease of use
				polyhedron* current_parent = *parent_iterator;

				// The message_counter counts the number of messages received by the parent
				current_parent->message_counter++;

				// If the child is the last one to send its message, the parent can as well be classified and
				// send its message further up.
				bool is_last  = (current_parent->message_counter == current_parent->number_of_children());

				// Certain properties need to be set if this is the first message received by the parent
				bool is_first = (current_parent->message_counter == 1);

				// This boolean watches for polyhedrons that are already out of the game for further MERGEing
				// and SPLITting purposes. This may seem quite straightforward at first, but because of all
				// the operations involved it may be quite complicated. For example a polyhedron laying in the
				// positive side of the MERGEing hyperplane should not be split, because it lays in the positive
				// part of the location parameter space relative to the SPLITting hyperplane, but because it
				// is merged with its MERGE negative counterpart, which is being SPLIT, the polyhedron itself
				// will be SPLIT after it has been merged and needs to retain all properties needed for the
				// purposes of SPLITting.
				bool out_of_the_game = true;

				if(shouldmerge)
				{
					// get the MERGE state of the child
					int child_state  = sender->get_state(MERGE);
					// get the MERGE state of the parent so far, the parent can be partially classified
					int parent_state = current_parent->get_state(MERGE);

					// In case this is the first message received by the parent, its state has not been set yet
					// and therefore it inherits the MERGE state of the child. On the other hand if the state
					// of the parent is 0, all the children so far were neutral and if the next child isn't 
					// neutral the parent should be in state of the child again. 
					if(parent_state == 0||is_first)
					{
						parent_state = current_parent->set_state(child_state, MERGE);						
					}					

					// If a child is contained in the hyperplane of a condition that should be removed and it is
					// not of multiplicity higher than 1, it will later serve as a merger for two of its parents
					// each lying on one side of the removed hyperplane (one being classified MERGE positive, the
					// other MERGE negative). Here we set the possible merger candidates.
					if(child_state == 0)
					{
						if(current_parent->mergechild == NULL)
						{
							current_parent->mergechild = sender;
						}							
					}					

					// If the parent obtained a message from the last one of its children we have to classify it
					// with respect to the MERGE condition.
					if(is_last)
					{						
						// If the parent is a toprow from the top row of the Hasse diagram, we alter the condition
						// sum and condition order with respect to on which side of the cutting hyperplane the
						// toprow is located.
						if(level == number_of_parameters-1)
						{
							// toprow on the positive side
							if(parent_state == 1)
							{
								((toprow*)current_parent)->condition_sum-=toremove->value;							
							}

							// toprow on the negative side
							if(parent_state == -1)
							{
								((toprow*)current_parent)->condition_sum+=toremove->value;							
							}
						}

						// lowering the condition order. 
						// REMARK: This maybe could be done more globally for the whole statistic.
						((toprow*)current_parent)->condition_order--;
						
						// If the parent is a candidate for being MERGEd
						if(current_parent->mergechild != NULL)
						{
							// It might not be out of the game
							out_of_the_game = false;

							// If the mergechild multiplicity is 1 it will disappear after merging
							if(current_parent->mergechild->get_multiplicity()==1)
							{
								// and because we need the child to have an address of the two parents it is
								// supposed to merge, we assign the address of current parent to one of the
								// two pointers existing in the child for this purpose regarding to its position
								// in the location parameter space with respect to the MERGE hyperplane.
								if(parent_state > 0)
								{							
									current_parent->mergechild->positiveparent = current_parent;							
								}

								if(parent_state < 0)
								{							
									current_parent->mergechild->negativeparent = current_parent;							
								}
							}
							else
							{
								// If the mergechild has higher multiplicity, it will not disappear after the
								// condition is removed and the parent will still be out of the game, because
								// no MERGEing will occur.
								out_of_the_game = true;
							}
						}						
						
						// If so far the parent is out of the game, it is the toprow polyhedron and there will 
						// be no SPLITting, we compute its probability integral by summing all the integral
						// from the simplices contained in it.
						if(out_of_the_game)
						{
							if((level == number_of_parameters - 1) && (!shouldsplit))
							{
								toprow* cur_par_toprow = ((toprow*)current_parent);
								cur_par_toprow->probability = 0.0;							

								for(set<simplex*>::iterator s_ref = current_parent->triangulation.begin();s_ref!=current_parent->triangulation.end();s_ref++)
								{
									double cur_prob = cur_par_toprow->integrate_simplex((*s_ref),'C');
									
									cur_par_toprow->probability += cur_prob;								
								}

								normalization_factor += cur_par_toprow->probability;								
							}
						}

						// If the parent is classified MERGE neutral, it will serve as a merger for two of its
						// parents so we report it to the for_merging list.
						if(parent_state == 0)
						{
							for_merging[level+1].push_back(current_parent);														
						}												
					}					
				}

				// In the second part of the classification procedure, we will classify the parent polyhedron
				// for the purposes of SPLITting. Since splitting comes from a parent that is being split by
				// creating a neutral child that cuts the split polyhedron in two parts, the created child has
				// to be connected to all the neutral grandchildren of the source parent. We therefore have to
				// report all such grandchildren of the parent. More complication is brought in by grandchildren
				// that have not been created in the process of splitting, but were classified SPLIT neutral
				// already in the classification stage. Such grandchildren and children were already present
				// in the Hasse diagram befor the SPLITting condition emerged. We call such object totallyneutral.
				// They have to be watched and treated separately.
				if(shouldsplit)
				{
					// We report the totally neutral children of the message sending child into the totally neutral
					// grandchildren list of current parent.
					current_parent->totallyneutralgrandchildren.insert(sender->totallyneutralchildren.begin(),sender->totallyneutralchildren.end());
					
					// We need to have the pointers from grandchildren to grandparents as well, we therefore set
					// the opposite relation as well.
					for(set<polyhedron*>::iterator tot_child_ref = sender->totallyneutralchildren.begin();tot_child_ref!=sender->totallyneutralchildren.end();tot_child_ref++)
					{
						(*tot_child_ref)->grandparents.insert(current_parent);
					}

					// If this is the first child to report its total neutrality, the parent inherits its state.
					if(current_parent->totally_neutral == NULL)
					{
						current_parent->totally_neutral = sender->totally_neutral;
					}
					// else the parent is totally neutral only if all the children up to now are totally neutral.
					else
					{
						current_parent->totally_neutral = current_parent->totally_neutral && sender->totally_neutral;
					}

					// For splitting purposes, we have to mark all the children of the given parent by their SPLIT
					// state, because when we split the parent, we create its positive and negative offsprings and
					// its children have to be assigned accordingly.
					switch(sender->get_state(SPLIT))
					{
					case 1:
						// child classified positive
						current_parent->positivechildren.push_back(sender);

						// all the vertices of the positive child are assigned to the positive and neutral vertex
						// set
						current_parent->positiveneutralvertices.insert(sender->vertices.begin(),sender->vertices.end());
					break;
					case 0:
						// child classified neutral
						current_parent->neutralchildren.push_back(sender);

						// all the vertices of the neutral child are assigned to both negative and positive vertex
						// sets
						if(level!=0)
						{
							current_parent->positiveneutralvertices.insert(sender->positiveneutralvertices.begin(),sender->positiveneutralvertices.end());
							current_parent->negativeneutralvertices.insert(sender->negativeneutralvertices.begin(),sender->negativeneutralvertices.end());						
						}
						else
						{
							current_parent->positiveneutralvertices.insert(*sender->vertices.begin());
							current_parent->negativeneutralvertices.insert(*sender->vertices.begin());
						}

						// if the child is totally neutral it is also assigned to the totallyneutralchildren
						if(sender->totally_neutral)
						{
							current_parent->totallyneutralchildren.insert(sender);
						}
							
					break;
					case -1:
						// child classified negative
						current_parent->negativechildren.push_back(sender);
						current_parent->negativeneutralvertices.insert(sender->vertices.begin(),sender->vertices.end());
					break;
					}

					// If the last child has sent its message to the parent, we have to decide if the polyhedron
					// needs to be split. 
					if(is_last)
					{						
						// If the polyhedron extends to both sides of the cutting hyperplane it needs to be SPLIT. Such
						// situation occurs if either the polyhedron has negative and also positive children or
						// if the polyhedron contains neutral children that cross the cutting hyperplane. Such
						// neutral children cannot be totally neutral, since totally neutral children lay within
						// the cutting hyperplane. If the polyhedron is to be cut its state is set to SPLIT neutral
						if((current_parent->negativechildren.size()>0&&current_parent->positivechildren.size()>0)
													||(current_parent->neutralchildren.size()>0&&current_parent->totallyneutralchildren.empty()))
						{
							for_splitting[level+1].push_back(current_parent);							
							current_parent->set_state(0, SPLIT);
						}
						else
						{
							// Else if the polyhedron has a positive number of negative children we set its state
							// to SPLIT negative. In such a case we subtract current condition from the overall
							// condition sum
							if(current_parent->negativechildren.size()>0)
							{
								// set the state
								current_parent->set_state(-1, SPLIT);

								// alter the condition sum
								if(level == number_of_parameters-1)
								{
									((toprow*)current_parent)->condition_sum-=toadd->value;
								}									
							}
							// If the polyhedron has a positive number of positive children we set its state
							// to SPLIT positive. In such a case we add current condition to the overall
							// condition sum
							else if(current_parent->positivechildren.size()>0)
							{
								// set the state 
								current_parent->set_state(1, SPLIT);

								// alter the condition sum
								if(level == number_of_parameters-1)
								{
									((toprow*)current_parent)->condition_sum+=toadd->value;									
								}
							}
							// Else the polyhedron only has children that are totally neutral. In such a case,
							// we mark it totally neutral as well and insert the SPLIT condition into the 
							// parent conditions of the polyhedron. No addition or subtraction is needed in 
							// this case.
							else
							{
								current_parent->raise_multiplicity();
								current_parent->totally_neutral = true;
								current_parent->parentconditions.insert(toadd);
							}

							// In either case we raise the condition order (statistical condition sum significance)
							((toprow*)current_parent)->condition_order++;

							// In case the polyhedron is a toprow and it will not be SPLIT, we compute its probability
							// integral with the altered condition.
							if(level == number_of_parameters - 1 && current_parent->mergechild == NULL)
							{
								toprow* cur_par_toprow = ((toprow*)current_parent);
								cur_par_toprow->probability = 0.0;								
								
								// We compute the integral as a sum over all simplices contained within the
								// polyhedron.
								for(set<simplex*>::iterator s_ref = current_parent->triangulation.begin();s_ref!=current_parent->triangulation.end();s_ref++)
								{
									double cur_prob = cur_par_toprow->integrate_simplex((*s_ref),'C');
									
									cur_par_toprow->probability += cur_prob;
								}

								normalization_factor += cur_par_toprow->probability;
							}

							// If the parent polyhedron is out of the game, so that it will not be MERGEd or
							// SPLIT any more, we will reset the lists specifying its relation with respect
							// to the SPLITting condition, so that they will be clear for future use.
							if(out_of_the_game)
							{
								current_parent->positivechildren.clear();
								current_parent->negativechildren.clear();
								current_parent->neutralchildren.clear();								
								current_parent->totallyneutralgrandchildren.clear();								
								current_parent->positiveneutralvertices.clear();
								current_parent->negativeneutralvertices.clear();
								current_parent->totally_neutral = NULL;
								current_parent->kids_rel_addresses.clear();
							}							
						}
					}
				}

				// Finally if the the parent polyhedron has been SPLIT and MERGE classified, we will send a message
				// about its classification to its parents.
				if(is_last)
				{
					current_parent->mergechild = NULL;
					current_parent->message_counter = 0;

					send_state_message(current_parent,toadd,toremove,level+1);
				}
			
			}

			// We clear the totally neutral children of the child here, because we needed them to be assigned as
			// totally neutral grandchildren to all its parents.
			sender->totallyneutralchildren.clear();			
		}		
	}
	
public:	
	c_statistic statistic;

	vertex* minimal_vertex;

	double min_ll;

	double log_nc;

	

	vector<multiset<my_ivec>> correction_factors;

	int number_of_parameters;

	/// A default constructor creates an emlig with predefined statistic representing only the range of the given
	/// parametric space, where the number of parameters of the needed model is given as a parameter to the constructor.
	emlig(int number_of_parameters, double alpha_deviation, double sigma_deviation, int nu)
	{	
		this->number_of_parameters = number_of_parameters;

		condition_order = nu;
						
		create_statistic(number_of_parameters, alpha_deviation, sigma_deviation);

		//step_me(10);

		min_ll = numeric_limits<double>::max();		

		
		double normalization_factor = 0;
		int counter = 0;
		for(polyhedron* top_ref = statistic.rows[number_of_parameters];top_ref!=statistic.get_end();top_ref=top_ref->next_poly)
		{
			counter++;
			toprow* cur_toprow = (toprow*)top_ref;
				
			set<simplex*>::iterator cur_simplex = cur_toprow->triangulation.begin();
			normalization_factor += cur_toprow->integrate_simplex(*cur_simplex,'X');
		}

		last_log_nc = NULL;
		log_nc = log(normalization_factor);		
		
		cout << "Prior constructed." << endl;
	}

	/// A constructor for creating an emlig when the user wants to create the statistic by himself. The creation of a
	/// statistic is needed outside the constructor. Used for a user defined prior distribution on the parameters.
	emlig(c_statistic statistic, int condition_order)
	{
		this->statistic = statistic;	

		min_ll = numeric_limits<double>::max();

		this->condition_order = condition_order;
	}


	void step_me(int marker)
	{
		set<int> orders;

		for(int i = 0;i<statistic.size();i++)
		{
			//int zero = 0;
			//int one  = 0;
			//int two  = 0;

			for(polyhedron* horiz_ref = statistic.rows[i];horiz_ref!=statistic.get_end();horiz_ref=horiz_ref->next_poly)
			{
				
				
				if(i==statistic.size()-1)
				{
					orders.insert(((toprow*)horiz_ref)->condition_order);
					
					/*
					cout << ((toprow*)horiz_ref)->condition_sum << "   " << ((toprow*)horiz_ref)->probability << endl;
					cout << "Condition: " << ((toprow*)horiz_ref)->condition_sum << endl;
					cout << "Order:" << ((toprow*)horiz_ref)->condition_order << endl;*/
				}
				

				// cout << "Stepped." << endl;

				if(marker==101)
				{
					if(!(*horiz_ref).negativechildren.empty()||!(*horiz_ref).positivechildren.empty()||!(*horiz_ref).neutralchildren.empty()||!(*horiz_ref).kids_rel_addresses.empty()||!(*horiz_ref).mergechild==NULL||!(*horiz_ref).negativeneutralvertices.empty())
					{
						cout << "Cleaning error!" << endl;
					}
				
				}

				/*
				for(set<simplex*>::iterator sim_ref = (*horiz_ref).triangulation.begin();sim_ref!=(*horiz_ref).triangulation.end();sim_ref++)
				{
					if((*sim_ref)->vertices.size()!=i+1)
					{
						cout << "Something is wrong." << endl;
					}
				}
				*/
				
				/*
				if(i==0)
				{
					cout << ((vertex*)horiz_ref)->get_coordinates() << endl;
				}
				*/

				/*
				char* string = "Checkpoint";


				if((*horiz_ref).parentconditions.size()==0)
				{
					zero++;
				}
				else if((*horiz_ref).parentconditions.size()==1)
				{
					one++;					
				}
				else
				{
					two++;
				}
				*/
				
			}
		}
		

		/*
		list<vec> table_entries;
		for(polyhedron* horiz_ref = statistic.rows[statistic.size()-1];horiz_ref!=statistic.row_ends[statistic.size()-1];horiz_ref=horiz_ref->next_poly)
		{
			toprow *current_toprow = (toprow*)(horiz_ref);
			for(list<set<vertex*>>::iterator tri_ref = current_toprow->triangulation.begin();tri_ref!=current_toprow->triangulation.end();tri_ref++)
			{
				for(set<vertex*>::iterator vert_ref = (*tri_ref).begin();vert_ref!=(*tri_ref).end();vert_ref++)
				{
					vec table_entry = vec();
					
					table_entry.ins(0,(*vert_ref)->get_coordinates()*current_toprow->condition.get(1,current_toprow->condition.size()-1)-current_toprow->condition.get(0,0));
					
					table_entry.ins(0,(*vert_ref)->get_coordinates());

					table_entries.push_back(table_entry);
				}
			}			
		}

		unique(table_entries.begin(),table_entries.end());

				
		
		for(list<vec>::iterator entry_ref = table_entries.begin();entry_ref!=table_entries.end();entry_ref++)
		{
			ofstream myfile;
			myfile.open("robust_data.txt", ios::out | ios::app);
			if (myfile.is_open())
			{
				for(int i = 0;i<(*entry_ref).size();i++)
				{
					myfile << (*entry_ref)[i] << ";";
				}
				myfile << endl;
			
				myfile.close();
			}
			else
			{
				cout << "File problem." << endl;
			}
		}
		*/
		

		return;
	}

	int statistic_rowsize(int row)
	{
		return statistic.row_size(row);
	}

	void add_condition(vec toadd)
	{
		vec null_vector = "";

		add_and_remove_condition(toadd, null_vector);
	}


	void remove_condition(vec toremove)
	{		
		vec null_vector = "";

		add_and_remove_condition(null_vector, toremove);	
	}

	void add_and_remove_condition(vec toadd, vec toremove)
	{

		// New condition arrived (new data are available). Here we will perform the Bayesian data update
		// step by splitting the location parameter space with respect to the new condition and computing
		// normalization integrals for each polyhedron in the location parameter space.
		
		// First we reset previous value of normalization factor and maximum value of the log likelihood.
		// Because there is a minus sign in the exponent of the likelihood, we really search for a minimum
		// and here we set min_ll to a high value.
		normalization_factor = 0;
		min_ll = numeric_limits<double>::max();

		// We translate the presence of a condition to add to a boolean. Also, if moving window version of
		// data update is used, we check for the presence of a condition to be removed from consideration.
		// To take care of addition and deletion of a condition in one method is computationally better than 
		// treating both cases separately.
		bool should_remove = (toremove.size() != 0);
		bool should_add    = (toadd.size() != 0);

		// We lower the number of conditions so far considered if we remove one.
		if(should_remove)
		{
			condition_order--;
		}

		// We raise the number of conditions so far considered if we add one.
		if(should_add)
		{
			condition_order++;
		}

		// We erase the support lists used in splitting/merging operations later on to keep track of the
		// split/merged polyhedrons.
		for_splitting.clear();
		for_merging.clear();

		// This is a somewhat stupid operation, where we fill the vector of lists by empty lists, so that
		// we can extend the lists contained in the vector later on.
		for(int i = 0;i<statistic.size();i++)
		{
			list<polyhedron*> empty_split;
			list<polyhedron*> empty_merge;

			for_splitting.push_back(empty_split);
			for_merging.push_back(empty_merge);
		}

		// We set`the iterator in the conditions list to a blind end() iterator
		list<condition*>::iterator toremove_ref = conditions.end();		

		// We search the list of conditions for existence of toremove and toadd conditions and check their
		// possible multiplicity. 
		for(list<condition*>::iterator ref = conditions.begin();ref!=conditions.end();ref++)
		{
			// If condition should be removed..
			if(should_remove)
			{
				// if it exists in the list
				if((*ref)->value == toremove)
				{
					// if it has multiplicity higher than 1
					if((*ref)->multiplicity>1)
					{
						// we just lower the multiplicity
						(*ref)->multiplicity--;

						// In this case the parameter space remains unchanged (we have to process no merging),
						// so we only alter the condition sums in all the cells and compute the integrals
						// over the cells with the subtracted condition
						alter_toprow_conditions(*ref,false);

						// By altering the condition sums in each individual unchanged cell, we have finished
						// all the tasks of this method related to merging and removing given condition. Therefore
						// we switch the should_remove switch to false.
						should_remove = false;
					}
					else
					{
						// In case the condition to be removed has a multiplicity of 1, we mark its position in
						// the vector of conditions by assigning its iterator to toremove_ref variable.
						toremove_ref = ref;							
					}					
				}
			}

			// If a condition should be added..
			if(should_add)
			{
				// We search the vector of conditions if a condition with the same value already exists.
				if((*ref)->value == toadd)
				{
					// If it does, there will be no further splitting necessary. We have to raise its multiplicity..
					(*ref)->multiplicity++;

					// Again as with the condition to be removed, if no splitting is performed, we only have to
					// perform the computations in the individual cells in the top row of Hasse diagram of the
					// complex of polyhedrons by changing the condition sums in individual cells and computing
					// integrals with changed condition sum. 
					alter_toprow_conditions(*ref,true);

					// We switch off any further operations on the complex by switching the should_add variable
					// to false.
					should_add = false;					
				}				
			}
		}	

		// Here we erase the removed condition from the conditions vector and assign a pointer to the
		// condition object of the removed condition, if there is such, else the pointer remains NULL.
		condition* condition_to_remove = NULL;
		if(should_remove)
		{
			if(toremove_ref!=conditions.end())
			{
				condition_to_remove = *toremove_ref;
				conditions.erase(toremove_ref);			
			}
		}

		// Here we create the condition object for a condition value to be added and we insert it in
		// the list of conditions in case new condition should be added, else the pointer is set to NULL.
		condition* condition_to_add = NULL;
		if(should_add)
		{
			condition_to_add = new condition(toadd);			
			conditions.push_back(condition_to_add);			
		}		
		
		//**********************************************************************************************
		//             Classification of points related to added and removed conditions
		//**********************************************************************************************
		// Here the preliminary and preparation part ends and we begin classifying individual vertices in
		// the bottom row of the representing Hasse diagram relative to the condition to be removed and the
		// one to be added. This classification proceeds further in a recursive manner. Each classified
		// polyhedron sends an information about its classification to its parent, when all the children of
		// given parents are classified, the parent can be itself classified and send information further to
		// its parent and so on.

		// We loop through all ther vertices
		for(polyhedron* horizontal_position = statistic.rows[0];horizontal_position!=statistic.get_end();horizontal_position=horizontal_position->next_poly)
		{		
			// Cast from general polyhedron to a vertex
			vertex* current_vertex = (vertex*)horizontal_position;
			
			// If a condition should be added or removed..
			if(should_add||should_remove)
			{
				// The coordinates are extended by a -1 representing there is no parameter multiplying the
				// regressor in the autoregressive model. The condition is passed to the method as a vector
				// (y_t,psi_{t-1}), where y_t is the value of regressor and psi_t is the vector of regressands.
				// Minus sign is needed, because the AR model equation reads y_t = theta*psi_{t-1}+e_t, which 
				// can be rewriten as (y_t, psi_{t-1})*(-1,theta)', where ' stands for transposition and * for
				// scalar product
				vec appended_coords = current_vertex->get_coordinates();
				appended_coords.ins(0,-1.0);				

				if(should_add)
				{
					// We compute the position of the vertex relative to the added condition
					double local_condition = appended_coords*toadd;// = toadd*(appended_coords.first/=appended_coords.second);

					// The method set_state classifies the SPLIT state of the vertex as positive, negative or
					// neutral
					current_vertex->set_state(local_condition,SPLIT);

					/// \TODO There should be a rounding error tolerance used here to insure we are not having too many points because of rounding error.
					// If the vertex lays on the hyperplane related to the condition cutting the location parameter
					// space in half, we say it is totally neutral. This way it will be different than the later
					// newly created vertices appearing on the cuts of line segments. In an environment, where
					// the data variables are continuous (they don't have positive probability mass at any point
					// in the data space) the occurence of a point on the cutting hyperplane has probability 0.
					// In real world application, where data are often discrete, we have to take such situation
					// into account.
					if(local_condition == 0)
					{
						// In certain scenarios this situation is rather rare. We might then want to know about
						// occurence of a point laying on the cutting hyperplane (Programmers note:Also such 
						// scenarios were not so well tested and computation errors may occur!)
						cout << "Condition to add: " << toadd << endl;
						cout << "Vertex coords: " << appended_coords << endl;

						// We classify the vertex totally neutral
						current_vertex->totally_neutral = true;

						// We raise its multiplicity and set current splitting condition as a parent condition
						// of the vertex, since if we later remove the original parent condition, the vertex 
						// has to have a parent condition its right to exist.
						current_vertex->raise_multiplicity();
						current_vertex->parentconditions.insert(condition_to_add);						
					}
					else
					{
						// If the vertex lays off the cutting hyperplane, we set its totally_neutral property
						// to false.
						current_vertex->totally_neutral = false;
					}
				}
			
				// Now we classify the vertex with respect to the MERGEing condition..
				if(should_remove)
				{					
					// We search the condition to be removed in the list of vertice's parent conditions
					set<condition*>::iterator cond_ref;					
					for(cond_ref = current_vertex->parentconditions.begin();cond_ref!=current_vertex->parentconditions.end();cond_ref++)
					{
						if(*cond_ref == condition_to_remove)
						{
							break;
						}
					}

					// If the list of parent conditions of the given vertex contain the condition that is being
					// removed, we erase it from the list, we set the vertice's MERGE state to neutral and we
					// insert the vertex into the set of polyhedrons that are supposed to be used for merging 
					// (themselves possibly being deleted). 

					// REMARK: One may think it would be easier to check the condition again computationally. 
					// Such design has been used before in the software, but due to rounding errors it was 
					// very unreliable. These rounding errors are avoided using current design.
					if(cond_ref!=current_vertex->parentconditions.end())
					{
						current_vertex->parentconditions.erase(cond_ref);
						current_vertex->set_state(0,MERGE);
						for_merging[0].push_back(current_vertex);
					}
					else
					{
						// If parent conditions of the vertex don't contain the condition to be removed, we
						// check in which halfspace it is located and set its MERGE state accordingly.
						double local_condition = toremove*appended_coords;
						current_vertex->set_state(local_condition,MERGE);
					}
				}				
			}

			// Once classified we proceed recursively by calling the send_state_message method
			send_state_message(current_vertex, condition_to_add, condition_to_remove, 0);		
			
		}

		// step_me(1);
		
		if(should_remove)
		{
			/*
			for(int i = 0;i<for_merging.size();i++)
			{
				for(list<polyhedron*>::iterator merge_ref = for_merging[i].begin();merge_ref!=for_merging[i].end();merge_ref++)
				{
					
					for(list<polyhedron*>::iterator par_ref = (*merge_ref)->children.begin();par_ref!=(*merge_ref)->children.end();par_ref++)
					{
						if(find((*par_ref)->parents.begin(),(*par_ref)->parents.end(),(*merge_ref))==(*par_ref)->parents.end())
						{
							cout << "Parent/child relations are not matched!" << endl;
						}
					}
					
					//cout << (*merge_ref)->get_state(MERGE) << ",";
				}

				// cout << endl;
			}
			*/		


			// Here we have finished the classification part and we have at hand two sets of polyhedrons used for
			// further operation on the location parameter space. The first operation will be merging of polyhedrons
			// with respect to the MERGE condition. For that purpose, we have a set of mergers in a list called
			// for_merging. After we are finished merging, we need to split the polyhedrons cut by the SPLIT
			// condition. These polyhedrons are gathered in the for_splitting list. As can be seen, the MERGE
			// operation is done from below, in the terms of the Hasse diagram and therefore we start to merge
			// from the very bottom row, from the vertices. On the other hand splitting is done from the top
			// and we therefore start with the segments that need to be split.

			// We start the MERGE operation here. Some of the vertices will disappear from the Hasse diagram. 
			// Because they are part of polyhedrons in the Hasse diagram above the segments, we need to remember
			// them in the separate set and get rid of them only after the process of merging all the polyhedrons
			// has been finished.
			cout << "Merging." << endl;
			set<vertex*> vertices_to_be_reduced;			
			
			// We loop through the vector list of polyhedrons for merging from the bottom row up. We keep track
			// of the number of the processed row.
			int k = 1;
			for(vector<list<polyhedron*>>::iterator vert_ref = for_merging.begin();vert_ref<for_merging.end();vert_ref++)
			{
				// Within a row we loop through all the polyhedrons that we use as mergers.
				for(list<polyhedron*>::iterator merge_ref = (*vert_ref).begin();merge_ref!=(*vert_ref).end();merge_ref++)
				{
					// ***************************************************
					//   First we treat the case of a multiple merger.
					// ***************************************************

					// If the multiplicity of the merger is greater than one, the merger will remain in the Hasse
					// diagram and its parents will remain split.
					if((*merge_ref)->get_multiplicity()>1)
					{
						// We remove the condition to be removed (the MERGE condition) from the list of merger's
						// parents.
						(*merge_ref)->parentconditions.erase(condition_to_remove);

						// If the merger is a vertex..
						if(k==1)
						{
							// ..we will later reduce its multiplicity (this is to prevent multiple reduction of
							// the same vertex) 
							vertices_to_be_reduced.insert((vertex*)(*merge_ref));
						}
						// If the merger is not a vertex..
						else
						{
							// lower the multiplicity of the merger
							(*merge_ref)->lower_multiplicity();
						}	

						// If the merger will not be split and it is not totally neutral with respect to SPLIT
						// condition (it doesn't lay in the hyperplane defined by the condition), we will not
						// need it for splitting purposes and we can therefore clean all the splitting related
						// properties, to be able to reuse them when new data arrive. A merger is never a toprow
						// so we do not need to integrate.
						if((*merge_ref)->get_state(SPLIT)!=0||(*merge_ref)->totally_neutral)
						{
							(*merge_ref)->positivechildren.clear();
							(*merge_ref)->negativechildren.clear();
							(*merge_ref)->neutralchildren.clear();						
							(*merge_ref)->totallyneutralgrandchildren.clear();						
							(*merge_ref)->positiveneutralvertices.clear();
							(*merge_ref)->negativeneutralvertices.clear();
							(*merge_ref)->totally_neutral = NULL;
							(*merge_ref)->kids_rel_addresses.clear();
						}
					}									
					// Else, if the multiplicity of the merger is equal to 1, we proceed with the merging part of
					// the algorithm.
					else
					{
						// A boolean that will be true, if after being merged, the new polyhedron should be split
						// in the next step of the algorithm.
						bool will_be_split = false;
						
						// The newly created polyhedron will be merged of a negative and positive part specified
						// by its merger.
						toprow* current_positive = (toprow*)(*merge_ref)->positiveparent;
						toprow* current_negative = (toprow*)(*merge_ref)->negativeparent;
						
						// An error check for situation that should not occur.
						if(current_positive->totally_neutral!=current_negative->totally_neutral)
						{
							throw new exception("Both polyhedrons must be totally neutral if they should be merged!");
						}						

						// *************************************************************************************
						//    Now we rewire the Hasse properties of the MERGE negative part of the merged
						//    polyhedron to the MERGE positive part - it will be used as the merged polyhedron
						// *************************************************************************************
						
						// Instead of establishing a new polyhedron and filling in all the necessary connections
						// and thus adding it into the Hasse diagram, we use the positive polyhedron with its
						// connections and we merge it with all the connections from the negative side so that
						// the positive polyhedron becomes the merged one.

						// We remove the MERGE condition from parent conditions.
						current_positive->parentconditions.erase(condition_to_remove);
						
						// We add the children from the negative part into the children list and remove from it the
						// merger.
						current_positive->children.insert(current_positive->children.end(),current_negative->children.begin(),current_negative->children.end());
						current_positive->children.remove(*merge_ref);

						// We reconnect the reciprocal addresses from children to parents.
						for(list<polyhedron*>::iterator child_ref = current_negative->children.begin();child_ref!=current_negative->children.end();child_ref++)
						{
							(*child_ref)->parents.remove(current_negative);
							(*child_ref)->parents.push_back(current_positive);													
						}

						// We loop through the parents of the negative polyhedron.
						for(list<polyhedron*>::iterator parent_ref = current_negative->parents.begin();parent_ref!=current_negative->parents.end();parent_ref++)
						{
							// Remove the negative polyhedron from its children
							(*parent_ref)->children.remove(current_negative);							

							// Remove it from the according list with respect to the negative polyhedron's
							// SPLIT state.
							switch(current_negative->get_state(SPLIT))
							{
							case -1:
								(*parent_ref)->negativechildren.remove(current_negative);
								break;
							case 0:
								(*parent_ref)->neutralchildren.remove(current_negative);								
								break;
							case 1:
								(*parent_ref)->positivechildren.remove(current_negative);
								break;
							}							
						}

						// We merge the vertices of the negative and positive part
						current_positive->vertices.insert(current_negative->vertices.begin(),current_negative->vertices.end());												

						// **************************************************************************
						//	 Now we treat the situation that one of the MERGEd polyhedrons is to be
						//   SPLIT. 
						// **************************************************************************

						if(current_negative->get_state(SPLIT)==0)
						{
							for_splitting[k].remove(current_negative);
						}						
						
						if(!current_positive->totally_neutral)
						{
							// If the positive polyhedron was not to be SPLIT and the negative polyhedron was..
							if(current_positive->get_state(SPLIT)!=0&&current_negative->get_state(SPLIT)==0)
							{
								//..we loop through the parents of the positive polyhedron..
								for(list<polyhedron*>::iterator parent_ref = current_positive->parents.begin();parent_ref!=current_positive->parents.end();parent_ref++)
								{
									//..and if the MERGE positive polyhedron is SPLIT positive, we remove it
									//from the list of SPLIT positive children..
									if(current_positive->get_state(SPLIT)==1)
									{
										(*parent_ref)->positivechildren.remove(current_positive);
									}
									//..or if the MERGE positive polyhedron is SPLIT negative, we remove it
									//from the list of SPLIT positive children..
									else
									{
										(*parent_ref)->negativechildren.remove(current_positive);
									}
									//..and we add it to the SPLIT neutral children, because the MERGE negative polyhedron
									//that is being MERGEd with it causes it to be SPLIT neutral (the hyperplane runs
									//through the merged polyhedron)
									(*parent_ref)->neutralchildren.push_back(current_positive);
								}

								// Because of the above mentioned reason, we set the SPLIT state of the MERGE positive
								// polyhedron to neutral
								current_positive->set_state(0,SPLIT);
								
								// and we add it to the list of polyhedrons to be SPLIT
								for_splitting[k].push_back(current_positive);							
							}
						
						
							// If the MERGEd polyhedron is to be split..
							if(current_positive->get_state(SPLIT)==0)
							{								
								// We need to fill the lists related to split with correct values, adding the SPLIT
								// positive, negative and neutral children to according list in the MERGE positive,
								// or future MERGEd polyhedron
								current_positive->negativechildren.insert(current_positive->negativechildren.end(),current_negative->negativechildren.begin(),current_negative->negativechildren.end());						
								current_positive->positivechildren.insert(current_positive->positivechildren.end(),current_negative->positivechildren.begin(),current_negative->positivechildren.end());												
								current_positive->neutralchildren.insert(current_positive->neutralchildren.end(),current_negative->neutralchildren.begin(),current_negative->neutralchildren.end());
							
								// and remove the merger, which will be later deleted from the lists of SPLIT classified
								// children.
								switch((*merge_ref)->get_state(SPLIT))
								{
								case -1:
									current_positive->negativechildren.remove(*merge_ref);
									break;
								case 0:
									current_positive->neutralchildren.remove(*merge_ref);
									break;
								case 1:
									current_positive->positivechildren.remove(*merge_ref);
									break;
								}							

								// We also have to merge the lists of totally neutral children laying in the SPLIT related
								// cutting hyperpalne and the lists of positive+neutral and negative+neutral vertices. 
								current_positive->totallyneutralgrandchildren.insert(current_negative->totallyneutralgrandchildren.begin(),current_negative->totallyneutralgrandchildren.end());
								// Because a vertex cannot be SPLIT, we don't need to remove the merger from the 
								// positive+neutral and negative+neutral lists
								current_positive->negativeneutralvertices.insert(current_negative->negativeneutralvertices.begin(),current_negative->negativeneutralvertices.end());							
								current_positive->positiveneutralvertices.insert(current_negative->positiveneutralvertices.begin(),current_negative->positiveneutralvertices.end());

								// And we set the will be split property to true
								will_be_split = true;
							}
						}
						
						// If the polyhedron will not be split (both parts are totally neutral or neither of them
						// was classified SPLIT neutral), we clear all the lists holding the SPLIT information for
						// them to be ready to reuse.
						if(!will_be_split)
						{							
							current_positive->positivechildren.clear();
							current_positive->negativechildren.clear();
							current_positive->neutralchildren.clear();							
							current_positive->totallyneutralgrandchildren.clear();								
							current_positive->positiveneutralvertices.clear();
							current_positive->negativeneutralvertices.clear();
							current_positive->totally_neutral = NULL;
							current_positive->kids_rel_addresses.clear();						
						}										
						
						// If both the merged polyhedrons are totally neutral, we have to rewire the addressing
						// in the grandparents from the negative to the positive (merged) polyhedron.
						if(current_positive->totally_neutral)
						{
							for(set<polyhedron*>::iterator grand_ref = current_negative->grandparents.begin();grand_ref!=current_negative->grandparents.end();grand_ref++)
							{
								(*grand_ref)->totallyneutralgrandchildren.erase(current_negative);
								(*grand_ref)->totallyneutralgrandchildren.insert(current_positive);
							}							
						}					

						// We clear the grandparents list for further reuse.
						current_positive->grandparents.clear();				
						
						
						// Delete the negative polyhedron from the Hasse diagram (rewire all the connections)
						statistic.delete_polyhedron(k,current_negative);

						// Delete the negative polyhedron object
						delete current_negative;

						// *********************************************
						//   Here we treat the deletion of the merger. 
						// *********************************************
						
						// We erase the vertices of the merger from all the respective lists.
						for(set<vertex*>::iterator vert_ref = (*merge_ref)->vertices.begin();vert_ref!=(*merge_ref)->vertices.end();vert_ref++)
						{
							if((*vert_ref)->get_multiplicity()==1)
							{
								current_positive->vertices.erase(*vert_ref);

								if(will_be_split)
								{
									current_positive->negativeneutralvertices.erase(*vert_ref);
									current_positive->positiveneutralvertices.erase(*vert_ref);								
								}
							}
						}
						
						// We remove the connection to the merger from the merger's children
						for(list<polyhedron*>::iterator child_ref = (*merge_ref)->children.begin();child_ref!=(*merge_ref)->children.end();child_ref++)
						{
							(*child_ref)->parents.remove(*merge_ref);
						}				

						// We remove the connection to the merger from the merger's grandchildren
						for(set<polyhedron*>::iterator grand_ch_ref = (*merge_ref)->totallyneutralgrandchildren.begin();grand_ch_ref!=(*merge_ref)->totallyneutralgrandchildren.end();grand_ch_ref++)
						{
							(*grand_ch_ref)->grandparents.erase(*merge_ref);
						}
						
						// We remove the connection to the merger from the merger's grandparents
						for(set<polyhedron*>::iterator grand_p_ref = (*merge_ref)->grandparents.begin();grand_p_ref!=(*merge_ref)->grandparents.end();grand_p_ref++)
						{
							(*grand_p_ref)->totallyneutralgrandchildren.erase(*merge_ref);
						}				

						// We remove the merger from the Hasse diagram
						statistic.delete_polyhedron(k-1,*merge_ref);						
						// And we delete the merger from the list of polyhedrons to be split
						for_splitting[k-1].remove(*merge_ref);						
						// If the merger is a vertex with multiplicity 1, we add it to the list of vertices to get
						// rid of at the end of the merging procedure.
						if(k==1)
						{							
							vertices_to_be_reduced.insert((vertex*)(*merge_ref));							
						}	

						// Triangulate the newly created polyhedron and compute its normalization integral if the
						// polyhedron is a toprow.
						normalization_factor += current_positive->triangulate(k==for_splitting.size()-1 && !will_be_split);
					}
				}			
			
				// And we go to the next row
				k++;

			}

			// At the end of the merging procedure, we delete all the merger's objects. These should now be already
			// disconnected from the Hasse diagram.
			for(int i = 1;i<for_merging.size();i++)
			{
				for(list<polyhedron*>::iterator merge_ref = for_merging[i].begin();merge_ref!=for_merging[i].end();merge_ref++)
				{
					delete (*merge_ref);
				}
			}
			
			// We also treat the vertices that we called to be reduced by either lowering their multiplicity or
			// deleting them in case the already have multiplicity 1.
			for(set<vertex*>::iterator vert_ref = vertices_to_be_reduced.begin();vert_ref!=vertices_to_be_reduced.end();vert_ref++)
			{
				if((*vert_ref)->get_multiplicity()>1)
				{
					(*vert_ref)->lower_multiplicity();
				}
				else
				{
					delete (*vert_ref);
				}
			}

			// Finally we delete the condition object
			delete condition_to_remove;
		}
		
		// This is a control check for errors in the merging procedure.
		/*
		vector<int> sizevector;
		for(int s = 0;s<statistic.size();s++)
		{
			sizevector.push_back(statistic.row_size(s));
			cout << statistic.row_size(s) << ", ";
		}
		cout << endl;
		*/

		// After the merging is finished or if there is no condition to be removed from the conditions list,
		// we split the location parameter space with respect to the condition to be added or SPLIT condition.
		if(should_add)
		{
			cout << "Splitting." << endl;

			// We reset the row counter
			int k = 1;			

			// Since the bottom row of the for_splitting list is empty - we can't split vertices, we start from
			// the second row from the bottom - the row containing segments
			vector<list<polyhedron*>>::iterator beginning_ref = ++for_splitting.begin();

			// We loop through the rows
			for(vector<list<polyhedron*>>::iterator vert_ref = beginning_ref;vert_ref<for_splitting.end();vert_ref++)
			{			

				// and we loop through the polyhedrons in each row
				for(list<polyhedron*>::reverse_iterator split_ref = vert_ref->rbegin();split_ref != vert_ref->rend();split_ref++)
				{
					// If we split a polyhedron by a SPLIT condition hyperplane, in the crossection of the two a
					// new polyhedron is created. It is totally neutral, because it lays in the condition hyperplane.
					polyhedron* new_totally_neutral_child;

					// For clear notation we rename the value referenced by split_ref iterator
					polyhedron* current_polyhedron = (*split_ref);
					
					// If the current polyhedron is a segment, the new totally neutral child will be a vertex and
					// we have to assign coordinates to it.
					if(vert_ref == beginning_ref)
					{
						// The coordinates will be computed from the equation of the straight line containing the
						// segment, obtained from the coordinates of the endpoints of the segment
						vec coordinates1 = ((vertex*)(*(current_polyhedron->children.begin())))->get_coordinates();						
						vec coordinates2 = ((vertex*)(*(++current_polyhedron->children.begin())))->get_coordinates();
						
						// For computation of the scalar product with the SPLIT condition, we need extended coordinates
						vec extended_coord2 = coordinates2;
						extended_coord2.ins(0,-1.0);						

						// We compute the parameter t an element of (0,1) describing where the segment is cut
						double t = (-toadd*extended_coord2)/(toadd(1,toadd.size()-1)*(coordinates1-coordinates2));						

						// And compute the coordinates as convex sum of the coordinates
						vec new_coordinates = (1-t)*coordinates2+t*coordinates1;						

						// cout << "c1:" << coordinates1 << endl << "c2:" << coordinates2 << endl << "nc:" << new_coordinates << endl;

						// We create a new vertex object
						vertex* neutral_vertex = new vertex(new_coordinates);						

						// and assign it to the new totally neutral child
						new_totally_neutral_child = neutral_vertex;
					}
					else
					{
						// If the split polyhedron isn't a segment, the totally neutral child will be a general
						// polyhedron. Because a toprow inherits from polyhedron, we make it a toprow for further
						// universality \TODO: is this really needed?
						toprow* neutral_toprow = new toprow();
						
						if(k==number_of_parameters)
						{
							// A toprow needs a valid condition
							neutral_toprow->condition_sum   = ((toprow*)current_polyhedron)->condition_sum; // tohle tu bylo driv: zeros(number_of_parameters+1);
							neutral_toprow->condition_order = ((toprow*)current_polyhedron)->condition_order+1;
						}

						// We assign it to the totally neutral child variable
						new_totally_neutral_child = neutral_toprow;
					}

					// We assign current SPLIT condition as a parent condition of the totally neutral child and also
					// the child inherits all the parent conditions of the split polyhedron
					new_totally_neutral_child->parentconditions.insert(current_polyhedron->parentconditions.begin(),current_polyhedron->parentconditions.end());
					new_totally_neutral_child->parentconditions.insert(condition_to_add);

					// The totally neutral child is a polyhedron belonging to my_emlig distribution
					new_totally_neutral_child->my_emlig = this;
					
					// We connect the totally neutral child to all totally neutral grandchildren of the polyhedron
					// being split. This is what we need the totally neutral grandchildren for. It complicates the
					// algorithm, because it is a second level dependence (opposed to the children <-> parents 
					// relations, but it is needed.)
					new_totally_neutral_child->children.insert(new_totally_neutral_child->children.end(),
														current_polyhedron->totallyneutralgrandchildren.begin(),
																current_polyhedron->totallyneutralgrandchildren.end());

					// We also create the reciprocal connection from the totally neutral grandchildren to the
					// new totally neutral child and add all the vertices of the totally neutral grandchildren
					// to the set of vertices of the new totally neutral child.
					for(set<polyhedron*>::iterator grand_ref = current_polyhedron->totallyneutralgrandchildren.begin(); grand_ref != current_polyhedron->totallyneutralgrandchildren.end();grand_ref++)
					{
						// parent connection
						(*grand_ref)->parents.push_back(new_totally_neutral_child);						

						// vertices
						new_totally_neutral_child->vertices.insert((*grand_ref)->vertices.begin(),(*grand_ref)->vertices.end());
					}
					
					// We create a condition sum for the split parts of the split polyhedron
					vec cur_pos_condition = ((toprow*)current_polyhedron)->condition_sum;
					vec cur_neg_condition = ((toprow*)current_polyhedron)->condition_sum;
					
					// If the split polyhedron is a toprow, we update the condition sum with the use of the SPLIT
					// condition. The classification of the intermediate row polyhedrons as toprows probably isn't
					// necessary and it could be changed for more elegance, but it is here for historical reasons.
					if(k == number_of_parameters)
					{
						cur_pos_condition = cur_pos_condition + toadd;
						cur_neg_condition = cur_neg_condition - toadd;
					}

					// We create the positive and negative parts of the split polyhedron completely from scratch,
					// using the condition sum constructed earlier. This is different from the merging part, where
					// we have reused one of the parts to create the merged entity. This way, we don't have to
					// clean up old information from the split parts and the operation will be more symetrical.
					toprow* positive_poly = new toprow(cur_pos_condition, ((toprow*)current_polyhedron)->condition_order+1);
					toprow* negative_poly = new toprow(cur_neg_condition, ((toprow*)current_polyhedron)->condition_order+1);

					// Set the new SPLIT positive and negative parts of the split polyhedrons as parents of the new
					// totally neutral child
					new_totally_neutral_child->parents.push_back(positive_poly);
					new_totally_neutral_child->parents.push_back(negative_poly);

					// and the new totally neutral child as a child of the SPLIT positive and negative parts
					// of the split polyhedron
					positive_poly->children.push_back(new_totally_neutral_child);
					negative_poly->children.push_back(new_totally_neutral_child);
					
					// The new polyhedrons belong to my_emlig
					positive_poly->my_emlig = this;
					negative_poly->my_emlig = this;

					// Parent conditions of the new polyhedrons are the same as parent conditions of the split polyhedron
					positive_poly->parentconditions.insert(current_polyhedron->parentconditions.begin(),current_polyhedron->parentconditions.end());
					negative_poly->parentconditions.insert(current_polyhedron->parentconditions.begin(),current_polyhedron->parentconditions.end());

					// We loop through the parents of the split polyhedron
					for(list<polyhedron*>::iterator parent_ref = current_polyhedron->parents.begin();parent_ref!=current_polyhedron->parents.end();parent_ref++)
					{
						// We set the new totally neutral child to be a totally neutral grandchild of the parent
						(*parent_ref)->totallyneutralgrandchildren.insert(new_totally_neutral_child);						

						// We remove the split polyhedron from both lists, where it should be present
						(*parent_ref)->neutralchildren.remove(current_polyhedron);
						(*parent_ref)->children.remove(current_polyhedron);

						// And instead set the newly created SPLIT negative and positive parts as children of 
						// the parent (maybe the parent will be split once we get to treating its row, but that
						// should be taken care of later) and we add it to the classified positive and negative
						// children list accordingly.
						(*parent_ref)->children.push_back(positive_poly);
						(*parent_ref)->children.push_back(negative_poly);
						(*parent_ref)->positivechildren.push_back(positive_poly);
						(*parent_ref)->negativechildren.push_back(negative_poly);
					}

					// Here we set the reciprocal connections to the ones set in the previous list. All the parents
					// of currently split polyhedron are added as parents of the SPLIT negative and positive parts.
					
					// for positive part..
					positive_poly->parents.insert(positive_poly->parents.end(),
												current_polyhedron->parents.begin(),
														current_polyhedron->parents.end());
					// for negative part..
					negative_poly->parents.insert(negative_poly->parents.end(),
												current_polyhedron->parents.begin(),
														current_polyhedron->parents.end());					

					// We loop through the positive children of the split polyhedron, remove it from their parents
					// lists and add the SPLIT positive part as their parent.
					for(list<polyhedron*>::iterator child_ref = current_polyhedron->positivechildren.begin();child_ref!=current_polyhedron->positivechildren.end();child_ref++)
					{
						(*child_ref)->parents.remove(current_polyhedron);
						(*child_ref)->parents.push_back(positive_poly);						
					}					

					// And again we set the reciprocal connections from the SPLIT positive part by adding
					// all the positive children of the split polyhedron to its list of children.
					positive_poly->children.insert(positive_poly->children.end(),
												current_polyhedron->positivechildren.begin(),
															current_polyhedron->positivechildren.end());

					// We loop through the negative children of the split polyhedron, remove it from their parents
					// lists and add the SPLIT negative part as their parent.
					for(list<polyhedron*>::iterator child_ref = current_polyhedron->negativechildren.begin();child_ref!=current_polyhedron->negativechildren.end();child_ref++)
					{
						(*child_ref)->parents.remove(current_polyhedron);
						(*child_ref)->parents.push_back(negative_poly);
					}

					// And again we set the reciprocal connections from the SPLIT negative part by adding
					// all the negative children of the split polyhedron to its list of children.
					negative_poly->children.insert(negative_poly->children.end(),
												current_polyhedron->negativechildren.begin(),
															current_polyhedron->negativechildren.end());

					// The vertices of the SPLIT positive part are the union of positive and neutral vertices of
					// the split polyhedron and vertices of the new neutral child
					positive_poly->vertices.insert(current_polyhedron->positiveneutralvertices.begin(),current_polyhedron->positiveneutralvertices.end());
					positive_poly->vertices.insert(new_totally_neutral_child->vertices.begin(),new_totally_neutral_child->vertices.end());

					// The vertices of the SPLIT negative part are the union of negative and neutral vertices of
					// the split polyhedron and vertices of the new neutral child
					negative_poly->vertices.insert(current_polyhedron->negativeneutralvertices.begin(),current_polyhedron->negativeneutralvertices.end());
					negative_poly->vertices.insert(new_totally_neutral_child->vertices.begin(),new_totally_neutral_child->vertices.end());
								
					// Triangulate the new totally neutral child without computing its normalization intergral 
					// (because the child is never a toprow polyhedron)
					new_totally_neutral_child->triangulate(false);

					// Triangulate the new SPLIT positive and negative parts of the split polyhedron and compute
					// their normalization integral if they are toprow polyhedrons
					normalization_factor += positive_poly->triangulate(k==for_splitting.size()-1);
					normalization_factor += negative_poly->triangulate(k==for_splitting.size()-1);
					
					// Insert all the newly created polyhedrons into the Hasse diagram
					statistic.append_polyhedron(k-1, new_totally_neutral_child);					
					statistic.insert_polyhedron(k, positive_poly, current_polyhedron);
					statistic.insert_polyhedron(k, negative_poly, current_polyhedron);					

					// and delete the split polyhedron from the diagram
					statistic.delete_polyhedron(k, current_polyhedron);

					// and also delete its object from the memory
					delete current_polyhedron;
				}

				// Goto a higher row of the for_splitting list
				k++;
			}
		}

		/*
		vector<int> sizevector;
		//sizevector.clear();
		for(int s = 0;s<statistic.size();s++)
		{
			sizevector.push_back(statistic.row_size(s));
			cout << statistic.row_size(s) << ", ";
		}
		
		cout << endl;
		*/

		// cout << "Normalization factor: " << normalization_factor << endl;	

		last_log_nc = log_nc;
		log_nc = log(normalization_factor); 

		/*
		for(polyhedron* topr_ref = statistic.rows[statistic.size()-1];topr_ref!=statistic.row_ends[statistic.size()-1]->next_poly;topr_ref=topr_ref->next_poly)
		{
			cout << ((toprow*)topr_ref)->condition << endl;
		}
		*/

		// step_me(101);
	}

	double _ll()
	{
		if(last_log_nc!=NULL)
		{
			return log_nc - last_log_nc;
		}
		else
		{
			throw new exception("You can not ask for log likelihood difference for a prior!");
		}
	}

	void set_correction_factors(int order)
		{
			for(int remaining_order = correction_factors.size();remaining_order<order;remaining_order++)
			{
				multiset<my_ivec> factor_templates;
				multiset<my_ivec> final_factors;				

				my_ivec orig_template = my_ivec();				

				for(int i = 1;i<number_of_parameters-remaining_order+1;i++)
				{					
					bool in_cycle = false;
					for(int j = 0;j<=remaining_order;j++)					{
						
						multiset<my_ivec>::iterator fac_ref = factor_templates.begin();

						do
						{
							my_ivec current_template;
							if(!in_cycle)
							{
								current_template = orig_template;
								in_cycle = true;
							}
							else
							{
								current_template = (*fac_ref);
								fac_ref++;
							}							
							
							current_template.ins(current_template.size(),i);

							// cout << "template:" << current_template << endl;
							
							if(current_template.size()==remaining_order+1)
							{
								final_factors.insert(current_template);
							}
							else
							{
								factor_templates.insert(current_template);
							}
						}
						while(fac_ref!=factor_templates.end());
					}
				}	

				correction_factors.push_back(final_factors);			

			}
		}

	pair<vec,simplex*> choose_simplex()
	{
		double rnumber = randu();

		// cout << "RND:" << rnumber << endl;

		// This could be more efficient (log n), but map::upper_bound() doesn't let me dereference returned iterator
		double  prob_sum     = 0;	
		toprow* sampled_toprow;				
		for(polyhedron* top_ref = statistic.rows[number_of_parameters];top_ref!=statistic.end_poly;top_ref=top_ref->next_poly)
		{
			// cout << "CDF:"<< (*top_ref).first << endl;

			toprow* current_toprow = ((toprow*)top_ref);

			prob_sum += current_toprow->probability;

			if(prob_sum >= rnumber*normalization_factor)
			{
				sampled_toprow = (toprow*)top_ref;
				break;
			}
			else
			{
				if(top_ref->next_poly==statistic.end_poly)
				{
					cout << "Error.";
				}
			}
		}				

		//// cout << "Toprow/Count: " << toprow_count << "/" << ordered_toprows.size() << endl;
		// cout << &sampled_toprow << ";";

		rnumber = randu();				

		set<simplex*>::iterator s_ref;
		prob_sum = 0;		
		for(s_ref = sampled_toprow->triangulation.begin();s_ref!=sampled_toprow->triangulation.end();s_ref++)
		{		
			prob_sum += (*s_ref)->probability;

			if(prob_sum/sampled_toprow->probability >= rnumber)
				break;
		}

		return pair<vec,simplex*>(sampled_toprow->condition_sum,*s_ref);	
	}

	pair<double,double> choose_sigma(simplex* sampled_simplex)
	{
		double sigma = 0;
		double pg_sum;					
		double rnumber = randu();			
					
		double sum_g = 0;
		for(int i = 0;i<sampled_simplex->positive_gamma_parameters.size();i++)
		{
			for(multimap<double,double>::iterator g_ref = sampled_simplex->positive_gamma_parameters[i].begin();g_ref != sampled_simplex->positive_gamma_parameters[i].end();g_ref++)
			{
				sum_g += (*g_ref).first/sampled_simplex->positive_gamma_sum;

							
				if(sum_g>rnumber)
				{					
					// tady je mozna chyba ve vaskove kodu
					GamRNG.setup(condition_order-number_of_parameters-1+i,(*g_ref).second);
																
					sigma = 1/GamRNG();
					// cout << "Sigma mean:   " << (*g_ref).second/(conditions.size()-number_of_parameters-1) << endl;								

					break;
				}							
			}

			if(sigma!=0)
			{
				break;
			}
		}	

		pg_sum = 0;
		int i = 0;
		for(vector<multimap<double,double>>::iterator v_ref = sampled_simplex->positive_gamma_parameters.begin();v_ref!=sampled_simplex->positive_gamma_parameters.end();v_ref++)
		{
			for(multimap<double,double>::iterator pg_ref = (*v_ref).begin();pg_ref!=(*v_ref).end();pg_ref++)
			{
				pg_sum += exp(-(*pg_ref).second/sigma)*pow((*pg_ref).second/sigma,condition_order-number_of_parameters-1+i)/sigma/fact(condition_order-number_of_parameters-2+i)*(*pg_ref).first;  // pg_sum += exp((sampled_simplex->min_beta-(*pg_ref).second)/sigma)*pow((*pg_ref).second/sigma,(int)conditions.size())*(*pg_ref).second/fact(conditions.size())*(*pg_ref).first;
			}

			i++;
		}	

		return pair<double,double>(sampled_simplex->positive_gamma_sum/pg_sum,sigma);
	}

	pair<vec,mat> sample(int n, bool rejection)
	{
		vec probabilities;
		mat samples;		
		
		while(samples.cols()<n)
		{
			pair<vec,simplex*> condition_and_simplex = choose_simplex();

			pair<double,double> probability_and_sigma = choose_sigma(condition_and_simplex.second);

			int dimension = condition_and_simplex.second->vertices.size();

			mat jacobian(dimension,dimension-1);
			vec gradient = condition_and_simplex.first;
				
			int row_count = 0;

			for(set<vertex*>::iterator vert_ref = condition_and_simplex.second->vertices.begin();vert_ref!=condition_and_simplex.second->vertices.end();vert_ref++)
			{
				jacobian.set_row(row_count,(*vert_ref)->get_coordinates());
				row_count++;
			}		
			
			ExpRNG.setup(1);

			vec sample_coords;
			double sample_sum = 0;
			for(int j = 0;j<dimension;j++)
			{
				double rnumber = ExpRNG();

				sample_sum += rnumber;

				sample_coords.ins(0,rnumber);				
			}

			sample_coords /= sample_sum;

			sample_coords = jacobian.T()*sample_coords;

			vec extended_coords = sample_coords;
			extended_coords.ins(0,-1.0);

			double exponent = extended_coords*condition_and_simplex.first;
			double sample_prob = 1*condition_and_simplex.second->volume/condition_and_simplex.second->probability/pow(2*probability_and_sigma.second,condition_order)*exp(-exponent/probability_and_sigma.second);//*probability_and_sigma.first;			

			sample_coords.ins(sample_coords.size(),probability_and_sigma.second);
			
			samples.ins_col(0,sample_coords);
			probabilities.ins(0,sample_prob);

			

			//cout << "C:" << sample_coords << "   p:" << sample_prob << endl;
			//pause(1);
		}

		if(rejection)
		{
			double max_prob = max(probabilities);

			set<int> indices;
			for(int i = 0;i<n;i++)
			{
				double randn = randu();

				if(probabilities.get(i)<randn*max_prob)
				{
					indices.insert(i);
				}
			}

			for(set<int>::reverse_iterator ind_ref = indices.rbegin();ind_ref!=indices.rend();ind_ref++)
			{
				samples.del_col(*ind_ref);				
			}

			return pair<vec,mat>(ones(samples.cols()),samples);
		}
		else
		{	
			return pair<vec,mat>(probabilities,samples);
		}
	}

	int logfact(int factor)
	{
		if(factor>1)
		{
			return log((double)factor)+logfact(factor-1);
		}
		else
		{
			return 0;
		}
	}
protected:

	/// A method for creating plain default statistic representing only the range of the parameter space.
    void create_statistic(int number_of_parameters, double alpha_deviation, double sigma_deviation)
	{
		/*
		for(int i = 0;i<number_of_parameters;i++)
		{
			vec condition_vec = zeros(number_of_parameters+1);
			condition_vec[i+1]  = 1;

			condition* new_condition = new condition(condition_vec);
			
			conditions.push_back(new_condition);
		}
		*/

		// An empty vector of coordinates.
		vec origin_coord;	

		// We create an origin - this point will have all the coordinates zero, but now it has an empty vector of coords.
		vertex *origin = new vertex(origin_coord);

		origin->my_emlig = this;
		
		/*
		// As a statistic, we have to create a vector of vectors of polyhedron pointers. It will then represent the Hasse
		// diagram. First we create a vector of polyhedrons..
		list<polyhedron*> origin_vec;

		// ..we fill it with the origin..
		origin_vec.push_back(origin);

		// ..and we fill the statistic with the created vector.
		statistic.push_back(origin_vec);
		*/

		statistic = *(new c_statistic());		
		
		statistic.append_polyhedron(0, origin);

		// Now we have a statistic for a zero dimensional space. Regarding to how many dimensional space we need to 
		// describe, we have to widen the descriptional default statistic. We use an iterative procedure as follows:
		for(int i=0;i<number_of_parameters;i++)
		{
			// We first will create two new vertices. These will be the borders of the parameter space in the dimension
			// of newly added parameter. Therefore they will have all coordinates except the last one zero. We get the 
			// right amount of zero cooridnates by reading them from the origin
			vec origin_coord = origin->get_coordinates();	

			

			// And we incorporate the nonzero coordinates into the new cooordinate vectors
			vec origin_coord1 = concat(origin_coord,-max_range); 
			vec origin_coord2 = concat(origin_coord,max_range);				 
					

			// Now we create the points
			vertex* new_point1 = new vertex(origin_coord1);
			vertex* new_point2 = new vertex(origin_coord2);

			new_point1->my_emlig = this;
			new_point2->my_emlig = this;
			
			//*********************************************************************************************************
			// The algorithm for recursive build of a new Hasse diagram representing the space structure from the old
			// diagram works so that you create two copies of the old Hasse diagram, you shift them up one level (points
			// will be segments, segments will be areas etc.) and you connect each one of the original copied polyhedrons
			// with its offspring by a parent-child relation. Also each of the segments in the first (second) copy is
			// connected to the first (second) newly created vertex by a parent-child relation.
			//*********************************************************************************************************


			/*
			// Create the vectors of vectors of pointers to polyhedrons to hold the copies of the old Hasse diagram
			vector<vector<polyhedron*>> new_statistic1;
			vector<vector<polyhedron*>> new_statistic2;
			*/

			c_statistic* new_statistic1 = new c_statistic();
			c_statistic* new_statistic2 = new c_statistic();

			
			// Copy the statistic by rows			
			for(int j=0;j<statistic.size();j++)
			{
				

				// an element counter
				int element_number = 0;

				/*
				vector<polyhedron*> supportnew_1;
				vector<polyhedron*> supportnew_2;

				new_statistic1.push_back(supportnew_1);
				new_statistic2.push_back(supportnew_2);
				*/

				// for each polyhedron in the given row
				for(polyhedron* horiz_ref = statistic.rows[j];horiz_ref!=statistic.get_end();horiz_ref=horiz_ref->next_poly)
				{	
					// Append an extra zero coordinate to each of the vertices for the new dimension
					// If vert_ref is at the first index => we loop through vertices
					if(j == 0)
					{
						// cast the polyhedron pointer to a vertex pointer and push a zero to its vector of coordinates
						((vertex*) horiz_ref)->push_coordinate(0);
					}
					/*
					else
					{
						((toprow*) (*horiz_ref))->condition.ins(0,0);
					}*/

					// if it has parents
					if(!horiz_ref->parents.empty())
					{
						// save the relative address of this child in a vector kids_rel_addresses of all its parents.
						// This information will later be used for copying the whole Hasse diagram with each of the
						// relations contained within.
						for(list<polyhedron*>::iterator parent_ref = horiz_ref->parents.begin();parent_ref != horiz_ref->parents.end();parent_ref++)
						{
							(*parent_ref)->kids_rel_addresses.push_back(element_number);							
						}						
					}

					// **************************************************************************************************
					// Here we begin creating a new polyhedron, which will be a copy of the old one. Each such polyhedron
					// will be created as a toprow, but this information will be later forgotten and only the polyhedrons
					// in the top row of the Hasse diagram will be considered toprow for later use.
					// **************************************************************************************************

					// First we create vectors specifying a toprow condition. In the case of a preconstructed statistic
					// this condition will be a vector of zeros. There are two vectors, because we need two copies of 
					// the original Hasse diagram.
					vec vec1;
					vec vec2;
					if(!horiz_ref->kids_rel_addresses.empty())
					{					
						vec1 = ((toprow*)horiz_ref)->condition_sum;
						vec1.ins(vec1.size(),-alpha_deviation);

						vec2 = ((toprow*)horiz_ref)->condition_sum;
						vec2.ins(vec2.size(),alpha_deviation);
					}
					else
					{						
						vec1.ins(0,-alpha_deviation);
						vec2.ins(0,alpha_deviation);

						vec1.ins(0,-sigma_deviation);
						vec2.ins(0,-sigma_deviation);
					}
					
					// cout << vec1 << endl;
					// cout << vec2 << endl;


					// We create a new toprow with the previously specified condition.
					toprow* current_copy1 = new toprow(vec1, this->condition_order);
					toprow* current_copy2 = new toprow(vec2, this->condition_order);

					current_copy1->my_emlig = this;
					current_copy2->my_emlig = this;

					// The vertices of the copies will be inherited, because there will be a parent/child relation
					// between each polyhedron and its offspring (comming from the copy) and a parent has all the
					// vertices of its child plus more.
					for(set<vertex*>::iterator vertex_ref = horiz_ref->vertices.begin();vertex_ref!=horiz_ref->vertices.end();vertex_ref++)
					{
						current_copy1->vertices.insert(*vertex_ref);
						current_copy2->vertices.insert(*vertex_ref);						
					}
					
					// The only new vertex of the offspring should be the newly created point.
					current_copy1->vertices.insert(new_point1);
					current_copy2->vertices.insert(new_point2);					
					
					// This method guarantees that each polyhedron is already triangulated, therefore its triangulation
					// is only one set of vertices and it is the set of all its vertices.
					simplex* t_simplex1 = new simplex(current_copy1->vertices);
					simplex* t_simplex2 = new simplex(current_copy2->vertices);					
					
					current_copy1->triangulation.insert(t_simplex1);
					current_copy2->triangulation.insert(t_simplex2);					
					
					// Now we have copied the polyhedron and we have to copy all of its relations. Because we are copying
					// in the Hasse diagram from bottom up, we always have to copy the parent/child relations to all the
					// kids and when we do that and know the child, in the child we will remember the parent we came from.
					// This way all the parents/children relations are saved in both the parent and the child.
					if(!horiz_ref->kids_rel_addresses.empty())
					{
						for(list<int>::iterator kid_ref = horiz_ref->kids_rel_addresses.begin();kid_ref!=horiz_ref->kids_rel_addresses.end();kid_ref++)
						{	
							polyhedron* new_kid1 = new_statistic1->rows[j-1];
							polyhedron* new_kid2 = new_statistic2->rows[j-1];

							// THIS IS NOT EFFECTIVE: It could be improved by having the list indexed for new_statistic, but
							// not indexed for statistic. Hopefully this will not cause a big slowdown - happens only offline.
							if(*kid_ref)
							{
								for(int k = 1;k<=(*kid_ref);k++)
								{
									new_kid1=new_kid1->next_poly;
									new_kid2=new_kid2->next_poly;
								}
							}
							
							// find the child and save the relation to the parent
							current_copy1->children.push_back(new_kid1);
							current_copy2->children.push_back(new_kid2);

							// in the child save the parents' address
							new_kid1->parents.push_back(current_copy1);
							new_kid2->parents.push_back(current_copy2);
						}						

						// Here we clear the parents kids_rel_addresses vector for later use (when we need to widen the
						// Hasse diagram again)
						horiz_ref->kids_rel_addresses.clear();
					}
					// If there were no children previously, we are copying a polyhedron that has been a vertex before.
					// In this case it is a segment now and it will have a relation to its mother (copywise) and to the
					// newly created point. Here we create the connection to the new point, again from both sides.
					else
					{
						// Add the address of the new point in the former vertex
						current_copy1->children.push_back(new_point1);
						current_copy2->children.push_back(new_point2);

						// Add the address of the former vertex in the new point
						new_point1->parents.push_back(current_copy1);
						new_point2->parents.push_back(current_copy2);
					}

					// Save the mother in its offspring
					current_copy1->children.push_back(horiz_ref);
					current_copy2->children.push_back(horiz_ref);

					// Save the offspring in its mother
					horiz_ref->parents.push_back(current_copy1);
					horiz_ref->parents.push_back(current_copy2);	
								
					
					// Add the copies into the relevant statistic. The statistic will later be appended to the previous
					// Hasse diagram
					new_statistic1->append_polyhedron(j,current_copy1);
					new_statistic2->append_polyhedron(j,current_copy2);
					
					// Raise the count in the vector of polyhedrons
					element_number++;			
					
				}
				
			}

			/*
			statistic.begin()->push_back(new_point1);
			statistic.begin()->push_back(new_point2);
			*/

			statistic.append_polyhedron(0, new_point1);
			statistic.append_polyhedron(0, new_point2);

			// Merge the new statistics into the old one. This will either be the final statistic or we will
			// reenter the widening loop. 
			for(int j=0;j<new_statistic1->size();j++)
			{
				/*
				if(j+1==statistic.size())
				{
					list<polyhedron*> support;
					statistic.push_back(support);
				}
				
				(statistic.begin()+j+1)->insert((statistic.begin()+j+1)->end(),new_statistic1[j].begin(),new_statistic1[j].end());
				(statistic.begin()+j+1)->insert((statistic.begin()+j+1)->end(),new_statistic2[j].begin(),new_statistic2[j].end());
				*/
				statistic.append_polyhedron(j+1,new_statistic1->rows[j],new_statistic1->row_ends[j]);
				statistic.append_polyhedron(j+1,new_statistic2->rows[j],new_statistic2->row_ends[j]);
			}			
		}

		/*
		vector<list<toprow*>> toprow_statistic;
		int line_count = 0;

		for(vector<list<polyhedron*>>::iterator polyhedron_ref = ++statistic.begin(); polyhedron_ref!=statistic.end();polyhedron_ref++)
		{
			list<toprow*> support_list;
			toprow_statistic.push_back(support_list);						

			for(list<polyhedron*>::iterator polyhedron_ref2 = polyhedron_ref->begin(); polyhedron_ref2 != polyhedron_ref->end(); polyhedron_ref2++)
			{
				toprow* support_top = (toprow*)(*polyhedron_ref2);

				toprow_statistic[line_count].push_back(support_top);
			}

			line_count++;
		}*/

		/*
		vector<int> sizevector;
		for(int s = 0;s<statistic.size();s++)
		{
			sizevector.push_back(statistic.row_size(s));
		}
		*/
		
	}
	
};



//! Robust Bayesian AR model for Multicriteria-Laplace-Inverse-Gamma density
class RARX //: public BM
{
private:
	bool has_constant;

	int window_size;	

	list<vec> conditions;

public:
	emlig* posterior;

	RARX(int number_of_parameters, const int window_size, bool has_constant, double alpha_deviation, double sigma_deviation, int nu)//:BM()
	{
		this->has_constant = has_constant;
		
		posterior = new emlig(number_of_parameters,alpha_deviation,sigma_deviation,nu);

		this->window_size = window_size;		
	};
	
	RARX(int number_of_parameters, const int window_size, bool has_constant)//:BM()
	{
		this->has_constant = has_constant;
		
		posterior = new emlig(number_of_parameters,1.0,1.0,number_of_parameters+3);

		this->window_size = window_size;		
	};

	void bayes(itpp::vec yt)
	{
		if(has_constant)
		{
			int c_size = yt.size();
			
			yt.ins(c_size,1.0);
		}		

		if(yt.size() == posterior->number_of_parameters+1)
		{
			conditions.push_back(yt);		
		}
		else
		{
			throw new exception("Wrong condition size for bayesian data update!");
		}

		//posterior->step_me(0);
		
		cout << "*************************************" << endl << "Current condition:" << yt << endl << "*************************************" << endl;

		/// \TODO tohle je spatne, tady musi byt jiny vypocet poctu podminek, kdyby nejaka byla multiplicitni, tak tohle bude spatne
		if(conditions.size()>window_size && window_size!=0)
		{
			posterior->add_and_remove_condition(yt,conditions.front());
			conditions.pop_front();

			//posterior->step_me(1);
		}
		else
		{
			posterior->add_condition(yt);
		}

		
				
	}

};



#endif //TRAGE_H
