#ifndef UI_H
#define UI_H

#include <sstream>
#include <iostream>
#include <stdio.h>
#include <string>
#include <typeinfo>
#include <map>
#include <utility>
#include <vector>
#include <xercesc/dom/DOM.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/util/XMLString.hpp>
#include <iostream>
#include <xercesc/framework/LocalFileFormatTarget.hpp>
#include <xercesc/framework/LocalFileInputSource.hpp>
#include <xercesc/dom/DOMWriter.hpp>
#include <xercesc/parsers/XercesDOMParser.hpp>

#ifdef XERCES_CPP_NAMESPACE_USE
XERCES_CPP_NAMESPACE_USE
#endif

using std::string;
using namespace std;

/*!
@brief Class initializing Xerces library

This class is used to initialize the Xerces library. Namely
to call static method XMLPlatformUtils::Initialize(). It should
be called just once, at previously to any other usage of the Xerces
library. This behaviour is assured by a quite complicated design
based on the use of an inner singleton class named 'XercesConnector'.
The point is to be the first initialized class even within the set of all
(global) static variables. For better understanding, find instances of
this class and look at their context.
*/
class AssertXercesIsAlive
{
private:
	//! Singleton class implementing the key property of the 'AssertXercesIsAlive' class 
	class XercesConnector
	{
	private:
		//! Default constructor, which is intentionally declared as private and called only from static
		//! method 'StayAlive()'. Therfore, it is called only once in the run of a application!
		XercesConnector()
		{
			// initialize the XML library
			XMLPlatformUtils::Initialize();
		}

		~XercesConnector()
		{
			// terminate the XML library
			XMLPlatformUtils::Terminate();
		}

	public:
		//! This method just touches the only instance of the XercesConnector class
		//! which is declared as a static local variable.
		//!
		//! This way we know that Xerces was initialized properly when returning from StayAlive().
		//! And what is more, the local nature of the inner variable prevents us before multiple 
		//! initializations.
		static void StayAlive()
		{
			static XercesConnector xc;		
		};
	};


public:
	//!default constructor
	AssertXercesIsAlive()
	{
		XercesConnector::StayAlive();
	}
};

/*!
@brief Abstract class declaring general properties of a frame for data binding 
*/
class BindingFrame
{
private:
	//! This private attribute has to be declared as the first attribute in the class.
	//! Only this way we can be sure it's constructor is called as the first
	//! and thus Xerces is initialized properly and right on time
	AssertXercesIsAlive dummy;

protected:
	//!default constructor
	BindingFrame();

	//! function which transcodes Xerces' XMLCh-based strings into C++ strings
	string XMLCh2str( const XMLCh* const  XMLCh_str );

	string removeSpaces(const string &str) 
	{
		std::string temp;
		for (unsigned int i = 0; i < str.length(); i++)
			if (str[i] == ' ') temp += '_';
			else temp += str[i];
		return temp;
	}

public:	
	//! This method parse DOMElement, finds proper DOMNode and fills binded data accordingly
	virtual void AssemblyComponentsFromXML( DOMElement &element ) = 0;

	//! A method for releasing memory allocated within the 'AssemblyComponentsFromXML()' method
	virtual void ReleaseMemory() {}

	//! This method reads binded data, fill them into a new DOMNode, which then 
	//! appends to the passed DOMElement
	virtual bool DisassemblyComponentsToXML( DOMElement &element ) = 0;		
};

/*!
@brief Abstract class declaring general properties of a frame for data binding 
*/
class Attribute 
{		
private:
	//! This private attribute has to be declared as the first attribute in the class.
	//! Only this way we can be sure it's constructor is called as the first
	//! and thus Xerces is initialized properly and right on time
	AssertXercesIsAlive dummy;

	//! an attribute name
	const XMLCh* const transcodedAttributeName;

public:
	//! Default constructor fixing a name of the related attribute
	Attribute( string attributeName );

	~Attribute();

	//! This method parse DOMElement, finds proper attribute and returns its value
	string& Get( DOMElement &element ) const;

	//! This method adds the passed string as an new attribute into the passed DOMElement
	void Set( DOMElement &element, const string &str ) const;	

	//! Static member, an instance related to an attribute named 'help'
	static const Attribute help;

	//! Static member, a constant instance related to an attribute named 'type'
	static const Attribute type;

	//! Static member, a constant instance related to the attribute named 'value'
	static const Attribute value;
};



/*!
@brief UserInfo is an abstract is for internal purposes only. Use CompoundUserInfo<T> or ValuedUserInfo<T> instead.
The raison d'etre of this class is to allow pointers to its templated descendants. 

Also, the main functions of the whole UserInfo library are included within this class, see
static methods 'Assembly' and 'Disassembly'.
*/
class UserInfo : protected BindingFrame
{
private:
	//! just a typedef shortuct for a constant pointer to UserInfo
	typedef UserInfo* const pUserInfo;

	//! static class encalupsating map of names to related UserInfos
	//! 
	//! The key property of this class is that it initilaized the internal map immediately
	//! when it is used for a first time.
	class StringToUIMap
	{
	private:
		//! Type definition of mapping which transforms type names to the related user infors
		typedef map< const string, pUserInfo > MappedString2UI;

		//! immediately initialized instance of type MappedString2UI
		static MappedString2UI& privateMap();

	public:
		//! add a pair key-userinfo into the internal map
		static void Add( string key, pUserInfo pInstance );

		//! search for an userinfo related to the passed key within the internal map
		static pUserInfo Retrieve( string key );
	};
				
	//! internal method assembling a typeless instance from components obtained by the 'AssemblyComponentsFromXML()' method
	virtual void* AssemblyTypelessInstance() = 0;
	
	//! internal method disassembling a typeless instance to components which are processed by the 'DisassemblyComponentsToXML()' method
	virtual bool DisassemblyTypelessInstance(void* pInstance) = 0;

	//! an user-friendly type name
	const string userFriendlyTypeName;

	//! a type name obtained by RTTI
	const string typeNameByRTTI;

protected:

	//! default constructor 
	UserInfo( const string& userFriendlyTypeName, const string& typeNameByRTTI )
		: userFriendlyTypeName ( removeSpaces( userFriendlyTypeName ) ), 
		  typeNameByRTTI( typeNameByRTTI )
	{	
		StringToUIMap::Add( userFriendlyTypeName, this );

		if( userFriendlyTypeName != typeNameByRTTI )
			// we have a common map for both groups of names, 
			// therefore, it is no use to add the same pair again
			StringToUIMap::Add( typeNameByRTTI, this );
	}

public:	
	//! This methods tries to assembly a new instance of type T (or some of its descendant types)
	//! according to a data stored in a DOMNode named tagName within a child nodes of the passed element.
	//! If an error occurs, it returns a NULL pointer.
	template<class T>
	static T* Assembly( DOMElement &element, const string tagName )
	{	
		XMLCh* transcodedTagName = XMLString::transcode( tagName.c_str() );		
		XMLString::upperCase( transcodedTagName );

		DOMNodeList* const nodeList = element.getElementsByTagName( transcodedTagName );
		XMLString::release( (XMLCh**)&transcodedTagName );
		if( !nodeList || nodeList->getLength() == 0 )
		{
			cerr << "Warning: there is not any tag named """ << tagName << """ in the passed DOM element of a XML docmument!";
			return NULL;
		}

		if( nodeList->getLength() > 1 )
		{
			cerr << "Warning: there is to many elements named """ << tagName << """ in the passed DOM element of a XML docmument. But the tag name has to be unique!";
			return NULL;
		}

		// this time we hold an element with the same name as the tagName is
		DOMElement* pTheOnlyElement = (DOMElement*) nodeList->item(0);

		// we get a velue stored in the "type" attribute 
		string userFriendlyTypeName = Attribute::type.Get( *pTheOnlyElement );
	
		// and finally we find a UserInfo related to this type
		pUserInfo pRelatedUI = StringToUIMap::Retrieve( userFriendlyTypeName );
		if( !pRelatedUI )
		{
			cerr << "Warning: there is not any UserInfo related to type named """ << userFriendlyTypeName << """, instance assembling terminated!";
			return NULL;
		}

		// prepare all components necessary for an instance assembling
		pRelatedUI->AssemblyComponentsFromXML( *pTheOnlyElement );		

		// instance assembling 
		void* pTypelessInstance = pRelatedUI->AssemblyTypelessInstance();

		// cleaning up
		pRelatedUI->ReleaseMemory();

		if( pTypelessInstance == NULL )
		{
			cerr << "Warning: there was some error when parsing a XML document, instance assembling terminated!";
			return NULL;
		}

		T* pInstance = NULL;
		try
		{
			// a "do it yourself" type check:)
			pInstance = (T*) pTypelessInstance;
			string resultingTypeNameByRTTI = typeid( *pInstance ).name();
			if( resultingTypeNameByRTTI != pRelatedUI->typeNameByRTTI )
				pInstance = NULL;
		}
		catch(...)
		{
			pInstance = NULL;
		}		
		if( pInstance == NULL )
			cerr << "Warning: UserInfo related to type """ << userFriendlyTypeName << """ have just returned instance of a different type! Instance assembling terminated!";

		return pInstance;
	}	

	//! This methods tries to disassembly an instance of type T (or some of its descendant types)
	//! and build DOM tree accordingly. Then, it creates a new DOMNode named according tagName
	//! and connecti it to the passed DOMElement as a new child node (with a help attribute filled).
	template<class T>
	static bool Disassembly( T& instance, DOMElement &element, const string tagName, const string help)
	{	
		pUserInfo pRelatedUI = StringToUIMap::Retrieve( typeid(instance).name() );
		if( !pRelatedUI )
			return false;

		// add a new element named according the passed tagName
		XMLCh* transcodedTagName = XMLString::transcode( tagName.c_str() );		
		XMLString::upperCase( transcodedTagName );
		DOMDocument* pDoc = element.getOwnerDocument();
		DOMElement* pCreatedElement = pDoc->createElement( transcodedTagName );		
		element.appendChild( pCreatedElement );
		XMLString::release( (XMLCh**)&transcodedTagName );
			
		// add attributes "type" and "help"
		Attribute::type.Set( *pCreatedElement, pRelatedUI->userFriendlyTypeName );		
		Attribute::help.Set( *pCreatedElement, help );
		
		// disassembly instance itself into its components
		bool result =  pRelatedUI->DisassemblyTypelessInstance( (void*) &instance );
		if( result )
			// disassembly all components of an instance 	
			result = pRelatedUI->DisassemblyComponentsToXML( *pCreatedElement );		
		return result;
	}	

	//! This methods tries to disassembly an instance of type T (or some of its descendant types)
	//! and build DOM tree accordingly. Then, it creates a new DOMNode named according tagName
	//! and connecti it to the passed DOMElement as a new child node.
	template<class T>
	static bool Disassembly( T &instance, DOMElement &element, const string tagName )
	{
		return Disassembly( instance, element, tagName, "" );
	}
};

/*!
@brief TypeUserInfo is still an abstract class, but contrary to the UserInfo class it is already 
templated. It serves as a bridge to non-abstract classes CompoundUserInfo<T> or ValuedUserInfo<T>.

There are two important features of this class. The first is a primitive mechanism bounding 
typeless methods DisassemblyTypelessInstance, resp. AssemblyTypelessInstance, to their typed 
virtual versions DisassemblyInstance, resp. AssemblyInstance. And the other is the only public 
attribute of this class, called 'instance', which is to be filled by the only instance of this 
class. Indeed, it is not  possible to create any other instance outside this class (or its 
descendant classes), as the constructor is intentionally protected.
*/
template<typename T> class TypedUserInfo : public UserInfo
{
private:

	bool DisassemblyTypelessInstance(void* pInstance)
	{
		try
		{
			return DisassemblyInstance( *(T*) pInstance );
		}
		catch (...)
		{
			return false;
		}
	}

	void* AssemblyTypelessInstance()
	{
		return (void*) AssemblyInstance( );
	}

	//! abstract method assembling a typed instance from components obtained by the 'AssemblyComponentsFromXML()' method
	virtual T* AssemblyInstance() = 0;

	//! abstract method disassembling a typed instance to components which are processed by the 'DisassemblyComponentsToXML()' method
	virtual bool DisassemblyInstance(T& instance) = 0;

protected:

	//! default constructor, which is intentionally declared as protected
	TypedUserInfo<T>( const string &userFriendlyTypeName) 
		: UserInfo( userFriendlyTypeName, typeid(T).name() ) 
	{	

	};

	//! destructor
	~TypedUserInfo<T>()
	{
	}

	//! the only instance of this class (each type T has its own instance)
	//! which is used as a factory for processing related userinfos
	static const TypedUserInfo<T>& instance;
};


/*!
@brief The main userinfo template class. You should derive this class whenever you need 
a new userinfo of a class which is compound from smaller elements (all having its
own userinfo class prepared).

To bind some inner element to its own userinfo class, and also to automate its assembling and 
disassembling, it is necessary to create a instance of an inner templated class BindedElement<T>.
Those attributes have to be initialized in constructor of a new compound userinfo this way:

\code
class BikeUI: public CompoundUserInfo<Bike>
{
private:
	BindedElement<int> year; 
	BindedElement<bool> lights; 
	BindedElement<string> manufacturer; 
public:
	BikeUI()
		:CompoundUserInfo<Bike>("bike"),
		year( this, "year", 0 ),
		lights( this, "electric lights", false ), 
		manufacturer( this, "manufacturer", "unknown")
	{
	}

	...
}
\endcode
*/
template<typename T> class CompoundUserInfo : public TypedUserInfo<T>
{
private:
	//! Elements binding inner XML tags to related userinfos
	vector<BindingFrame*> bindedElements;

protected:

	/*!
	@brief Templated class binding inner element with its XML tag and automating data transfers
	in both directions.
	*/
	template<typename U> class BindedElement: public BindingFrame
	{
	private:
		string name;
		string help;				
		bool release;

		U* pValue;

		const U defaultValue;
		
 
	public:
		U value;

		BindedElement<U>( CompoundUserInfo<T> *parent, string name, U defaultValue, string help ) 
			: name( removeSpaces( name )), help(help), defaultValue( defaultValue )
		{
			parent->bindedElements.push_back( this );
			pValue = NULL;
			value = defaultValue;
		}

		BindedElement<U>( CompoundUserInfo<T> *parent, string name, U defaultValue ) 
			: name( removeSpaces( name )), help(""), defaultValue( defaultValue ), value( defaultValue)
		{
			parent->bindedElements.push_back( this );
			pValue = NULL;
			value = defaultValue;
		}

		~BindedElement<U>()
		{
		}

		void AssemblyComponentsFromXML( DOMElement &element )
		{
			pValue = UserInfo::Assembly<U>( element, name );
			if( pValue ) value = *pValue;			
		}

		void ReleaseMemory()
		{
			if( pValue != NULL )
				delete pValue;
		}

		bool DisassemblyComponentsToXML( DOMElement &element )
		{
			return UserInfo::Disassembly( value, element, name, help );
		}
	};

private:

	void AssemblyComponentsFromXML( DOMElement &element )
	{
		for( unsigned int ind = 0; ind < bindedElements.size(); ind++ )
			bindedElements[ind]->AssemblyComponentsFromXML( element );
	}

	void ReleaseMemory()
	{			
		for( unsigned int ind = 0; ind < bindedElements.size(); ind++ )
			bindedElements[ind]->ReleaseMemory();
	}

	bool DisassemblyComponentsToXML( DOMElement &element )
	{
		for( unsigned int ind = 0; ind < bindedElements.size(); ind++ )
			if( !bindedElements[ind]->DisassemblyComponentsToXML( element ) )
				return false;
		return true;
	}

protected:

	CompoundUserInfo<T>( string userFriendlyTypeName )
		: TypedUserInfo<T>( userFriendlyTypeName )
	{
	}

};


/*!
@brief The main userinfo template class. It should be derived whenever you need 
a new userinfo of a class which does not contain any subelements. It is the case of
basic classes(or types) like int, string, double, etc.

The only thing left is to translate its public string attribute 'value' into a value 
of type T and also implement conversion in the other way round. For that, an overloading
of methods T* AssemblyInstance() /  bool DisassemblyInstance(T  &instance) is fruitful.
See some valued userinfo below as an example.
*/
template<typename T> class ValuedUserInfo : public TypedUserInfo<T>
{
private:
	void AssemblyComponentsFromXML( DOMElement &element )
	{		
		value = Attribute::value.Get( element );
	}

	bool DisassemblyComponentsToXML( DOMElement &element )
	{
		Attribute::value.Set( element, value );
		return true;
	}

protected:
	ValuedUserInfo<T>( string userFriendlyTypeName )
		: TypedUserInfo<T>( userFriendlyTypeName )
	{
	}

	~ValuedUserInfo<T>()
	{
	}

	//! string variable which is automatically binded to a proper DOMElement attribute
	string value;
};

/*!
@brief This class serves to load and/or save DOMElements into/from files
stored on a hard-disk.

Firstly, you associate new RootElement instance with some filename during a time of its 
construtcion. Then, you disassembly some object into the new RootElement instance,
and save it into the file this way:
\code
	CAudi audi;
	RootElement root("cars.xml");
	UserInfo::Disassembly( audi, root, "TT");
	root.Save();
\endcode

In the other way round, when loading object from a XML file, the appropriate code looks like this:
\code
	RootElement root("cars.xml");
	root.Load();
	UserInfo::Assembly<T>(root,"TT");
\endcode
*/
class RootElement 
{
private:
	//! This private attribute has to be declared as the first attribute in the class.
	//! Only this way we can be sure it's constructor is called as the first
	//! and thus Xerces is initialized properly and right on time
	const AssertXercesIsAlive dummy;

	//! DOMDocument containing the root element this instance is associated to
	DOMDocument* pDoc;

	const XMLCh* const transcodedFileName;

	//! DOMImplementation is a base class for the all DOM oparations
	DOMImplementation *pImplementation;

	//! This DOMWriter is used to export internal data into xml file
	DOMWriter *pSerializer;

	void Clean();

public:
	//! attach new RootElement instance to a file (typically with an XML extension)
	RootElement( char* fileName );

	~RootElement();

	//! this method loads root element and all its subnodes from the attached file
	bool Load( void ) ;

	//! this method saves all the previsoulsy attached DOMElements into the file 
	void Save ( void );

	//! this operator allows to use a RootElement instance whenever a DOMElement variable is accepted
	operator DOMElement&();
};


//////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////// BASIC VALUED USER INFOS ///////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////

class BoolUI: public ValuedUserInfo<bool>
{
private:

	bool* AssemblyInstance()
	{		
		if( value == "true" )
			return new bool( true );
		else if( value == "false" )
			return new bool( false );
		else return NULL;
	}

	bool DisassemblyInstance(bool &instance)
	{
		if( instance )
			value = "true";
		else
			value = "false";
		return true; 
	}

public:

	BoolUI()
		: ValuedUserInfo<bool>("bool")
	{
	}
};


class IntUI: public ValuedUserInfo<int>
{
private:

	int* AssemblyInstance()
	{
		return new int( atoi( value.c_str()) );
	}

	bool DisassemblyInstance(int &instance)
	{
		char buff[30];
		sprintf(buff, "%d", instance );
		value = buff;
		return true; 
	}

public:
	IntUI():ValuedUserInfo<int>("int")
	{
	}
};



class DoubleUI: public ValuedUserInfo<double>
{
private:

	double* AssemblyInstance()
	{
		return new double( atof( value.c_str()) );
	}

	bool DisassemblyInstance(double &instance)
	{
		char buff[30];
		sprintf(buff, "%f", instance );
		value = buff;
		return true; 
	}

public:
	DoubleUI():ValuedUserInfo<double>("double")
	{
	}
};


class StringUI: public ValuedUserInfo<string>
{
private:
	string* AssemblyInstance()
	{
		return new string( value );
	}

	bool DisassemblyInstance(string &instance)
	{
		value = instance;
		return true;
	}

public:
	StringUI():ValuedUserInfo<string>("string")
	{
	}
};


#endif // #ifndef UI_H