/*!
  \file
  \brief Bayesian Filtering for generalized autoregressive (ARX) model
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef AR_H
#define AR_H

#include <itpp/itbase.h>
#include "../stat/libFN.h"
#include "../stat/libEF.h"

using namespace itpp;

/*!
* \brief Linear Autoregressive model with Gaussian noise

Regression of the following kind:
\f[
y_t = \theta_1 \psi_1 + \theta_2 + \psi_2 +\ldots + \theta_n \psi_n + r e_t
\f]
where unknown parameters \c rv are \f$[\theta r]\f$, regression vector \f$\psi=\psi(y_{1:t},u_{1:t})\f$ is a known function of past outputs and exogeneous variables \f$u_t\f$. Distrubances \f$e_t\f$ are supposed to be normally distributed:
\f[
e_t \sim \mathcal{N}(0,1).
\f]

Extension for time-variant parameters \f$\theta_t,r_t\f$ may be achived using exponential forgetting (Kulhavy and Zarrop, 1993). In such a case, the forgetting factor \c frg \f$\in <0,1>\f$ should be given in the constructor. Time-invariant parameters are estimated for \c frg = 1.
*/
class ARX: public BMEF {
protected:
	//! Posterior estimate of \f$\theta,r\f$ in the form of Normal-inverse Wishart density
	egiw est;
	//! cached value of est.V
	ldmat &V;
	//! cached value of est.nu
	double &nu;
public:
	//! Full constructor
	ARX ( const RV &rv, const mat &V0, const double &nu0, const double frg0=1.0 ) : BMEF ( rv,frg0 ),est ( rv,V0,nu0 ), V ( est._V() ), nu ( est._nu() )
	{last_lognc=est.lognc();};

	//!Copy constructor
	ARX ( const ARX &A0 ) : BMEF ( A0),est ( rv,A0.V,A0.nu ), V ( est._V() ), nu ( est._nu() ) {};

	//!Auxiliary function
	ARX* _copy_(bool changerv=false);
	
//	//! Set parameters given by moments, \c mu (mean of theta), \c R (mean of R) and \c C (variance of theta)
//	void set_parameters ( const vec &mu, const mat &R, const mat &C, double dfm){};
	//! Set sufficient statistics
	void set_parameters ( const ldmat &V0, const double &nu0 )
	{est._V() =V0;est._nu() =nu0;last_lognc=est.lognc();}
	void set_statistics ( const BMEF* BM0 );
	//! Returns sufficient statistics
	void get_parameters ( mat &V0, double &nu0 ) {V0=est._V().to_mat(); nu0=est._nu();}
	//! Here \f$dt = [y_t psi_t] \f$.
	void bayes ( const vec &dt, const double w );
	void bayes ( const vec &dt ) {bayes ( dt,1.0 );};
	const epdf& _epdf() const {return est;}
	double logpred ( const vec &dt ) const;
	void flatten (BMEF* B ) {
		ARX* A=dynamic_cast<ARX*>(B);
		// nu should be equal to B.nu
		est.pow ( A->nu/nu);
		if(evalll){last_lognc=est.lognc();}
	}

	//! Brute force structure estimation.\return indeces of accepted regressors.
	ivec structure_est ( egiw Eg0 );
};


#endif // AR_H


