/*!
  \file
  \brief Bayesian Models (bm) that use Bayes rule to learn from observations
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef BM_H
#define BM_H

#include <itpp/itbase.h>
//#include <std>

using namespace itpp;

/*!
* \brief Class representing variables, most often random variables

* More?...
*/
class RV {
	int size,len;
	ivec ids;
	ivec sizes;
	ivec times;
	ivec obs;
	Array<std::string> names;

private:
	void init ( ivec in_ids, Array<std::string> in_names, ivec in_sizes, ivec in_times, ivec in_obs );
public:
	//! Full constructor which is called by the others
	RV ( ivec in_ids, Array<std::string> in_names, ivec in_sizes, ivec in_times, ivec in_obs );
	//! default constructor
	RV ( ivec ids );
	//! Empty constructor will be set later
	RV ();
	
	//! Printing output e.g. for debugging.
	friend std::ostream &operator<< ( std::ostream &os, const RV &rv );

	//! Return length (number of scalars) of the RV.
	int count() {return size;}
	//TODO why not inline and later??
	
	//! Find indexes of another rv in self
	ivec find(RV rv2);
	//! Add (concat) another variable to the current one
	RV add(RV rv2);
	//! Subtract  another variable from the current one
	RV subt(RV rv2);
	//! Select only variables at indeces ind
	RV subselect(ivec ind);
	//! Select only variables at indeces ind
	RV operator()(ivec ind);
	//! Generate new \c RV with \c time shifted by delta.
	void t(int delta);
};




//! Class representing function of variables
class fnc {
	RV rv;
};

//! Bayesian Model of the world, i.e. all uncertainty is modeled by probabilities.
class BM {
public:
	//!Logarithm of marginalized data likelihood.
	double ll;

	/*! \brief Incremental Bayes rule
	@param dt vector of input data
	@param evall If true, the filter will compute likelihood of the data record and store it in \c ll
	*/
	virtual void bayes ( const vec &dt, bool evall=true ) = 0;
	//! Batch Bayes rule (columns of Dt are observations)
	void bayes ( mat Dt );
};

//! Probability density function with numerical statistics, e.g. posterior density.
class epdf {
	RV rv;
public:
	//! Returns the required moment of the epdf
//	virtual vec moment ( const int order = 1 );
	//! Returns a sample from the density, $x \sim epdf(rv)$
	virtual vec sample (){};
	virtual double eval(const vec &val){};
};

//! Conditional probability density, e.g. modeling some dependencies.
class mpdf {
	//! modeled random variable
	RV rv;
	//! random variable in condition
	RV rvc;
public:

	//! Returns the required moment of the epdf
//	virtual fnc moment ( const int order = 1 );
	//! Returns a sample from the density conditioned on \c cond, $x \sim epdf(rv|cond)$
	virtual vec samplecond (vec &cond, double lik){};
	virtual void condition (vec &cond){};
};

/*! \brief Abstract class for discrete-time sources of data.

The class abstracts operations of: (i) data aquisition, (ii) data-preprocessing, (iii) scaling of data, and (iv) data resampling from the task of estimation and control.
Moreover, for controlled systems, it is able to receive the desired control action and perform it in the next step. (Or as soon as possible).

*/
class DS {
	//!Observed variables, returned by \c getdata().
	RV Drv; 
	//!Action variables, accepted by \c write().
	RV Urv; //
public:
	//! Returns full vector of observed data 
	void getdata(vec &dt);
	//! Returns data records at indeces.
	void getdata(vec &dt, ivec &indeces);
	//! Accepts action variable and schedule it for application.
	void write(vec &ut);
	//! Accepts action variables at specific indeces	
	void write(vec &ut, ivec &indeces);
	/*! \brief Method that assigns random variables to the datasource. 
	Typically, the datasource will be constructed without knowledge of random variables. This method will associate existing variables with RVs.
	
	(Inherited from m3k, may be deprecated soon).
	*/
	void linkrvs(RV &drv, RV &urv);
	
	//! Moves from $t$ to $t+1$, i.e. perfroms the actions and reads response of the system. 
	void step();
};


#endif // BM_H
