/*!
  \file
  \brief TR 2525 file for testing Toy Problem of mpf for Covariance Estimation
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/


#include <itpp/itbase.h>
#include <estim/libKF.h>
#include <estim/libPF.h>
#include <stat/libFN.h>

#include "pmsm.h"
#include "simulator.h"
#include "sim_profiles.h"

using namespace itpp;
//!Extended Kalman filter with unknown \c Q
class EKF_unQ : public EKFCh , public BMcond {
public:
	//! Default constructor
	EKF_unQ ( RV rx, RV ry,RV ru,RV rQ ) :EKFCh ( rx,ry,ru ),BMcond ( rQ ) {};
	void condition ( const vec &Q0 ) {
		Q.setD ( Q0,0 );
		//from EKF
		preA.set_submatrix ( dimy+dimx,dimy,Q._Ch() );
	};
};


int main() {
	// Kalman filter
	int Ndat = 9000;
	double h = 1e-6;
	int Nsimstep = 125;
	int Npart = 20;

	// internal model
	IMpmsm fxu;
	//                  Rs    Ls        dt       Fmag(Ypm)    kp   p    J     Bf(Mz)
	fxu.set_parameters ( 0.28, 0.003465, Nsimstep*h, 0.1989, 1.5 ,4.0, 0.04, 0.0 );
	// observation model
	OMpmsm hxu;

	vec mu0= "0.0 0.0 0.0 0.0";
	vec Qdiag ( "1e-5 1e-5 0.0001 0.0001" ); //zdenek: 0.01 0.01 0.0001 0.0001
	vec Rdiag ( "1e-8 1e-8" ); //var(diff(xth)) = "0.034 0.034"
	chmat Q ( Qdiag );
	chmat R ( Rdiag );
	EKFCh KFE ( rx,ry,ru );
	KFE.set_parameters ( &fxu,&hxu,Q,R );
	KFE.set_est ( mu0, chmat ( zeros ( 4 ) ) );

	RV rQ ( "{Q }","4" );
	EKF_unQ KFEp ( rx,ry,ru,rQ );
	KFEp.set_parameters ( &fxu,&hxu,Q,R );
	KFEp.set_est ( mu0, chmat ( zeros ( 4 ) ) );

	mgamma_fix evolQ ( rQ,rQ );
	MPF<EKF_unQ> M ( rx,rQ,evolQ,evolQ,Npart,KFEp );
	// initialize
	evolQ.set_parameters ( 10.0, Qdiag, 1.0 ); //sigma = 1/10 mu
	evolQ.condition (Qdiag ); //Zdenek default
	epdf& pfinit=evolQ._epdf();
	M.set_est ( pfinit );
	evolQ.set_parameters ( 100.0, Qdiag, 0.99 ); //sigma = 1/10 mu
	
	//

	const epdf& KFEep = KFE._epdf();
	const epdf& Mep = M._epdf();

	mat Xt=zeros ( Ndat ,4 ); //true state from simulator
	mat Dt=zeros ( Ndat,2+2 ); //observation
	mat XtE=zeros ( Ndat, 4 );
	mat Qtr=zeros ( Ndat, 4 );
	mat XtM=zeros ( Ndat,4+4 ); //Q + x

	// SET SIMULATOR
	pmsmsim_set_parameters ( 0.28,0.003465,0.1989,0.0,4,1.5,0.04, 200., 3e-6, h );
	vec dt ( 2 );
	vec ut ( 2 );
	vec xt ( 4 );
	vec xtm=zeros(4);
	double Ww=0.0;
	vec vecW="1 2 4 9 4 2 0 -4 -9 -16 -4 0 0";

	for ( int tK=1;tK<Ndat;tK++ ) {
		//Number of steps of a simulator for one step of Kalman
		for ( int ii=0; ii<Nsimstep;ii++ ) {
			//simulator
			sim_profile_vec01t(Ww,vecW);
			pmsmsim_step ( Ww );
		};
		ut(0) = KalmanObs[4];
		ut(1) = KalmanObs[5];
		xt = fxu.eval(xtm,ut) + diag(sqrt(Qdiag))*randn(4);
		dt = hxu.eval(xt,ut);
		xtm = xt;
		
		
		//Variances 
		if (tK==1000)  Qdiag(0)*=10; 
		if (tK==2000) Qdiag(0)/=10; 
		if (tK==3000)  Qdiag(1)*=10; 
		if (tK==4000) Qdiag(1)/=10; 
		if (tK==5000)  Qdiag(2)*=100; 
		if (tK==6000) Qdiag(2)/=100; 
		if (tK==7000)  Qdiag(3)*=100; 
		if (tK==8000) Qdiag(3)/=100; 
		
		//estimator
		KFE.bayes ( concat ( dt,ut ) );
		M.bayes ( concat ( dt,ut ) );

		Xt.set_row ( tK, xt); //vec from C-array
		Dt.set_row ( tK, concat ( dt,ut));
		Qtr.set_row ( tK, Qdiag);
		XtE.set_row ( tK,KFEep.mean() );
		XtM.set_row ( tK,Mep.mean() );
	}

	it_file fou ( "mpf_test.it" );

	fou << Name ( "xth" ) << Xt;
	fou << Name ( "Dt" ) << Dt;
	fou << Name ( "Qtr" ) << Qtr;
	fou << Name ( "xthE" ) << XtE;
	fou << Name ( "xthM" ) << XtM;
	//Exit program:

	return 0;
}

