/*!
  \file
  \brief TR 2525 file for testing Toy Problem of mpf for Covariance Estimation
  \author Vaclav Smidl.

  \ingroup PMSM

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/


#include <itpp/itbase.h>
#include <estim/libKF.h>
#include <estim/libPF.h>
#include <stat/libFN.h>

#include "pmsm.h"
#include "simulator.h"
#include "sim_profiles.h"

using namespace itpp;

//!Extended Kalman filter with unknown \c Q
class EKFCh_cond : public EKFCh , public BMcond {
public:
	//! Default constructor
	EKFCh_cond ( RV rx, RV ry,RV ru,RV rC ) :EKFCh ( rx,ry,ru ),BMcond ( rC ) {};
	void condition ( const vec &val ) {
		pfxu->condition( val );
	};
};

class IMpmsm_delta :  public IMpmsm {
	protected:
		vec ud;
	public:
		IMpmsm_delta() :IMpmsm(),ud(2) {};
	//! Set mechanical and electrical variables

		void condition(const vec &val){ud = val;}
		vec eval ( const vec &x0, const vec &u0 ) {
		// last state
			double iam = x0 ( 0 );
			double ibm = x0 ( 1 );
			double omm = x0 ( 2 );
			double thm = x0 ( 3 );
			double uam = u0 ( 0 );
			double ubm = u0 ( 1 );

			vec xk=zeros ( 4 );
		//ia
			xk ( 0 ) = ( 1.0- Rs/Ls*dt ) * iam + Ypm/Ls*dt*omm * sin ( thm ) + (uam+ud(0))*dt/Ls;
		//ib
			xk ( 1 ) = ( 1.0- Rs/Ls*dt ) * ibm - Ypm/Ls*dt*omm * cos ( thm ) + (ubm+ud(1))*dt/Ls;
		//om
			xk ( 2 ) = omm + kp*p*p * Ypm/J*dt* ( ibm * cos ( thm )-iam * sin ( thm ) ) - p/J*dt*Mz;
		//th
			xk ( 3 ) = thm + omm*dt; // <0..2pi>
			if ( xk ( 3 ) >pi ) xk ( 3 )-=2*pi;
			if ( xk ( 3 ) <-pi ) xk ( 3 ) +=2*pi;
			return xk;
		}

};


int main() {
	// Kalman filter
	int Ndat = 9000;
	double h = 1e-6;
	int Nsimstep = 125;
	int Npart = 200;

	// internal model
	IMpmsm_delta fxu;
	//                  Rs    Ls        dt       Fmag(Ypm)    kp   p    J     Bf(Mz)
	fxu.set_parameters ( 0.28, 0.003465, Nsimstep*h, 0.1989, 1.5 ,4.0, 0.04, 0.0 );
	// observation model
	OMpmsm hxu;

	vec mu0= "0.0 0.0 0.0 0.0";
	vec Qdiag ( "0.6 0.6 0.001 0.000001" ); //zdenek: 0.01 0.01 0.0001 0.0001
	vec Rdiag ( "1e-8 1e-8" ); //var(diff(xth)) = "0.034 0.034"
	chmat Q ( Qdiag );
	chmat R ( Rdiag );
	EKFCh KFE ( rx,ry,ru );
	KFE.set_parameters ( &fxu,&hxu,Q,R );
	KFE.set_est ( mu0, chmat ( zeros ( 4 ) ) );

	RV rUd ( "{ud }", "2" );
	EKFCh_cond KFEp ( rx,ry,ru,rUd );
	KFEp.set_parameters ( &fxu,&hxu,Q,R );
	KFEp.set_est ( mu0, chmat ( zeros ( 4 ) ) );

	mlnorm<ldmat> evolUd ( rUd,rUd );
	MPF<EKFCh_cond> M ( rx,rUd,evolUd,evolUd,Npart,KFEp );
	// initialize
	vec Ud0="0 0";
	evolUd.set_parameters ( eye(2), vec_2(0.0,0.0), ldmat(eye(2)) ); 
	evolUd.condition (Ud0 );
	epdf& pfinit=evolUd._epdf();
	M.set_est ( pfinit );
	evolUd.set_parameters ( eye(2), vec_2(0.0,0.0), ldmat(0.1*eye(2)) ); 
	
	mat Xt=zeros ( Ndat ,4 ); //true state from simulator
	mat Dt=zeros ( Ndat,2+2 ); //observation
	mat XtE=zeros ( Ndat, 4 );
	mat Qtr=zeros ( Ndat, 4 );
	mat XtM=zeros ( Ndat,2+4 ); //W + x

	// SET SIMULATOR
	pmsmsim_set_parameters ( 0.28,0.003465,0.1989,0.0,4,1.5,0.04, 200., 3e-6, h );
	vec dt ( 2 );
	vec ut ( 2 );
	vec xt ( 4 );
	vec xtm=zeros(4);
	double Ww=0.0;
	vec vecW="1 2 4 9 4 2 0 -4 -9 -16 -4 0 0 0";
	vecW*=10.0;

	for ( int tK=1;tK<Ndat;tK++ ) {
		//Number of steps of a simulator for one step of Kalman
		for ( int ii=0; ii<Nsimstep;ii++ ) {
			//simulator
			sim_profile_vec01t(Ww,vecW);
			pmsmsim_step ( Ww );
		};
		ut(0) = KalmanObs[0];
		ut(1) = KalmanObs[1];
		dt(0) = KalmanObs[2];
		dt(1) = KalmanObs[3];
		xt = vec(x,4);
		
		//estimator
		KFE.bayes ( concat ( dt,ut ) );
		M.bayes ( concat ( dt,ut ) );

		Xt.set_row ( tK, xt); //vec from C-array
		Dt.set_row ( tK, concat ( dt,ut));
		Qtr.set_row ( tK, Qdiag);
		XtE.set_row ( tK,KFE._e()->mean() );
		XtM.set_row ( tK,M._e()->mean() );
	}

	it_file fou ( "mpf_u_delta.it" );

	fou << Name ( "xth" ) << Xt;
	fou << Name ( "Dt" ) << Dt;
	fou << Name ( "Qtr" ) << Qtr;
	fou << Name ( "xthE" ) << XtE;
	fou << Name ( "xthM" ) << XtM;
	//Exit program:

	return 0;
}

