/*
  \file
  \brief Models for synchronous electric drive using IT++ and BDM
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#include <itpp/itbase.h>
#include <estim/libKF.h>
#include <estim/libPF.h>
#include <stat/libFN.h>

#include "pmsm.h"

using namespace bdm;
/*
// PMSM with Q on Ia and Ib given externally
class EKF_unQ : public EKF<chmat> , public BMcond {
public:
	EKF_unQ( rx,ry,ru,rQ):EKF<chmat>(rx,ry,ru),BMcond(rQ){};
	void condition(const vec &Q0){};
};*/


int main() {
	// Kalman filter
	int Ndat = 1000;

//	cout << KF;
	// internal model
	IMpmsm fxu;
	//                  Rs    Ls        dt       Fmag(Ypm) kp   p    J     Bf(Mz)
	fxu.set_parameters ( 0.28, 0.003465, 20*1e-6, 0.1989,   1.5 ,4.0, 0.04, 0.0 );
	// observation model
	OMpmsm hxu;

	vec mu0= "100 100 100 1";
	vec Qdiag ( "0.1 0.1 0.01 0.01" );
	vec Rdiag ( "0.02 0.02" );
	vec vQ = "0.01:0.01:100";
	chmat Q ( Qdiag );
	chmat R ( Rdiag );
	EKFCh KFE ( rx,ry,ru );
	KFE.set_est ( mu0, chmat ( 1000*ones ( 4 ) ) );
	KFE.set_parameters ( &fxu,&hxu,Q,R );

	mat ll(100,Ndat);

	EKFCh* kfArray[100];

	for ( int i=0;i<100;i++ ) {
		vec Qid ( Qdiag );
		Qid ( 0 ) = vQ ( i ); Qid ( 1 ) = vQ ( i );
		kfArray[i]= new EKFCh ( rx,ry,ru );
		kfArray[i]->set_est ( mu0, chmat ( 1000*ones ( 4 ) ) );
		kfArray[i]->set_parameters ( &fxu,&hxu,chmat ( Qid ),R );
	}

	const epdf& KFEep = KFE._epdf();

	//simulator values
	vec dt ( 2 );
	vec wt ( 2 );
	vec ut ( 2 );
	vec xt=mu0;
	vec et ( 4 );

	mat Xt=zeros ( 4,Ndat );
	mat XtE=zeros ( 4,Ndat );
	Xt.set_col ( 0,KFEep.mean() );

	for ( int t=1;t<Ndat;t++ ) {
		//simulator
		UniRNG.sample_vector ( 2,wt );

		if ( rem ( t,500 ) <200 ) ut = rem ( t,500 ) *ones ( 2 );
		else
			ut=zeros ( 2 );

		NorRNG.sample_vector ( 4,et );
		NorRNG.sample_vector ( 2,wt );
		xt = fxu.eval ( xt,ut ) + Q.sqrt_mult ( et );
		dt = hxu.eval ( xt,ut ) + R.sqrt_mult ( wt );

		//estimator
		KFE.bayes ( concat ( dt,ut ) );
		for ( int i=0;i<100;i++ ) {kfArray[i]->bayes( concat ( dt,ut ) );ll(i,t)=ll(i,t-1) + kfArray[i]->_ll();
		}
		
		Xt.set_col ( t,xt );
		XtE.set_col ( t,KFEep.mean() );
	}

	it_file fou ( "pmsm.it" );

	fou << Name ( "xth" ) << Xt;
	fou << Name ( "xthE" ) << XtE;
	fou << Name ( "ll" ) << ll;
	//Exit program:
	return 0;

}
