/*!
  \file
  \brief Common DataSources.
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef DS_H
#define DS_H


#include "libBM.h"
#include "libEF.h"


namespace bdm {
	/*!
	* \brief Memory storage of off-line data column-wise

	The data are stored in an internal matrix \c Data . Each column of Data corresponds to one discrete time observation \f$t\f$. Access to this matrix is via indexes \c rowid and \c delays.

	The data can be loaded from a file.
	*/
	class MemDS : public DS {
		//! internal matrix of data
		mat Data;
		//! active column in the Data matrix
		int time;
		//!  vector of rows that are presented in Dt
		ivec rowid;
		//! vector of delays that are presented in Dt
		ivec delays;

	public:
		void getdata ( vec &dt );
		void getdata ( vec &dt, const ivec &indeces );
		void linkrvs ( RV &drv, RV &urv );
		void write ( vec &ut ) {it_error ( "MemDS::write is not supported" );}
		void write ( vec &ut,ivec &indexes ) {it_error ( "MemDS::write is not supported" );}
		void step();
		//!Default constructor
		MemDS ( mat &Dat, ivec &rowid, ivec &delays );
	};

	/*!
	\brief Generator of ARX data

	*/
	class ArxDS : public DS {
		//! Rv of the regressor
		RV Rrv;
		//! Rv of the history (full regressor)
		RV Hrv;
		//! Internal storage of results
		vec Y;
		//! History, ordered as \f$[u_t, y_{t-1 }, u_{t-1}, \ldots]\f$ 
		vec H;
		//! temporary variable for regressor
		vec rgr;
		//! data link: val = y, cond = u; local = rgr;
		datalink_e2e rgrlnk;
		//! model of Y - linear Gaussian 
		mlnorm<chmat> model;
		public:
		void getdata ( vec &dt ){it_assert_debug(dt.length()==Y.length(),"ArxDS"); 
			dt=concat(Y,H.left(Urv.count()));};
		void getdata ( vec &dt, const ivec &indexes ){it_assert_debug(dt.length()==Y.length(),"ArxDS"); dt=Y(indexes);};
		void write ( vec &ut ){it_assert_debug(ut.length()==Urv.count(),"ArxDS"); H.set_subvector(0,ut);};
		void write ( vec &ut, const ivec &indexes ){it_assert_debug(ut.length()==Urv.count(),"ArxDS"); set_subvector(H,indexes,ut);};
		void step();
		//!Default constructor
		ArxDS ( RV &drv, RV &urv, RV &rrv);
		//! Set parameters of the internal model
		void set_parameters(const mat &Th0, const vec mu0, const chmat &sqR0)
		{model.set_parameters(Th0, mu0, sqR0); };
	};

}; //namespace

#endif // DS_H
