/*!
 * \file
 * \brief Matrices in decomposed forms (LDL', LU, UDU', etc).
 * \author Vaclav Smidl.
 *
 * -----------------------------------
 * BDM++ - C++ library for Bayesian Decision Making under Uncertainty
 *
 * Using IT++ for numerical operations
 * -----------------------------------
 */

#ifndef DC_H
#define DC_H

#include <itpp/itbase.h>

using namespace itpp;

/*! \brief Virtual class for representation of double symmetric matrices in square-root form.

All operations defined on this class should be optimized for the chosed decomposition.
*/
class sqmat
{
	public:
		/*!
		 * Perfroms a rank-1 update by outer product of vectors: $V = V + w v v'$.
		 * @param  v Vector forming the outer product to be added
		 * @param  w weight of updating; can be negative

		 BLAS-2b operation.
		 */
		virtual void opupdt ( const vec &v, double w ) =0;

		/*! \brief Conversion to full matrix.
		*/

		virtual mat to_mat() =0;

		/*! \brief Inplace symmetric multiplication by a SQUARE matrix $C$, i.e. $V = C*V*C'$
		@param C multiplying matrix,
		*/
		virtual void mult_sym ( const mat &C ) =0;
		
		/*! \brief Inplace symmetric multiplication by a SQUARE transpose of matrix $C$, i.e. $V = C'*V*C$
		@param C multiplying matrix,
		*/
		virtual void mult_sym_t ( const mat &C ) =0;


		/*!
		\brief Logarithm of a determinant.

		*/
		virtual double logdet() const =0;

		/*!
		\brief Multiplies square root of $V$ by vector $x$.

		Used e.g. in generating normal samples.
		*/
		virtual vec sqrt_mult (const vec &v ) =0;

		/*!
		\brief Evaluates quadratic form $x= v'*V*v$;

		*/
		virtual double qform (const vec &v ) =0;

//	//! easy version of the
//	sqmat inv();

		//! Clearing matrix so that it corresponds to zeros.
		virtual void clear() =0;

		//! Reimplementing common functions of mat: cols().
		int cols() const {return dim;};

		//! Reimplementing common functions of mat: cols().
		int rows() const {return dim;};

	protected:
		int dim;
};


/*! \brief Fake sqmat. This class maps sqmat operations to operations on full matrix.

This class can be used to compare performance of algorithms using decomposed matrices with perormance of the same algorithms using full matrices;
*/
class fsqmat: public sqmat
{
	protected:
		mat M;
	public:
		void opupdt ( const vec &v, double w );
		mat to_mat() ;
		void mult_sym ( const mat &C);
		void mult_sym_t ( const mat &C);
		void mult_sym ( const mat &C, fsqmat &U);
		void mult_sym_t ( const mat &C, fsqmat &U);
		void clear();

		//! Default initialization
		fsqmat(); // mat will be initialized OK
		//! Default initialization with proper size
		fsqmat(const int dim0); // mat will be initialized OK
		//! Constructor
		fsqmat ( const mat &M );

		/*! \brief Matrix inversion preserving the chosen form.

		@param Inv a space where the inverse is stored.

		*/
		virtual void inv ( fsqmat &Inv );

		double logdet() const {return log ( det ( M ) );};
		double qform (const  vec &v ) {return ( v* ( M*v ) );};
		vec sqrt_mult (const vec &v ) {it_error ( "not implemented" );return v;};

		fsqmat& operator += ( const fsqmat &A ) {M+=A.M;return *this;};
		fsqmat& operator -= ( const fsqmat &A ) {M-=A.M;return *this;};
		fsqmat& operator *= ( double x ) {M*=x;return *this;};
//		fsqmat& operator = ( const fsqmat &A) {M=A.M; return *this;};

		friend std::ostream &operator<< ( std::ostream &os, fsqmat &sq );

};

class ldmat: sqmat
{
	public:

		//! Construct by copy of L and D.
		ldmat ( const mat &L, const vec &D );
		//! Construct by decomposition of full matrix V.
		ldmat (const mat &V );
		//! Construct diagonal matrix with diagonal D0
		ldmat ( vec D0 );
		//!Default constructor
		ldmat ();
		//! Default initialization with proper size
		ldmat(const int dim0);

		// Reimplementation of compulsory operatios

		void opupdt ( const vec &v, double w );
		mat to_mat();
		void mult_sym ( const mat &C);
		void mult_sym_t ( const mat &C);
		void add ( const ldmat &ld2, double w=1.0 );
		double logdet() const;
		double qform (const vec &v );
//	sqmat& operator -= ( const sqmat & ld2 );
		void clear();
		int cols() const;
		int rows() const;
		vec sqrt_mult ( const vec &v );

		/*! \brief Matrix inversion preserving the chosen form.

		@param Inv a space where the inverse is stored.

		*/
		virtual void inv ( ldmat &Inv );

		/*! \brief Symmetric multiplication of $U$ by a general matrix $C$, result of which is stored in the current class.

		@param U a space where the inverse is stored.

		*/
		void mult_sym ( const mat &C, ldmat &U);

		/*! \brief Symmetric multiplication of $U$ by a transpose of a general matrix $C$, result of which is stored in the current class.

		@param U a space where the inverse is stored.

		*/
		void mult_sym_t ( const mat &C, ldmat &U);


		/*! \brief Transforms general $A'D0 A$ into pure $L'DL$

		The new decomposition fullfills: $A'*diag(D)*A = self.L'*diag(self.D)*self.L$

		@param A general matrix
		@param D0 general vector

		*/
		void ldform (const mat &A,const vec &D0 );

		ldmat& operator += ( const ldmat &ldA );
		ldmat& operator -= ( const ldmat &ldA );
		ldmat& operator *= ( double x );

		friend std::ostream &operator<< ( std::ostream &os, ldmat &sq );

	protected:
		vec D;
		mat L;

};

//////// Operations:

inline ldmat& ldmat::operator += ( const ldmat &ldA )  {this->add ( ldA );return *this;}
inline ldmat& ldmat::operator -= ( const ldmat &ldA )  {this->add ( ldA,-1.0 );return *this;}
inline int ldmat::cols() const {return dim;}
inline int ldmat::rows() const {return dim;}

#endif // DC_H
