
/*!
  \file
  \brief UI for ARX estimators
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef ARX_UI_H
#define ARX_UI_H


#include "arx.h"
#include "stat/loggers_ui.h"

/*! UI for ARX estimator

The ARX is constructed from a structure with fields:
\code
estimator = {
    type = "ARX";
	y = {type="rv", ...}   // description of output variables
	rgr = {type="rv", ...} // description of regressor variables
	constant = true;       // boolean switch if the constant term is modelled or not

	//optional fields
	dV0 = [1e-3, 1e-5, 1e-5, 1e-5]; 
                           // default: 1e-3 for y, 1e-5 for rgr
	nu0 = 6;               // default: rgrlen + 2
	frg = 1.0;             // forgetting, default frg=1.0
};
\endcode

The estimator will assign names of the posterior in the form ["theta_i" and "r_i"]
*/
class UIARX : public UIbuilder {
public:
	UIARX() :UIbuilder ( "ARXest" ) {};
	bdmroot* build ( Setting &S ) const {
		RV *yrv;  UIbuild(S["y"],yrv);
		RV *rrv; UIbuild(S["rgr"],rrv);
		int ylen = yrv->_dsize();
		int rgrlen = rrv->_dsize();
		
		//init
		mat V0;
		if ( S.exists ( "dV0" ) ) {
			V0=diag ( getvec(S["dV0"]) );
		}
		else {
			V0=concat ( 1e-3*ones ( ylen ), 1e-5*ones ( rgrlen ) );
		}
		double nu0;
		if ( S.exists ( "nu0" ) ) {
			nu0=double(S["nu0"]);
		}
		else {
			nu0 = rgrlen+ylen+2;
		}
		double frg;
		if ( S.exists ( "frg" ) ) {
			frg = S["frg"];
		}
		else {
			frg = 1.0;
		}
		ARX* A=new ARX;
		A->set_parameters(frg);
		A->set_statistics(ylen,V0, nu0);
		A->set_drv(concat(*yrv,*rrv));
		
		//name results (for logging)
		A->set_rv(RV("{theta r }", vec_2(ylen*rgrlen, ylen*ylen)));
		delete yrv; delete rrv;
		return A;
	};
};

UIREGISTER ( UIARX );

#endif // DS_UI_H
