/*!
  \file
  \brief DataSource for experiments with realistic simulator of the PMSM model
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#include <stat/loggers.h>
#include "simulator.h"
#include "pmsm.h"

//! Simulator of PMSM machine with predefined profile on omega
class pmsmDS : public DS {

protected:
	//! indeces of logged variables
	int L_x, L_ou, L_oy, L_iu, L_optu;
	//! Setpoints of omega in timespans given by dt_prof
	vec profileWw;
	//! time-step for profiles
	double dt_prof;
	//! Number of miliseconds per discrete time step
	int Dt;
	//! options for logging, string in which each character has a meaning:
	//! \c u - log predictions of 'true' voltage
	string opt;
public:
	//! Constructor with fixed sampling period
	pmsmDS (string opt0="" ) : Dt ( 125 ), opt(opt0) {}
	void set_parameters ( double Rs0, double Ls0, double Fmag0, double Bf0, double p0, double kp0, double J0, double Uc0, double DT0, double dt0 ) {
		pmsmsim_set_parameters ( Rs0, Ls0, Fmag0, Bf0, p0, kp0, J0, Uc0, DT0, dt0 );
	}
	void getdata ( vec &dt ) {dt=vec ( KalmanObs,6 );}
	void write ( vec &ut ) {}

	void step() {
		static int ind=0;
		static double dW; // increase of W
		static double Ww; // W
		if ( t>=dt_prof*ind ) {
			ind++;
			if ( ind<profileWw.length() ) {
				//linear increase
				if ( profileWw.length() ==1 ) {
					Ww=profileWw ( 0 ); dW=0.0;}
				else {
					dW = profileWw ( ind )-profileWw ( ind-1 );
					dW *=125e-6/dt_prof;
				}
			}
			else {
				dW = 0;
			}
		}
		Ww += dW;
		//Simulate Dt seconds!
		for ( int i=0;i<Dt;i++ ) {	pmsmsim_step ( Ww );}
	};

	void log_add ( logger &L ) {
		L_x = L.add ( rx, "x" );
		L_oy = L.add ( ry, "obs" );
		L_ou = L.add ( ru, "obs" );
		L_iu = L.add ( ru, "true" );
		// log differences
		if (opt.find("u")==string::npos){
			L_optu = L.add(ru, "model");
		}
	}

	void logit ( logger &L ) {
		L.logit ( L_x, vec ( x,4 )	);
		L.logit ( L_oy, vec_2 ( KalmanObs[2],KalmanObs[3] ) );
		L.logit ( L_ou, vec_2 ( KalmanObs[0],KalmanObs[1] ) );
		L.logit ( L_iu, vec_2 ( KalmanObs[4],KalmanObs[5] ) );
		if (opt.find("u")==string::npos){
			double sq3=sqrt(3.0);
			double ua,ub;
			double i1=x[0];
			double i2=0.5*(-i1+sq3*x[1]);
			double i3=0.5*(-i1-sq3*x[1]);
			double u1=KalmanObs[0];
			double u2=0.5*(-u1+sq3*KalmanObs[1]);
			double u3=0.5*(-u1-sq3*KalmanObs[1]);
			
			double du1=0.7*(double(i1>0.1) - double(i1<-0.1))+0.05*i1;
			double du2=0.7*(double(i2>0.1) - double(i2<-0.1))+0.05*i2;
			double du3=0.7*(double(i3>0.1) - double(i3<-0.1))+0.05*i3;
			ua = (2.0*(u1-du1)-(u2-du2)-(u3-du3))/3.0;
			ub = ((u2-du2)-(u3-du3))/sq3;
			L.logit( L_optu , vec_2 (ua,ub));
		}
		}

	void set_profile ( double dt, const vec &Ww ) {dt_prof=dt; profileWw=Ww;}
};
