/*!
  \file
  \brief Bayesian Filtering using stochastic sampling (Particle Filters)
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef PF_H
#define PF_H


#include "../stat/libEF.h"

namespace bdm {

/*!
* \brief Trivial particle filter with proposal density equal to parameter evolution model.

Posterior density is represented by a weighted empirical density (\c eEmp ).
*/

class PF : public BM {
protected:
	//!number of particles;
	int n;
	//!posterior density
	eEmp est;
	//! pointer into \c eEmp
	vec &_w;
	//! pointer into \c eEmp
	Array<vec> &_samples;
	//! Parameter evolution model
	mpdf *par;
	//! Observation model
	mpdf *obs;
public:
	//! \name Constructors
	//!@{
	PF ( ) :est(), _w ( est._w() ),_samples ( est._samples() ) {};
	PF( mpdf *par0, mpdf *obs0, epdf *epdf0, int n0 ) :
			est ( ),_w ( est._w() ),_samples ( est._samples() ) 
			{ par = par0; obs=obs0; est.set_parameters ( ones(n0),epdf0 ); };
	void set_parameters ( mpdf *par0, mpdf *obs0, int n0 ) 
			{ par = par0; obs=obs0; n=n0; est.set_n(n);};
	void set_statistics (const vec w0, epdf *epdf0){est.set_parameters ( w0,epdf0 );};
	//!@}
	//! Set posterior density by sampling from epdf0
	void set_est ( const epdf &epdf0 );
	void bayes ( const vec &dt );
	//!access function
	vec* __w() {return &_w;}
};

/*!
\brief Marginalized Particle filter

Trivial version: proposal = parameter evolution, observation model is not used. (it is assumed to be part of BM).
*/

template<class BM_T>

class MPF : public PF {
	BM_T* Bms[10000];

	//! internal class for MPDF providing composition of eEmp with external components

class mpfepdf : public epdf  {
	protected:
		eEmp &E;
		vec &_w;
		Array<const epdf*> Coms;
	public:
		mpfepdf ( eEmp &E0) :
				epdf ( ), E ( E0 ),  _w ( E._w() ),
				Coms ( _w.length() ) {
		};
		void set_elements ( int &i, double wi, const epdf* ep )
		{_w ( i ) =wi; Coms ( i ) =ep;};

		void set_n(int n){E.set_n(n); Coms.set_length(n);}
		vec mean() const {
			// ugly
			vec pom=zeros ( Coms(0)->dimension() );
			for ( int i=0; i<_w.length(); i++ ) {pom += Coms ( i )->mean() * _w ( i );}
			return concat ( E.mean(),pom );
		}
		vec variance() const {
			// ugly
			vec pom=zeros ( Coms ( 0 )->dimension() );
			vec pom2=zeros (  Coms ( 0 )->dimension() );
			for ( int i=0; i<_w.length(); i++ ) {
				pom += Coms ( i )->mean() * _w ( i );
				pom2 += ( Coms ( i )->variance() + pow ( Coms ( i )->mean(),2 ) ) * _w ( i );
			}
			return concat ( E.variance(),pom2-pow ( pom,2 ) );
		}

		vec sample() const {it_error ( "Not implemented" );return 0;}

		double evallog ( const vec &val ) const {it_error ( "not implemented" ); return 0.0;}
	};

	//! estimate joining PF.est with conditional
	mpfepdf jest;

public:
	//! Default constructor.
	MPF ( mpdf *par0, mpdf *obs0, int n, const BM_T &BMcond0 ) : PF (), jest ( est ) {
		PF::set_parameters(par0,obs0,n);
		jest.set_n(n);
		//
		//TODO test if rv and BMcond.rv are compatible.
//		rv.add ( rvlin );
		//

		if ( n>10000 ) {it_error ( "increase 10000 here!" );}

		for ( int i=0;i<n;i++ ) {
			Bms[i] = new BM_T ( BMcond0 ); //copy constructor
			const epdf& pom=Bms[i]->posterior();
			jest.set_elements ( i,1.0/n,&pom );
		}
	};

	~MPF() {
	}

	void bayes ( const vec &dt );
	const epdf& posterior() const {return jest;}
	const epdf* _e() const {return &jest;} //Fixme: is it useful?
	//! Set postrior of \c rvc to samples from epdf0. Statistics of Bms are not re-computed! Use only for initialization!
	void set_est ( const epdf& epdf0 ) {
		PF::set_est ( epdf0 );  // sample params in condition
		// copy conditions to BMs

		for ( int i=0;i<n;i++ ) {Bms[i]->condition ( _samples ( i ) );}
	}

	//!Access function
	BM* _BM ( int i ) {return Bms[i];}
};

template<class BM_T>
void MPF<BM_T>::bayes ( const vec &dt ) {
	int i;
	vec lls ( n );
	vec llsP ( n );
	ivec ind;
	double mlls=-std::numeric_limits<double>::infinity();

#pragma omp parallel for
	for ( i=0;i<n;i++ ) {
		//generate new samples from paramater evolution model;
		_samples ( i ) = par->samplecond ( _samples ( i ) );
		llsP ( i ) = par->_e()->evallog ( _samples ( i ) );
		Bms[i]->condition ( _samples ( i ) );
		Bms[i]->bayes ( dt );
		lls ( i ) = Bms[i]->_ll(); // lls above is also in proposal her must be lls(i) =, not +=!!
		if ( lls ( i ) >mlls ) mlls=lls ( i ); //find maximum likelihood (for numerical stability)
	}

	double sum_w=0.0;
	// compute weights
#pragma omp parallel for
	for ( i=0;i<n;i++ ) {
		_w ( i ) *= exp ( lls ( i ) - mlls ); // multiply w by likelihood
		sum_w+=_w ( i );
	}

	if ( sum_w  >0.0 ) {
		_w /=sum_w; //?
	}
	else {
		cout<<"sum(w)==0"<<endl;
	}


	double eff = 1.0/ ( _w*_w );
	if ( eff < ( 0.3*n ) ) {
		ind = est.resample();
		// Resample Bms!

#pragma omp parallel for
		for ( i=0;i<n;i++ ) {
			if ( ind ( i ) !=i ) {//replace the current Bm by a new one
				//fixme this would require new assignment operator
				// *Bms[i] = *Bms[ind ( i ) ];

				// poor-man's solution: replicate constructor here
				// copied from MPF::MPF
				delete Bms[i];
				Bms[i] = new BM_T ( *Bms[ind ( i ) ] ); //copy constructor
				const epdf& pom=Bms[i]->posterior();
				jest.set_elements ( i,1.0/n,&pom );
			}
		};
		cout << '.';
	}
}

}
#endif // KF_H


