/*!
  \file
  \brief UI for Common probability density functions
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef DS_UI_H
#define DS_UI_H


#include "libDS.h"
#include "stat/loggers_ui.h"

/*! UI for mgnorm

The mgnorm is constructed from a structure with fields:
\code
system = {
	type = "mgnorm";
	// function for mean value evolution
	g = {type="fnc"; ... }

	// variance
	R = [1, 0,
         0, 1];
	// --OR --
	dR = [1, 1];

	// == OPTIONAL ==

	// description of y variables
	y = {type="rv"; names=["y", "u"];};
	// description of u variable
	u = {type="rv"; names=[];}
};
\endcode

Result if
*/
class UImgnorm : public UIbuilder {
public:
	UImgnorm() :UIbuilder ( "mgnorm" ) {};
	bdmroot* build ( Setting &S ) const {
		mgnorm<chmat>* mgn;
		fnc* g; UIbuild ( S["g"],g );

		mat R;
		if ( S.exists ( "dR" ) ) {R=diag ( getvec ( S["dR"] ) );}
		else {
			if ( S.exists ( "R" ) ) {R=getmat ( S["R"],g->dimension() );}
		}
		
		mgn->set_parameters(g,R);
		return mgn;
	};
};

UIREGISTER ( UImgnorm);

/*! UI for migamma_ref

The migamma_ref is constructed from a structure with fields:
\code
system = {
	type = "migamma_ref";
	ref = [1e-5; 1e-5; 1e-2 1e-3];            // reference vector
	l = 0.999;                                // constant l
	k = 0.1;                                  // constant k
	
	// == OPTIONAL ==
	// description of y variables
	y = {type="rv"; names=["y", "u"];};
	// description of u variable
	u = {type="rv"; names=[];}
};
\endcode

Result if
 */
class UImigamma_ref : public UIbuilder {
	public:
		UImigamma_ref() :UIbuilder ( "migamma_ref" ) {};
		bdmroot* build ( Setting &S ) const {
			migamma_ref* mig=new migamma_ref;

			mig->set_parameters(S["k"],getvec(S["ref"]),S["l"]);
			return mig;
		};
};
UIREGISTER ( UImigamma_ref );


/*! UI for mlognorm

The mlognorm is constructed from a structure with fields:
\code
system = {
	type = "mlognorm";
	k = 0.1;                                  // constant k
	mu0 = [1., 1.];
	
	// == OPTIONAL ==
	// description of y variables
	y = {type="rv"; names=["y", "u"];};
	// description of u variable
	u = {type="rv"; names=[];}
};
\endcode

 */
class UImlognorm : public UIbuilder {
	public:
		UImlognorm() :UIbuilder ( "mlognorm" ) {};
		bdmroot* build ( Setting &S ) const {
			mlognorm* mln=new mlognorm;

			vec mu0=getvec(S["mu0"]);
			mln->set_parameters(mu0.length(),S["k"]);
			mln->condition(mu0);
			return mln;
		};
};
UIREGISTER ( UImlognorm );

#endif // DS_UI_H
