/*
  \file
  \brief Models for synchronous electric drive using IT++ and BDM
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/


#include <estim/libKF.h>
#include <estim/libPF.h>
#include <stat/libFN.h>

#include "pmsm.h"
#include "simulator.h"
#include "sim_profiles.h"

using namespace bdm;
//!Extended Kalman filter with unknown \c Q
class EKF_unQ : public EKFCh , public BMcond {
public:
	//! Default constructor
	void condition ( const vec &Q0 ) {
		Q.setD ( Q0,0 );
		//from EKF
		preA.set_submatrix ( dimy+dimx,dimy,Q._Ch() );
	};
};


int main() {
	// Kalman filter
	int Ndat = 9000;
	double h = 1e-6;
	int Nsimstep = 125;
	int Npart = 200;

	// internal model
	IMpmsm fxu;
	//                  Rs    Ls        dt       Fmag(Ypm)    kp   p    J     Bf(Mz)
	fxu.set_parameters ( 0.28, 0.003465, Nsimstep*h, 0.1989, 1.5 ,4.0, 0.04, 0.0 );
	// observation model
	OMpmsm hxu;

	vec mu0= "0.0 0.0 0.0 0.0";
	vec Qdiag ( "0.001 0.001 1e-6 1e-10" ); //zdenek: 0.01 0.01 0.0001 0.0001
	vec Rdiag ( "1e-10 1e-10" ); //var(diff(xth)) = "0.034 0.034"
	chmat Q ( Qdiag );
	chmat R ( Rdiag );
	EKFCh KFE ;
	KFE.set_parameters ( &fxu,&hxu,Q,R );
	KFE.set_est ( mu0, chmat ( 1*ones ( 4 ) ) );

	RV rQ ( "{Q }","2" );
	EKF_unQ KFEp;
	KFEp.set_parameters ( &fxu,&hxu,Q,R );
	KFEp.set_est ( mu0, chmat ( zeros ( 4 ) ) );

	mgamma evolQ ;
	MPF<EKF_unQ> M (&evolQ,&evolQ,Npart,KFEp );
	// initialize
	evolQ.set_parameters ( 10.0, "0.01 0.01" ); //sigma = 1/10 mu
	evolQ.condition ( "0.01 0.01" ); //Zdenek default
	epdf& pfinit=evolQ._epdf();
	M.set_est ( pfinit );
	evolQ.set_parameters ( 10.0,  "0.01 0.01");

	//

	const epdf& KFEep = KFE.posterior();
	const epdf& Mep = M.posterior();

	mat Xt=zeros ( Ndat ,9 ); //true state from simulator
	mat Dt=zeros ( Ndat,4+2 ); //observation
	mat XtE=zeros ( Ndat, 4 );
	mat XtM=zeros ( Ndat,2+4 ); //Q + x

	// SET SIMULATOR
	pmsmsim_set_parameters ( 0.28,0.003465,0.1989,0.0,4,1.5,0.04, 200., 3e-6, h );
	double Ww=0.0;
	static int k_rampa=1;
	static long k_rampa_tmp=0;
	vec dt ( 2 );
	vec ut ( 2 );
	vec vecW="0.1 0.2 0.4 0.9 0.4 0.2 0.0 -0.4 -0.9 -1.6 -0.4 0.0 0.0";

	for ( int tK=1;tK<Ndat;tK++ ) {
		//Number of steps of a simulator for one step of Kalman
		for ( int ii=0; ii<Nsimstep;ii++ ) {
			//simulator
			sim_profile_vec01t(Ww,vecW);
			pmsmsim_step ( Ww );
		};
		// collect data
		ut ( 0 ) = 0.0;//KalmanObs[0];
		ut ( 1 ) = 0.0;//KalmanObs[1];
		dt ( 0 ) = KalmanObs[2];
		dt ( 1 ) = KalmanObs[3];

		//estimator
		KFE.bayes ( concat ( dt,ut ) );
		M.bayes ( concat ( dt,ut ) );

		Xt.set_row ( tK,vec ( x,9 ) ); //vec from C-array
		Dt.set_row ( tK, concat ( dt,ut,vec_1(sqrt(pow(ut(0),2)+pow(ut(1),2))), vec_1(sqrt(pow(dt(0),2)+pow(dt(1),2))) ) );
		XtE.set_row ( tK,KFEep.mean() );
		XtM.set_row ( tK,Mep.mean() );
	}

	it_file fou ( "pmsm_sim.it" );

	fou << Name ( "xth" ) << Xt;
	fou << Name ( "Dt" ) << Dt;
	fou << Name ( "xthE" ) << XtE;
	fou << Name ( "xthM" ) << XtM;
	//Exit program:

	return 0;
}

