/*!
  \file
  \brief UI for Common DataSources.
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef DS_UI_H
#define DS_UI_H


#include "libDS.h"
#include "stat/loggers_ui.h"

/*! UI for ArxDS using factorized description!

The ArxDS is constructed from a structure with fields:
\code
system = {
	type = "ArxDS";
	// description of y variables
	y = {type="rv"; names=["y", "u"];};
	// description of u variable
	u = {type="rv"; names=[];}
	// description of regressor
	rgr = {type="rv";
		names = ["y","y","y","u"];
		times = [-1, -2, -3, -1];
	}

	// theta
	theta = [0.8, -0.3, 0.4, 1.0,
	         0.0, 0.0, 0.0, 0.0];
	// offset (optional)
	offset = [0.0, 0.0];
	//variance
	r = [0.1, 0.0,
	     0.0, 1.0];
	//options: L_theta = log value of theta,
	opt = "L_theta";
};
\endcode

Result is ARX data source offering with full history as Drv.
*/
class UIArxDS : public UIbuilder {
public:
	UIArxDS() :UIbuilder ( "ArxDS" ) {};
	bdmroot* build ( Setting &S ) const {
		RV *yrv; UIbuild(S["y"],yrv);
		RV *urv; UIbuild(S["u"],urv);
		RV *rrv; UIbuild(S["rgr"],rrv);
		
		ArxDS* tmp = new ArxDS;
		mat Th=getmat ( S["theta"], rrv->_dsize() );
		vec mu0;
		if ( S.exists ( "offset" ) ) {
			mu0=getvec ( S["offset"] );
		}
		else {
			mu0=zeros ( yrv->_dsize() );
		}
		chmat sqR ( getmat ( S["r"],yrv->_dsize() ) );
		tmp->set_parameters ( Th,mu0,sqR );
		tmp->set_drv(*yrv,*urv,*rrv);
		if (S.exists("opt")){tmp->set_options(S["opt"]);}
		return tmp;
	};
};

UIREGISTER ( UIArxDS );

/*! UI for stateDS 

The DS is constructed from a structure with fields:
\code
system = {
	type = "stateDS";
	//Internal model
	IM = { type = "mpdf"; //<-- valid offspring! e.g. "mlnorm"
		rv = { //description of x_t 
			names=["name1",...];
			sizes=[2,1]; // optional default=[1,1...];
			times=[0,0]; // optional default=[0,0...];
			}
		rvu= { //description of  u_t 
			//optional default=empty
			}	

		// remaining fields depending on the chosen type
		};
	//Observation model
	OM = { type = "mpdf-offspring";
		rv = {}; //description of d_t 
		rvu = {type="internal", path="system.IM.rvu"}; //description of u_t 
	
		//remaining fields
	}
};
\endcode

 */
class UIstateDS : public UIbuilder {
	public:
		UIstateDS() :UIbuilder ( "stateDS" ) {};
		bdmroot* build ( Setting &S ) const {
			RV* rvtmp;
			UIbuild(S["IM"]["rvu"], rvtmp);
			mpdf* IM;
			UIbuild(S["IM"],IM);
			mpdf* OM;
			UIbuild(S["OM"],OM);
			stateDS *DS=new stateDS(IM,OM,0);
			//DS->set_drv(rvtmp);
			return DS;
		}
};
UIREGISTER ( UIstateDS );

/*! \brief UI for reading data from files

\code
\endcode
*/

class UIITppFileDS : public UIbuilder {
	public:
		UIITppFileDS() :UIbuilder ( "ITppFileDS" ) {};
		bdmroot* build ( Setting &S ) const {
			RV* rvtmp;
			UIbuild(S["rv"], rvtmp);
			ITppFileDS *DS=new ITppFileDS(S["filename"],S["varname"]);
			DS->set_drv(*rvtmp,RV());
			return DS;
		}
};
UIREGISTER ( UIITppFileDS );

#endif // DS_UI_H
