#ifndef UIBUILD
#define UIBUILD

#include <itpp/itbase.h>
#include "stat/libBM.h"
#include "libconfig/libconfig.h++"

#include <sstream>
#include <iostream>
#include <stdio.h>
#include <string>
#include <typeinfo>
#include <map>
#include <utility>
#include <vector>
#include <iostream>

using std::string;
using namespace std;
using namespace libconfig;

#define UIREGISTER(className) template<> ParticularUI<className>& ParticularUI<className>::ui = ParticularUI<className>(#className)

#define ASSERT_UITYPE(S,Type) it_assert_debug(S.getType()==Setting::Type, string("Wrong setting type, see input path \"")+string(S.getPath())+string("\""))

namespace bdm
{

/*!
@brief This class serves to load and/or save DOMElements into/from files
stored on a hard-disk.

Firstly, you associate new RootElement instance with some filename during a time of its 
construtcion. Then, you save some object into the new RootElement instance,
and save it into the file this way:
\code
	CAudi audi;
	RootElement root("cars.xml");
	UserInfo::Save( audi, root, "TT");
	root.Save();
\endcode

In the other way round, when loading object from a XML file, the appropriate code looks like this:
\code
	RootElement root("cars.xml");
	root.Load();
	UserInfo::Build<T>(root,"TT");
\endcode
*/

class UIFile : public Config
{
private:
	const string fileName;

public:
	//! attach new RootElement instance to a file (typically with an XML extension)
	UIFile( const string &file_name );

	//! loads root element from a file
	void Load();

	//! Save UserInfo to the file (typically with an XML extension)
	void Save ();

	operator Setting&();
};


/*!
@brief UserInfo is an abstract is for internal purposes only. Use CompoundUserInfo<T> or ParticularUI<T> instead.
The raison d'etre of this class is to allow pointers to its templated descendants. 

Also, the main functions of the whole UserInfo library are included within this class, see
static methods 'Build' and 'Save'.


/*!\brief Builds computational object from a UserInfo structure

Return value is a pointer to the created object (memory management issue?)
/

*/
class UI 
{
private:
	//! just a typedef shortuct for a constant pointer to UI
	typedef UI* pUI;

	//! static class encalupsating map of names to related UserInfos
	//! 
	//! The key property of this class is that it initilaized the internal map immediately
	//! when it is used for a first time.
	class StringToUIMap
	{
	private:
		//! Type definition of mapping which transforms type names to the related user infors
		typedef map< const string, pUI > MappedString2UI;

		//! immediately initialized instance of type MappedString2UI
		static MappedString2UI& privateMap();

	public:
		//! add a pair key-userinfo into the internal map
		static void Add( const string &className, pUI pInstance );

		//! search for an userinfo related to the passed key within the internal map
		static pUI Retrieve( const string &className );
	};
				
	//! internal method assembling a typeless instance from components obtained by the 'AssemblyComponentsFromSetting()' method
	virtual bdmroot* New() = 0;
	
	//! type name defined by user
	const string className;

protected:

	//! default constructor 
	UI( const string& className ) : className ( className )
	{	
		StringToUIMap::Add( className, this );
	}

	//! Virtual destructor for future use;
	virtual ~UI(){};

private:

   	//! This methods tries to save an instance of type T (or some of its descendant types)
	//! and build DOM tree accordingly. Then, it creates a new DOMNode named according className
	//! and connecti it to the passed Setting as a new child node.
	template<class T> static void ToSetting( T &instance, Setting &element )
	{
		string &className = ParticularUI<T>::ui.className;
			
		// add attribute "class" 
		Setting &type = root.add( "class", Setting::TypeString );
		type = className;
	
		try
		{
			// instance disassembling 
			instance.ToSetting( root );
		}
		catch(SettingException xcptn)
		{
			it_error ( "UI: the method " + className + ".ToSetting(Setting&) has thrown an exception when filling the setting " + xcptn.getPath() + ". Try to correct this method." );
		}	
	}

	template<class T> static T* FromSetting( const Setting &element )
	{			
		RevealIfLinked revealed_link( element );
		const Setting &root = revealed_link.root();

		ASSERT_UITYPE(root,TypeGroup);

		// we get a velue stored in the "class" attribute 
		string className;
		if( !root.lookupValue( "class", className ) )
		{
			stringstream msg;
			msg << "UI: the class identifier is missing within the setting """ << root.getPath() <<  """. Check line " 
				<< root.getSourceLine() << ".";
			it_error ( msg.str() );
		}

	
		// and finally we find a UserInfo related to this type
		pUI pRelatedUI = StringToUIMap::Retrieve( className );
		if( !pRelatedUI)
			it_error ( "UI: class " + className + " was not properly registered. Use the macro ""UIREGISTER([class name]);"" within your code." );
		
		bdmroot* pTypelessInstance = pRelatedUI->New();

		T* pInstance = NULL;
		try
		{
			pInstance = (T*) pTypelessInstance ;
		}
		catch(...)
		{
			it_error ( "UI: class " + className + " is not a descendant of the desired output class. Try to call the UI::Build function with a different type parameter." );
		}
		
		try
		{
			// instance assembling 
			pInstance->FromSetting( root );
		}
		catch(SettingException xcptn)
		{
			string msg = "UI: the method " + className + ".FromSetting(Setting&) has thrown an exception when parsing the setting " + xcptn.getPath() + ". Try to correct this method.";
			it_error ( msg );
		}
		return pInstance;
	}	



public:

	// vraci true, kdyz to byl platny link, jinak false.. v pripade chyby konci it_errorem..
	// do elementu vrati setting prislusny po rozbaleni linku, jinak ponecha beze zmeny
	class RevealIfLinked	
	{
	private:
		UIFile *file;
		const Setting * result;

	public:

		RevealIfLinked( const Setting &element )
		{
			file = NULL;
			result = &element;

			if( element.getType() !=  Setting::TypeString )
				return;

			string link = (string) element;
			size_t aerobase = link.find('@');
			if( aerobase != string::npos )
			{
				string file_name = link.substr( aerobase + 1, link.length() );
				file = new UIFile( file_name );
				file->Load();
				// TODO OSETRIT FALSE, vyhodit iterr

				result = &(Setting&)(*file);

				link = link.substr( 0, aerobase );
			}
			else
				while ( !result->isRoot() ) 
					result = &result->getParent();

			if( !result->exists( link ) )
			{
				// vyhodi chybu v pripade chyby
				printf("");
			}
			result = &(*result)[link];
			return;
		}

		~RevealIfLinked()
		{
			if( file ) delete file;
		}

		
		const Setting& root()
		{
			return *result;
		}
	};

	private:
		static const Setting* ToChildSetting( const Setting &element, const int index )
		{
			if( !element.isAggregate())
				return NULL;

			if( element.getLength() <= index )
				return NULL;

			return &element[index];
		}

		static const Setting* ToChildSetting( const Setting &element, const string &name )
		{
			if( !element.isGroup() )
				return NULL;

			if( !element.exists( name ) )
				return NULL;

			return &element[name];
		}

		static Setting& ToChildSetting( Setting &element, const int index )
		{
			if( !element.isAggregate())
			{
				// TODO CO ZA TYP? ASI NE GROUP, COZ..
			}
			if( element.getLength() <= index )
			{
				stringstream msg;
				msg << "UI: there is not any child with index " << index << " in the parsed setting " 
					<< element.getPath() <<  ", check line " << element.getSourceLine() << ".";

				it_error ( msg.str() );
			}
			return element[index];
		}

		static Setting& ToChildSetting( Setting &element, const string &name )
		{
			ASSERT_UITYPE(element,TypeGroup);
			if( !element.exists( name ) )
			{
				stringstream msg;
				msg << "UI: there is not any child named """ << name << """ in the parsed setting " 
					<< element.getPath() <<  ", check line " << element.getSourceLine() << ".";

				it_error ( msg.str() );
			}
			return element[name];
		}

	//! This methods tries to build a new double matrix 
	static bool FromSetting( mat& matrix, const Setting &root )
	{
		ASSERT_UITYPE(root,TypeList);
		if( root.getLength() != 3 )
		{
			stringstream msg;
			msg << "UI: the setting " << root.getPath() <<  """ supposed to represent a matrix element has wrong syntax"". Check line " 
				<< root.getSourceLine() << ".";
			it_error ( msg.str() );
		}

		Setting &cols_setting = root[0];
		Setting &rows_setting = root[1];
		Setting &values = root[2];

		ASSERT_UITYPE(cols_setting,TypeInt);
		ASSERT_UITYPE(rows_setting,TypeInt);
		ASSERT_UITYPE(values,TypeArray);

		int cols = cols_setting;
		int rows = rows_setting;
		
		if( values.getLength() != cols * rows )
		{
			stringstream msg;
			msg << "UI: the lenght of array containing matrix values within setting " << root.getPath() <<  """ is improper, check line " 
				<< root.getSourceLine() << ".";
			it_error ( msg.str() );
		}

		matrix.set_size( rows, cols );

		if( cols == 0 || rows == 0 )
			return true;

		if( !values[0].isNumber() )
		{
			stringstream msg;
			msg << "UI: the array containing matrix values within setting " << root.getPath() <<  """ has to contain numeric values only! Check line " 
				<< root.getSourceLine() << ".";
			it_error ( msg.str() );
		}

		// Build matrix row-wise
		int k = 0;
		for( int i=0;i<rows;i++ ) 
			for( int j=0; j<cols; j++)
				matrix(i,j) = values[k++];

		return true;
	}

	//! This methods tries to save a double matrix 
	static void ToSetting( mat &matrix, Setting &root )
	{
		Setting &cols = root.add( Setting::TypeInt );
		cols = matrix.cols();

		Setting &rows = root.add( Setting::TypeInt );
		rows = matrix.rows();

		Setting &values = root.add( Setting::TypeArray );

		// Build matrix row-wise
		for( int i=0; i<matrix.rows(); i++ ) 
			for( int j=0; j<matrix.cols(); j++)
			{
				Setting &newField = values.add(Setting::TypeFloat);
				newField = matrix(i,j);
			}
	}

	//! This methods tries to build a new integer vector
	static bool FromSetting( ivec &vec, const Setting &root )
	{
		ASSERT_UITYPE(root,TypeArray);

		int len = root.getLength();
		vec.set_length( len );
		if( len == 0 ) return true;
		ASSERT_UITYPE(root[0],TypeInt);

		for( int i=0; i < len; i++ ) 
			vec(i) = root[i];
		
		return true;
	}

	//! This methods tries to save an integer vector
	static void ToSetting( ivec &vec, Setting &root )
	{
		for( int i=0; i<vec.length(); i++ ) 
		{
			Setting &newField = root.add(Setting::TypeInt);
			newField = vec(i);
		}
	}

	//! This methods tries to build a new array of strings 
	static bool FromSetting( Array<string> &string_array, const Setting &root)
	{
		ASSERT_UITYPE(root,TypeArray);

		int len = root.getLength();
		string_array.set_length( len );
		if( len == 0 ) return true;
		ASSERT_UITYPE(root[0],TypeString);

		for( int i=0; i < len; i++ ) 
			string_array(i) = (string)root[i];

		return true;
	}

	//! This methods tries to save an array of strings
	static void ToSetting( Array<string> &string_array, Setting &root )
	{
		for( int i=0; i<string_array.length(); i++ ) 
		{
			Setting &newField = root.add(Setting::TypeString);
			newField = string_array(i);
		}
	}

public: 

	//! This methods tries to build a new instance of type T (or some of its descendant types)
	//! according to a data stored in a DOMNode named className within a child nodes of the passed element.
	//! If an error occurs, it returns a NULL pointer.

	//! Prototype of a UI builder. Return value is by the second argument since it type checking via \c dynamic_cast.
	template<class T> static T* Build( Setting &element, const int index )
	{
		return FromSetting<T>( ToChildSetting( element, index ) );
	}

	template<class T> static T* Build( Setting &element, const string &name )
	{			
		return FromSetting<T>( ToChildSetting( element, name ) );
	}

   	//! This methods tries to save an instance of type T (or some of its descendant types)
	//! and build DOM tree accordingly. Then, it creates a new DOMNode named according className
	//! and connecti it to the passed Setting as a new child node.
	template<class T> static void Save( T &instance, Setting &element )
	{
		Setting &root = element.add( Setting::TypeGroup );		
		ToSetting( instance, root );
	}

   	//! This methods tries to save an instance of type T (or some of its descendant types)
	//! and build DOM tree accordingly. Then, it creates a new DOMNode named according className
	//! and connecti it to the passed Setting as a new child node.
	template<class T> static void Save( T &instance, Setting &element, const string &name )
	{
		Setting &root = element.add( name, Setting::TypeGroup );		
		ToSetting( instance, root );
	}

	//! This methods tries to build a new double matrix 
	static bool Get( mat& matrix, const Setting &element, const string &name )
	{
		const Setting *root = ToChildSetting( element, name );
		if( !root ) return false;				
		return FromSetting( matrix, *root );
	}

	//! This methods tries to build a new double matrix 
	static bool Get( mat& matrix, const Setting &element, const int index )
	{
		const Setting *root = ToChildSetting( element, index );
		if( !root ) return false;				
		return FromSetting( matrix, *root );
	}

	//! This methods tries to save a double matrix 
	static void Save( mat &matrix, Setting &element, const string &name )
	{
		Setting &root = element.add( name, Setting::TypeList );		
		ToSetting( matrix, root );
	}

	//! This methods tries to save a double matrix 
	static void Save( mat &matrix, Setting &element )
	{
		Setting &root = element.add( Setting::TypeList );		
		ToSetting( matrix, root );
	}


	//! This methods tries to build a new double vec 
	static bool Get( ivec& vec, const Setting &element, const string &name )
	{
		const Setting *root = ToChildSetting( element, name );
		if( !root ) return false;				
		return FromSetting( vec, *root );
	}

	//! This methods tries to build a new double vec 
	static bool Get( ivec& vec, const Setting &element, const int index )
	{
		const Setting *root = ToChildSetting( element, index );
		if( !root ) return false;				
		return FromSetting( vec, *root );
	}

	//! This methods tries to save a double vec 
	static void Save( ivec &vec, Setting &element, const string &name )
	{
		Setting &root = element.add( name, Setting::TypeArray );		
		ToSetting( vec, root );
	}

	//! This methods tries to save a double vec 
	static void Save( ivec &vec, Setting &element)
	{
		Setting &root = element.add( Setting::TypeArray );		
		ToSetting( vec, root );
	}


	//! This methods tries to build a new double string_array 
	static bool Get( Array<string> &string_array, const Setting &element, const string &name )
	{
		const Setting *root = ToChildSetting( element, name );
		if( !root ) return false;				
		return FromSetting( string_array, *root );
	}

	//! This methods tries to build a new double string_array 
	static bool Get( Array<string> &string_array, const Setting &element, const int index )
	{
		const Setting *root = ToChildSetting( element, index );
		if( !root ) return false;				
		return FromSetting( string_array, *root );
	}

	//! This methods tries to save a double string_array 
	static void Save( Array<string> &string_array, Setting &element, const string &name )
	{
		Setting &root = element.add( name, Setting::TypeArray );		
		ToSetting( string_array, root );
	}

	//! This methods tries to save a double string_array 
	static void Save( Array<string> &string_array, Setting &element )
	{
		Setting &root = element.add( Setting::TypeArray );		
		ToSetting( string_array, root );
	}


};


/*!
@brief The main userinfo template class. You should derive this class whenever you need 
a new userinfo of a class which is compound from smaller elements (all having its
own userinfo class prepared).
*/
template<typename T> class ParticularUI : private UI
{
	// to permit acces to the ParticularUI<T>::ui to the UI class
	friend UI;

	//! default constructor, which is intentionally declared as private
	ParticularUI<T>( const string &className) : UI( className ) 
	{	
	};

	//! the only instance of this class (each type T has its own instance)
	//! which is used as a factory for processing related UI
	static ParticularUI<T>& ui;	

	bdmroot* New()
	{
		return new T();
	}
};


}

/*! Recursive build of objects defined in the same file

\code 
{type="internal";
path="system.profile.[0]";    // Path from the root 
};
\endcode
 */



/*! Recursive build of objects defined in external file

\code 
{type="external";
filename="my_file.cfg";       // name of file from which to read
path="system.profile.[0]";    // Path in the external file
};
\endcode
/


//! [Debugging] Print values in current S to cout
void UI_DBG ( Setting &S, const string &spc );


/*
//! Auxiliary function allowing recursivity in S (too complex, remove?)
template<class T>
void UIcall ( Setting &S, void ( *func ) ( Setting&, T ), T Tmp ) {
	ASSERT_UITYPE ( S,TypeGroup );
	// Check if field "class" is present, if not it is not a valid UI
	it_assert_debug ( S.exists ( "class" ), string ( S.getPath() ) +" is not a valid UI!" );

	const string typ=S["class"];
	if ( typ=="internal" ) {
		try {
			Setting* Stmp = &S;
			do {Stmp=& ( Stmp->getParent() );}
			while ( !Stmp->isRoot() );
			Setting& intS=Stmp->lookup ( ( const char* ) S["path"] );
			func ( intS, Tmp ); // <======== calling func
			return;
		}
		catch ( ... ) {
			it_error ( "Internal field " + string ( S.getPath() ) + " not valid" );
		}
	}
	if ( typ=="external" ) {
		UIFile C ( S["filename"] );
		try {
			func ( C.lookup ( ( const char* ) S["path"] ), Tmp );
		}
		catch ( ... ) {
			it_error ( "External field " + string ( S.getPath() ) + " not valid" );
		}
		return;
	}

	// v======================= calling final func
	func ( S, Tmp );
};

}

*/

#endif // #ifndef UIBUILD