#ifndef USER_INFO_H
#define USER_INFO_H

#include <stdio.h>
#include <string>
#include <typeinfo>
#include <map>

#include "libconfig/libconfig.h++"
#include "../bdmroot.h"
#include "itpp/itbase.h"

#include <stdexcept>

using std::string;
using namespace std;
using namespace libconfig;

namespace bdm
{
	
/*!
  \def UIREGISTER(class_name)
  Macro for registration of class into map of UserInfos -- registered class is scriptable 

  TODO napsat i to, ze UIREG musi byt v hacku..
*/
#ifndef BDMLIB 
#define UIREGISTER(class_name) template<> const ParticularUI<class_name>& ParticularUI<class_name>::ui = ParticularUI<class_name>(#class_name) 
#else
#define UIREGISTER(class_name)
#endif

//!  macro assertting that the setting SET is of the SettingType TYPE
#define ASSERT_UITYPE(SET,TYPE) it_assert_debug(SET.getType()==Setting::TYPE, string("Wrong setting type, see input path \"")+string(SET.getPath())+string("\""))

//! exception used in UI::build if it fails it can be caught and handled - see merger_mex.h
class UIbuildException : public std::invalid_argument {
	public:
		UIbuildException() : std::invalid_argument("class name") { }
};

/*!
@brief This class serves to load and/or save user-infos into/from 
configuration files stored on a hard-disk.

Firstly, save some user-infos into the new UIFile instance. Then, 
call the save method with a filename as its only argument:

\code
	CAudi audi;
	UIFile file;
	UI::save( audi, file, "TT");
	file.save("cars.cfg");
\endcode

In the other way round, when loading object from a configuration file, 
the appropriate code looks like this:

\code
	UIFile file("cars.cfg");
	CAudi *audi = UI::build<CAudi>(file,"TT");
\endcode
*/
class UIFile : public Config
{
public:
	//! create empty file instance prepared to store Settings
	UIFile();

	//! creates instance and fills it from the configuration file file_name
	UIFile( const string &file_name );

	//! save all the stored Settings into the configuration file file_name
	void save(const string &file_name);

	//! this operator allows the ability of substituting Setting parameter by UIFile instance
	operator Setting&();
};



/*!
@brief UI is an abstract class and it is intended for internal purposes only

This class exists mainly to allow pointers to its templated descendant ParticularUI<T>. Next, 
it collects all the auxiliary functions useful to prepare some concret user-infos, see static 
methods 'build', 'get' and 'save'. 
*/
class UI 
{
private:
	//! Atatic class encalupsating two maps, one mapping names to UI instances and the other mapping type_infos to class names
	//! 
	//! The key property of this class is that it initilaize the internal map immediately
	//! when it is used for a first time. Therefore, we do not have to care about the 
	//! order of calls to UIREGISTER macro, which operates with both these mappings.
	class MappedUI
	{
	private:
		//! Type definition of mapping which transforms class names to the related UI instances
		typedef map< const string, const UI* const > StringToUIMap;

		//! Type definition of mapping which transforms RTTI type_infos to the related class names
		typedef map< const type_info * const, const string > TypeInfoToStringMap;

		//! immediately initialized instance of type StringToUIMap
		static StringToUIMap& mapped_strings();

		//! immediately initialized instance of type TypeInfoToStringMap
		static TypeInfoToStringMap& mapped_type_infos();

		//! method for reporting a error when an attempt to operate with an unregistered class occures
		static void unregistered_class_error( const string &unregistered_class_name );

	public:
		//! add a pair key-userinfo into the internal map
		static void add_class( const string &class_name, const type_info * const class_type_info, const UI* const ui );

		//! search for an userinfo related to the passed class name within the internal map
		static const UI& retrieve_ui( const string &class_name );

		//! search for an class name related to the passed type_info within the internal map
		static const string& retrieve_class_name( const type_info* const class_type_info );
	};

	//! Method assembling a typeless instance, it is implemented in descendant class ParticularUI<T>
	virtual root* new_instance() const = 0;
	
   	//! Method switching from the \a element to its child Setting according the passed \a index, it also does all the necessary error-checking 
	static const Setting& to_child_setting( const Setting &element, const int index );

   	//! Method switching from the \a element to its child Setting according the passed \a name, it also does all the necessary error-checking 
	static const Setting& to_child_setting( const Setting &element, const string &name );

	//! This methods converts a Setting into a matrix 
	static void from_setting( mat& matrix, const Setting &element );	
	//! This methods converts a Setting into an integer vector
	static void from_setting( ivec &vector, const Setting &element );
	//! This methods converts a Setting into a string
	static void from_setting( string &str, const Setting &element );
	//! This methods converts a Setting into a real vector
	static void from_setting( vec &vector, const Setting &element );
	//! This methods converts a Setting into a integer scalar 
	static void from_setting( int &integer, const Setting &element );
	//! This methods converts a Setting into a real scalar 
	static void from_setting( double &real, const Setting &element );
	//! This methods converts a Setting into a class T descendant
	template<class T> static void from_setting( T* &instance, const Setting &element )
	{			
		const SettingResolver link( element );

		ASSERT_UITYPE(link.result,TypeGroup);

		// we get a velue stored in the "class" attribute 
		string class_name;
		if( !link.result.lookupValue( "class", class_name ) )
			ui_error( "the obligatory ""class"" identifier is missing", link.result );
	
		// then we find a user-info related to this type
		const UI& related_UI = MappedUI::retrieve_ui( class_name );
		
		root* typeless_instance = related_UI.new_instance();

		instance = NULL;
		//try catch does not work!!!
		instance = dynamic_cast<T*>(typeless_instance);
		if (!instance){
			throw UIbuildException();
		}
// 		catch(...)
// 		{
			// TODO pouzit ui_error? 
// 			it_error ( "UI error: class " + class_name + " is not a descendant of the desired output class. Try to call the UI::build function with a different type parameter." );
// 		}
// 		
		try
		{
			instance->from_setting( link.result );
		}
		catch(SettingException xcptn)
		{
			// TODO pouzit ui_error? 
			it_error ( "UI error: the method " + class_name + ".from_setting(Setting&) has thrown an exception when parsing the setting " + xcptn.getPath() + ". Try to correct this method." );
		}
	}	

	//! This methods converts a Setting into a new templated array , 
	// TODO efektivne jen pro vect, mat a string, pro dalsi je nutne pridat from_setting metodu.. ale to asi necceme
	template<class T> static void from_setting( Array<T> &array_to_load, const Setting &element )
	{
		const SettingResolver link( element );

		ASSERT_UITYPE(link.result,TypeList);

		int len = link.result.getLength();
		array_to_load.set_length( len );
		if( len == 0 ) return;
		
		for( int i=0; i < len; i++ ) 
			from_setting( array_to_load(i), link.result[i] ); 
	}

	//! Method for reporting user-info errors related to some concrete Setting
	static void ui_error( string message, const Setting &element );

protected:
	//! Default constructor for internal use only, see \sa ParticularUI<T>
	UI( const string& class_name, const type_info * const class_type_info ) 
	{	
		MappedUI::add_class( class_name, class_type_info, this );
	}

public: 

	/*!
	@brief This class serves to expand links used in configuration file. 

	TODO - napsat co dela, a hlavne proc je to takhle implementovany.. otevreny soubor!

	ABSOLUTE PATH.. 

	Firstly, save some user-infos into the new UIFile instance. Then, 
	call the save method with a filename as its only argument:

	\code
		CAudi audi;
		UIFile file;
		UI::save( audi, file, "TT");
		file.save("cars.cfg");
	\endcode

	*/
	class SettingResolver	
	{
	private:
		//! If necessary, this pointer stores an addres of an opened UIFile, else it equals NULL
		UIFile *file;

		//! This method initialize result reference, i.e., it executes the main code of SettingResolver class
		//!
		//! This code could be also located directly in constructor. The only reason why we made this 
		//! method is the keyword 'const' within the declaration of result reference (TODO funguje odkaz?). Such a reference
		//! have to be intialized before any other constructor command, exactly in the way it is implemented now.
		const Setting &initialize_reference( UIFile* &file, const Setting &potential_link);

	public:
		//! Reference to a resolved link or to the original Setting in the case it does not contain a link
		const Setting &result;

		//! If potential_link contains a link to some other setting, it is resolved here. Anyway, the Setting reference result is prepared for use.
		SettingResolver( const Setting &potential_link );
		
		//! An opened UIFile file is closed here if necessary.
		~SettingResolver();		
	};

	//TODO
	//! \name Initialization of bdm::root descendant classes according the values stored in a Setting variable
	//!@{
	//! Return value is by the second argument since it type checking via \c dynamic_cast.
	template<class T> static T* build( const Setting &element, const int index )
	{
		T* instance;
		from_setting<T>( instance, to_child_setting( element, index ) );
		return instance;
	}
	//! VS: addition for root elements
	template<class T> static T* build( const Setting &element )
	{
		T* instance;
		from_setting<T>( instance,  element );
		return instance;
	}

	template<class T> static T* build( const Setting &element, const string &name )
	{			
		T* instance;
		from_setting<T>( instance, to_child_setting( element, name ) );
		return instance;
	}
	//!@}

	//! \name Initialization of structures according the values stored in a Setting variable - TODO VYCET?!
	//!@{
	//! This methods tries to build a new double matrix 
	template<class T> static void get( T &instance, const Setting &element, const string &name )
	{
		from_setting( instance, to_child_setting( element, name ) );
	}

	//! This methods tries to build a new double matrix 
	template<class T> static void get( T &instance, const Setting &element, const int index )
	{
		from_setting( instance, to_child_setting( element, index ) );
	}

	//! This methods tries to build a new double matrix 
	template<class T> static void get( T &instance, const Setting &element  )
	{
		from_setting( instance, element );
	}

	//! This methods tries to build a new double matrix 
	template<class T> static void get( Array<T> &array_to_load, const Setting &element, const string &name )
	{
		from_setting( array_to_load, to_child_setting( element, name ) );
	}

	//! This methods tries to build a new double matrix 
	template<class T> static void get( Array<T> &array_to_load, const Setting &element, const int index )
	{
		from_setting( array_to_load, to_child_setting( element, index ) );
	}

	//! This methods tries to build a new double matrix 
	template<class T> static void get( Array<T> &array_to_load, const Setting &element  )
	{
		from_setting( array_to_load, element );
	}
	//!@}

	template< class T> static void save( const T * const instance, Setting &element, const string &name = "")
	{
		Setting &set = (name == "") ? element.add( Setting::TypeGroup )							
									 : element.add( name, Setting::TypeGroup );		

		const string &class_name = MappedUI::retrieve_class_name( &typeid(*instance) );
			
		// add attribute "class" 
		Setting &type = set.add( "class", Setting::TypeString );
		type = class_name;

		try
		{
			instance->to_setting( set );
		}
		catch(SettingException xcptn)
		{
			it_error ( "UI error: the method " + class_name + ".to_setting(Setting&) has thrown an exception when filling the setting " + xcptn.getPath() + ". Try to correct this method." );
		}	
	}

	//! This methods tries to save a double vec 
	template<class T> static void save( const Array<T> &array_to_save, Setting &element, const string &name = "" )
	{
		ASSERT_UITYPE(element,TypeGroup);
		Setting &list = (name == "") ? element.add( Setting::TypeList )							
									 : element.add( name, Setting::TypeList );		
		for( int i=0; i<array_to_save.length(); i++ ) 
			save( array_to_save(i), list );
	}


	//! This methods tries to save a double matrix 
	static void save( const mat &matrix, Setting &element, const string &name = "" );

	//! This methods tries to save a double vec 
	static void save( const ivec &vec, Setting &element, const string &name = "" );
	
	static void save( const vec &vector, Setting &element, const string &name = "" );
	//! This methods tries to save a double vec 
	static void save( const string &str, Setting &element, const string &name = "" );

	static void save( const int &integer, Setting &element, const string &name = "" );

	static void save( const double &real, Setting &element, const string &name = "" );	

};


/*!
@brief The main userinfo template class. You should derive this class whenever you need 
a new userinfo of a class which is compound from smaller elements (all having its
own userinfo class prepared).
*/
template<typename T> class ParticularUI : private UI
{
	public:

	//! default constructor, which is intentionally declared as private
	ParticularUI<T>( const string &class_name) : UI( class_name, &typeid(T) ) 
	{	cout << class_name << endl;
	};

	//! the only instance of this class (each type T has its own instance)
	//! which is used as a factory for processing related UI
	static const ParticularUI<T>& ui;	

	root* new_instance() const
	{
		return new T();
	}
};




}

/*! Recursive build of objects defined in the same file

\code 
{type="internal";
path="system.profile.[0]";    // Path from the root 
};
\endcode
 */



/*! Recursive build of objects defined in external file

\code 
{type="external";
filename="my_file.cfg";       // name of file from which to read
path="system.profile.[0]";    // Path in the external file
};
\endcode
/

*/

#endif // #ifndef USER_INFO_H
