/*
 * ACML version 3.6.0 Copyright AMD,NAG 2003,2004,2005
 */

/*
  This is the ACML header file. It contains function prototypes
  to allow a C programmer to call ACML routines via their C
  or Fortran interfaces.

  C interfaces to ACML routines differ from FORTRAN interfaces in
  the following major respects:

    (i) The FORTRAN interface names are appended by an underscore.

   (ii) The C interfaces contain no workspace arguments; all
        workspace memory is allocated internally.

  (iii) Scalar input arguments are passed by value in C interfaces.
        FORTRAN interfaces pass all arguments (except for character
        string "length" arguments that are normally hidden from
        FORTRAN programmers) by reference.

   (iv) Most arguments that are passed as character string pointers
        to FORTRAN interfaces are passed by value as single
        characters to C interfaces. The character string "length"
        arguments of FORTRAN interfaces are not required in the
        C interfaces.

  It is important to note that in both the FORTRAN and C interfaces,
  2-dimensional arrays are assumed to be stored in column-major order.
  e.g. the matrix A = [ 1.0 2.0 ]
                      [ 3.0 4.0 ]
  would be stored in memory as 1.0, 3.0, 2.0, 4.0. This storage order
  corresponds to a FORTRAN-style 2-D array declaration A(2,2), but not
  to an array declared as a[2][2] in C which would be stored in
  row-major order as 1.0, 2.0, 3.0, 4.0.

  As an example, compare the FORTRAN and C interfaces of LAPACK
  routine dsytrf as implemented in ACML.
  FORTRAN:
    void dsytrf_(char *uplo, int *n, double *a, int *lda, int *ipiv,
                 double *work, int *lwork, int *info, int uplo_len);

  C:
    void dsytrf(char uplo, int n, double *a, int lda, int *ipiv, int *info);

  C code calling both the above variants might look like this:

    double *a;
    int *ipiv;
    double *work;
    int n, lda, lwork, info;

    //
    // Assume that all arrays and variables are allocated and initialized
    // as required by dsytrf.
    //
    // Call the FORTRAN version of dsytrf. The first argument is a character
    // string, and the last argument is the length of that string. The input
    // scalar arguments n, lda and lwork, as well as the output scalar
    // argument info, are all passed by reference.
    dsytrf_("Upper", &n, a, &lda, ipiv, work, &lwork, &info, 5);

    //
    // Call the C version of dsytrf. The first argument is a character,
    // workspace is not required, and input scalar arguments n and lda are
    // passed by value. Output scalar argument info is passed by reference.
    dsytrf('U', n, a, lda, ipiv, &info);

*/

#ifndef _ACML_H
#define _ACML_H

/* Under Windows math.h defines "complex" to mean "_complex". */
#include <math.h>
#undef complex

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* ------------------------------------------------------------------ */

/* A complex datatype for use by the C interfaces to ACML routines */
#ifndef _ACML_COMPLEX
#define _ACML_COMPLEX
typedef struct
{
  float real, imag;
} complex;
typedef struct
{
  double real, imag;
} doublecomplex;
#endif /* !defined(_ACML_COMPLEX) */

/*
   These typedefs are for routines that are arguments to other routines,
   e.g. ACML_CGEES_SELECT describes the argument "select" of routine cgees.
 */
typedef int (* ACML_CGEES_SELECT)(complex *);
typedef int (* ACML_CGEESX_SELECT)(complex *);
typedef int (* ACML_CGGES_SELCTG)(complex *, complex *);
typedef int (* ACML_CGGESX_SELCTG)(complex *, complex *);
typedef int (* ACML_DGEES_SELECT)(double *, double *);
typedef int (* ACML_DGEESX_SELECT)(double *, double *);
typedef int (* ACML_DGGES_DELCTG)(double *, double *, double *);
typedef int (* ACML_DGGESX_DELCTG)(double *, double *, double *);
typedef int (* ACML_SGEES_SELECT)(float *, float *);
typedef int (* ACML_SGEESX_SELECT)(float *, float *);
typedef int (* ACML_SGGES_SELCTG)(float *, float *, float *);
typedef int (* ACML_SGGESX_SELCTG)(float *, float *, float *);
typedef int (* ACML_ZGEES_SELECT)(doublecomplex *);
typedef int (* ACML_ZGEESX_SELECT)(doublecomplex *);
typedef int (* ACML_ZGGES_DELCTG)(doublecomplex *, doublecomplex *);
typedef int (* ACML_ZGGESX_DELCTG)(doublecomplex *, doublecomplex *);
typedef void (* ACML_DRANDINITIALIZEUSER_UINI)(int *, int *, int *, int *, int *, int *, int *);
typedef void (* ACML_DRANDINITIALIZEUSER_UGEN)(int *, int *, double *, int *);
typedef void (* ACML_SRANDINITIALIZEUSER_UINI)(int *, int *, int *, int *, int *, int *, int *);
typedef void (* ACML_SRANDINITIALIZEUSER_UGEN)(int *, int *, float *, int *);

/* ------------------------------------------------------------------ */
/* Below are prototypes for the C interfaces to ACML routines */

/* BLAS C interfaces */
extern void caxpy(int n, complex *alpha, complex *x, int incx, complex *y, int incy);
extern void ccopy(int n, complex *x, int incx, complex *y, int incy);
extern complex cdotc(int n, complex *x, int incx, complex *y, int incy);
extern complex cdotu(int n, complex *x, int incx, complex *y, int incy);
extern void cgbmv(char transa, int m, int n, int nsub, int nsuper, complex *alpha, complex *a, int lda, complex *x, int incx, complex *beta, complex *y, int incy);
extern void cgemm(char transa, char transb, int m, int n, int k, complex *alpha, complex *a, int lda, complex *b, int ldb, complex *beta, complex *c, int ldc);
extern void cgemv(char transa, int m, int n, complex *alpha, complex *a, int lda, complex *x, int incx, complex *beta, complex *y, int incy);
extern void cgerc(int m, int n, complex *alpha, complex *x, int incx, complex *y, int incy, complex *a, int lda);
extern void cgeru(int m, int n, complex *alpha, complex *x, int incx, complex *y, int incy, complex *a, int lda);
extern void chbmv(char uplo, int n, int ndiag, complex *alpha, complex *a, int lda, complex *x, int incx, complex *beta, complex *y, int incy);
extern void chemm(char side, char uplo, int m, int n, complex *alpha, complex *a, int lda, complex *b, int ldb, complex *beta, complex *c, int ldc);
extern void chemv(char uplo, int n, complex *alpha, complex *a, int lda, complex *x, int incx, complex *beta, complex *y, int incy);
extern void cher(char uplo, int n, float alpha, complex *x, int incx, complex *a, int lda);
extern void cher2(char uplo, int n, complex *alpha, complex *x, int incx, complex *y, int incy, complex *a, int lda);
extern void cher2k(char uplo, char transa, int n, int k, complex *alpha, complex *a, int lda, complex *b, int ldb, float beta, complex *c, int ldc);
extern void cherk(char uplo, char transa, int n, int k, float alpha, complex *a, int lda, float beta, complex *c, int ldc);
extern void chpmv(char uplo, int n, complex *alpha, complex *a, complex *x, int incx, complex *beta, complex *y, int incy);
extern void chpr(char uplo, int n, float alpha, complex *x, int incx, complex *a);
extern void chpr2(char uplo, int n, complex *alpha, complex *x, int incx, complex *y, int incy, complex *a);
extern void crotg(complex *a, complex *b, float *c, complex *s);
extern void cscal(int n, complex *alpha, complex *y, int incy);
extern void csrot(int n, complex *x, int incx, complex *y, int incy, float c, float s);
extern void csscal(int n, float alpha, complex *y, int incy);
extern void cswap(int n, complex *x, int incx, complex *y, int incy);
extern void csymm(char side, char uplo, int m, int n, complex *alpha, complex *a, int lda, complex *b, int ldb, complex *beta, complex *c, int ldc);
extern void csyr2k(char uplo, char transa, int n, int k, complex *alpha, complex *a, int lda, complex *b, int ldb, complex *beta, complex *c, int ldc);
extern void csyrk(char uplo, char transa, int n, int k, complex *alpha, complex *a, int lda, complex *beta, complex *c, int ldc);
extern void ctbmv(char uplo, char transa, char diag, int n, int ndiag, complex *a, int lda, complex *y, int incy);
extern void ctbsv(char uplo, char transa, char diag, int n, int ndiag, complex *a, int lda, complex *y, int incy);
extern void ctpmv(char uplo, char transa, char diag, int n, complex *a, complex *y, int incy);
extern void ctpsv(char uplo, char transa, char diag, int n, complex *a, complex *y, int incy);
extern void ctrmm(char side, char uplo, char transa, char diag, int m, int n, complex *alpha, complex *a, int lda, complex *b, int ldb);
extern void ctrmv(char uplo, char transa, char diag, int n, complex *a, int lda, complex *y, int incy);
extern void ctrsm(char side, char uplo, char transa, char diag, int m, int n, complex *alpha, complex *a, int lda, complex *b, int ldb);
extern void ctrsv(char uplo, char transa, char diag, int n, complex *a, int lda, complex *y, int incy);
extern double dasum(int n, double *x, int incx);
extern void daxpy(int n, double alpha, double *x, int incx, double *y, int incy);
extern void dcopy(int n, double *x, int incx, double *y, int incy);
extern double ddot(int n, double *x, int incx, double *y, int incy);
extern void dgbmv(char transa, int m, int n, int nsub, int nsuper, double alpha, double *a, int lda, double *x, int incx, double beta, double *y, int incy);
extern void dgemm(char transa, char transb, int m, int n, int k, double alpha, double *a, int lda, double *b, int ldb, double beta, double *c, int ldc);
extern void dgemv(char transa, int m, int n, double alpha, double *a, int lda, double *x, int incx, double beta, double *y, int incy);
extern void dger(int m, int n, double alpha, double *x, int incx, double *y, int incy, double *a, int lda);
extern double dnrm2(int n, double *x, int incx);
extern void drot(int n, double *x, int incx, double *y, int incy, double c, double s);
extern void drotg(double *a, double *b, double *c, double *s);
extern void drotm(int n, double *x, int incx, double *y, int incy, double *param);
extern void drotmg(double d1, double d2, double b1, double b2, double *param);
extern void dsbmv(char uplo, int n, int ndiag, double alpha, double *a, int lda, double *x, int incx, double beta, double *y, int incy);
extern void dscal(int n, double alpha, double *y, int incy);
extern double dsdot(int n, float *x, int incx, float *y, int incy);
extern void dspmv(char uplo, int n, double alpha, double *a, double *x, int incx, double beta, double *y, int incy);
extern void dspr(char uplo, int n, double alpha, double *x, int incx, double *a);
extern void dspr2(char uplo, int n, double alpha, double *x, int incx, double *y, int incy, double *a);
extern void dswap(int n, double *x, int incx, double *y, int incy);
extern void dsymm(char side, char uplo, int m, int n, double alpha, double *a, int lda, double *b, int ldb, double beta, double *c, int ldc);
extern void dsymv(char uplo, int n, double alpha, double *a, int lda, double *x, int incx, double beta, double *y, int incy);
extern void dsyr(char uplo, int n, double alpha, double *x, int incx, double *a, int lda);
extern void dsyr2(char uplo, int n, double alpha, double *x, int incx, double *y, int incy, double *a, int lda);
extern void dsyr2k(char uplo, char transa, int n, int k, double alpha, double *a, int lda, double *b, int ldb, double beta, double *c, int ldc);
extern void dsyrk(char uplo, char transa, int n, int k, double alpha, double *a, int lda, double beta, double *c, int ldc);
extern void dtbmv(char uplo, char transa, char diag, int n, int ndiag, double *a, int lda, double *y, int incy);
extern void dtbsv(char uplo, char transa, char diag, int n, int ndiag, double *a, int lda, double *y, int incy);
extern void dtpmv(char uplo, char transa, char diag, int n, double *a, double *y, int incy);
extern void dtpsv(char uplo, char transa, char diag, int n, double *a, double *y, int incy);
extern void dtrmm(char side, char uplo, char transa, char diag, int m, int n, double alpha, double *a, int lda, double *b, int ldb);
extern void dtrmv(char uplo, char transa, char diag, int n, double *a, int lda, double *y, int incy);
extern void dtrsm(char side, char uplo, char transa, char diag, int m, int n, double alpha, double *a, int lda, double *b, int ldb);
extern void dtrsv(char uplo, char transa, char diag, int n, double *a, int lda, double *y, int incy);
extern double dzasum(int n, doublecomplex *x, int incx);
extern double dznrm2(int n, doublecomplex *x, int incx);
extern int icamax(int n, complex *x, int incx);
extern int idamax(int n, double *x, int incx);
extern int isamax(int n, float *x, int incx);
extern int izamax(int n, doublecomplex *x, int incx);
extern int lsame(char ca, char cb);
extern float sasum(int n, float *x, int incx);
extern void saxpy(int n, float alpha, float *x, int incx, float *y, int incy);
extern float scasum(int n, complex *x, int incx);
extern float scnrm2(int n, complex *x, int incx);
extern void scopy(int n, float *x, int incx, float *y, int incy);
extern float sdot(int n, float *x, int incx, float *y, int incy);
extern float sdsdot(int n, float sb, float *sx, int incx, float *sy, int incy);
extern void sgbmv(char transa, int m, int n, int nsub, int nsuper, float alpha, float *a, int lda, float *x, int incx, float beta, float *y, int incy);
extern void sgemm(char transa, char transb, int m, int n, int k, float alpha, float *a, int lda, float *b, int ldb, float beta, float *c, int ldc);
extern void sgemv(char transa, int m, int n, float alpha, float *a, int lda, float *x, int incx, float beta, float *y, int incy);
extern void sger(int m, int n, float alpha, float *x, int incx, float *y, int incy, float *a, int lda);
extern float snrm2(int n, float *x, int incx);
extern void srot(int n, float *x, int incx, float *y, int incy, float c, float s);
extern void srotg(float *a, float *b, float *c, float *s);
extern void srotm(int n, float *x, int incx, float *y, int incy, float *param);
extern void srotmg(float d1, float d2, float b1, float b2, float *param);
extern void ssbmv(char uplo, int n, int ndiag, float alpha, float *a, int lda, float *x, int incx, float beta, float *y, int incy);
extern void sscal(int n, float alpha, float *x, int incx);
extern void sspmv(char uplo, int n, float alpha, float *a, float *x, int incx, float beta, float *y, int incy);
extern void sspr(char uplo, int n, float alpha, float *x, int incx, float *a);
extern void sspr2(char uplo, int n, float alpha, float *x, int incx, float *y, int incy, float *a);
extern void sswap(int n, float *x, int incx, float *y, int incy);
extern void ssymm(char side, char uplo, int m, int n, float alpha, float *a, int lda, float *b, int ldb, float beta, float *c, int ldc);
extern void ssymv(char uplo, int n, float alpha, float *a, int lda, float *x, int incx, float beta, float *y, int incy);
extern void ssyr(char uplo, int n, float alpha, float *x, int incx, float *a, int lda);
extern void ssyr2(char uplo, int n, float alpha, float *x, int incx, float *y, int incy, float *a, int lda);
extern void ssyr2k(char uplo, char transa, int n, int k, float alpha, float *a, int lda, float *b, int ldb, float beta, float *c, int ldc);
extern void ssyrk(char uplo, char transa, int n, int k, float alpha, float *a, int lda, float beta, float *c, int ldc);
extern void stbmv(char uplo, char transa, char diag, int n, int ndiag, float *a, int lda, float *y, int incy);
extern void stbsv(char uplo, char transa, char diag, int n, int ndiag, float *a, int lda, float *y, int incy);
extern void stpmv(char uplo, char transa, char diag, int n, float *a, float *y, int incy);
extern void stpsv(char uplo, char transa, char diag, int n, float *a, float *y, int incy);
extern void strmm(char side, char uplo, char transa, char diag, int m, int n, float alpha, float *a, int lda, float *b, int ldb);
extern void strmv(char uplo, char transa, char diag, int n, float *a, int lda, float *y, int incy);
extern void strsm(char side, char uplo, char transa, char diag, int m, int n, float alpha, float *a, int lda, float *b, int ldb);
extern void strsv(char uplo, char transa, char diag, int n, float *a, int lda, float *y, int incy);
extern void xerbla(const char *srname, int *info);
extern void zaxpy(int n, doublecomplex *alpha, doublecomplex *x, int incx, doublecomplex *y, int incy);
extern void zcopy(int n, doublecomplex *x, int incx, doublecomplex *y, int incy);
extern doublecomplex zdotc(int n, doublecomplex *x, int incx, doublecomplex *y, int incy);
extern doublecomplex zdotu(int n, doublecomplex *x, int incx, doublecomplex *y, int incy);
extern void zdrot(int n, doublecomplex *cx, int incx, doublecomplex *cy, int incy, double c, double s);
extern void zdscal(int n, double alpha, doublecomplex *y, int incy);
extern void zgbmv(char transa, int m, int n, int nsub, int nsuper, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *x, int incx, doublecomplex *beta, doublecomplex *y, int incy);
extern void zgemm(char transa, char transb, int m, int n, int k, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *beta, doublecomplex *c, int ldc);
extern void zgemv(char transa, int m, int n, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *x, int incx, doublecomplex *beta, doublecomplex *y, int incy);
extern void zgerc(int m, int n, doublecomplex *alpha, doublecomplex *x, int incx, doublecomplex *y, int incy, doublecomplex *a, int lda);
extern void zgeru(int m, int n, doublecomplex *alpha, doublecomplex *x, int incx, doublecomplex *y, int incy, doublecomplex *a, int lda);
extern void zhbmv(char uplo, int n, int ndiag, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *x, int incx, doublecomplex *beta, doublecomplex *y, int incy);
extern void zhemm(char side, char uplo, int m, int n, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *beta, doublecomplex *c, int ldc);
extern void zhemv(char uplo, int n, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *x, int incx, doublecomplex *beta, doublecomplex *y, int incy);
extern void zher(char uplo, int n, double alpha, doublecomplex *x, int incx, doublecomplex *a, int lda);
extern void zher2(char uplo, int n, doublecomplex *alpha, doublecomplex *x, int incx, doublecomplex *y, int incy, doublecomplex *a, int lda);
extern void zher2k(char uplo, char transa, int n, int k, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *b, int ldb, double beta, doublecomplex *c, int ldc);
extern void zherk(char uplo, char transa, int n, int k, double alpha, doublecomplex *a, int lda, double beta, doublecomplex *c, int ldc);
extern void zhpmv(char uplo, int n, doublecomplex *alpha, doublecomplex *a, doublecomplex *x, int incx, doublecomplex *beta, doublecomplex *y, int incy);
extern void zhpr(char uplo, int n, double alpha, doublecomplex *x, int incx, doublecomplex *a);
extern void zhpr2(char uplo, int n, doublecomplex *alpha, doublecomplex *x, int incx, doublecomplex *y, int incy, doublecomplex *a);
extern void zrotg(doublecomplex *a, doublecomplex *b, double *c, doublecomplex *s);
extern void zscal(int n, doublecomplex *alpha, doublecomplex *y, int incy);
extern void zswap(int n, doublecomplex *x, int incx, doublecomplex *y, int incy);
extern void zsymm(char side, char uplo, int m, int n, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *beta, doublecomplex *c, int ldc);
extern void zsyr2k(char uplo, char transa, int n, int k, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *beta, doublecomplex *c, int ldc);
extern void zsyrk(char uplo, char transa, int n, int k, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *beta, doublecomplex *c, int ldc);
extern void ztbmv(char uplo, char transa, char diag, int n, int ndiag, doublecomplex *a, int lda, doublecomplex *y, int incy);
extern void ztbsv(char uplo, char transa, char diag, int n, int ndiag, doublecomplex *a, int lda, doublecomplex *y, int incy);
extern void ztpmv(char uplo, char transa, char diag, int n, doublecomplex *a, doublecomplex *y, int incy);
extern void ztpsv(char uplo, char transa, char diag, int n, doublecomplex *a, doublecomplex *y, int incy);
extern void ztrmm(char side, char uplo, char transa, char diag, int m, int n, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *b, int ldb);
extern void ztrmv(char uplo, char transa, char diag, int n, doublecomplex *a, int lda, doublecomplex *y, int incy);
extern void ztrsm(char side, char uplo, char transa, char diag, int m, int n, doublecomplex *alpha, doublecomplex *a, int lda, doublecomplex *b, int ldb);
extern void ztrsv(char uplo, char transa, char diag, int n, doublecomplex *a, int lda, doublecomplex *y, int incy);

/* LAPACK C interfaces */
extern int ilaenv(int ispec, const char *name, const char *opts, int n1, int n2, int n3, int n4);
extern void ilaenvset(int ispec, const char *name, const char *opts, int n1, int n2, int n3, int n4, int nvalue, int *info);
extern void cbdsqr(char uplo, int n, int ncvt, int nru, int ncc, float *d, float *e, complex *vt, int ldvt, complex *u, int ldu, complex *c, int ldc, int *info);
extern void cgbbrd(char vect, int m, int n, int ncc, int kl, int ku, complex *ab, int ldab, float *d, float *e, complex *q, int ldq, complex *pt, int ldpt, complex *c, int ldc, int *info);
extern void cgbcon(char norm, int n, int nsub, int nsuper, complex *a, int lda, int *ipiv, float anorm, float *rcond, int *info);
extern void cgbequ(int m, int n, int nsub, int nsuper, complex *a, int lda, float *rowsc, float *colsc, float *rowcnd, float *colcnd, float *amax, int *info);
extern void cgbrfs(char transa, int n, int nsub, int nsuper, int nrhs, complex *a, int lda, complex *af, int ldaf, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void cgbsv(int n, int nsub, int nsuper, int nrhs, complex *a, int lda, int *ipiv, complex *b, int ldb, int *info);

/* N.B. In this C interface to cgbsvx, rpvgrw returns what is returned in
   the first element of RWORK by the FORTRAN interface */
extern void cgbsvx(char fact, char transa, int n, int nsub, int nsuper, int nrhs, complex *a, int lda, complex *af, int ldaf, int *ipiv, char equed, float *rowsc, float *colsc, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, float *rpvgrw, int *info);

extern void cgbtrf(int m, int n, int nsub, int nsuper, complex *a, int lda, int *ipiv, int *info);
extern void cgbtrs(char transa, int n, int nsub, int nsuper, int nrhs, complex *a, int lda, int *ipiv, complex *b, int ldb, int *info);
extern void cgebak(char job, char side, int n, int ilo, int ihi, float *scale, int m, complex *v, int ldv, int *info);
extern void cgebal(char job, int n, complex *a, int lda, int *ilo, int *ihi, float *scale, int *info);
extern void cgebrd(int m, int n, complex *a, int lda, float *d, float *e, complex *tauq, complex *taup, int *info);
extern void cgecon(char norm, int n, complex *a, int lda, float anorm, float *rcond, int *info);
extern void cgeequ(int m, int n, complex *a, int lda, float *rowsc, float *colsc, float *rowcnd, float *colcnd, float *amax, int *info);
extern void cgees(char jobz, char sortev, ACML_CGEES_SELECT select, int n, complex *a, int lda, int *nout, complex *w, complex *z, int ldz, int *info);
extern void cgeesx(char jobz, char sortev, ACML_CGEESX_SELECT select, char sense, int n, complex *a, int lda, int *nout, complex *w, complex *z, int ldz, float *rcone, float *rconv, int *info);
extern void cgeev(char jobvl, char jobvr, int n, complex *a, int lda, complex *w, complex *vl, int ldvl, complex *vr, int ldvr, int *info);
extern void cgeevx(char balanc, char jobvl, char jobvr, char sense, int n, complex *a, int lda, complex *w, complex *vl, int ldvl, complex *vr, int ldvr, int *ilo, int *ihi, float *scale, float *abnrm, float *rconde, float *rcondv, int *info);
extern void cgegs(char jobvsl, char jobvsr, int n, complex *a, int lda, complex *b, int ldb, complex *alpha, complex *beta, complex *vsl, int ldvsl, complex *vsr, int ldvsr, int *info);
extern void cgegv(char jobvl, char jobvr, int n, complex *a, int lda, complex *b, int ldb, complex *alpha, complex *beta, complex *vl, int ldvl, complex *vr, int ldvr, int *info);
extern void cgehrd(int n, int ilo, int ihi, complex *a, int lda, complex *tau, int *info);
extern void cgelqf(int m, int n, complex *a, int lda, complex *tau, int *info);
extern void cgels(char trans, int m, int n, int nrhs, complex *a, int lda, complex *b, int ldb, int *info);
extern void cgelsd(int m, int n, int nrhs, complex *a, int lda, complex *b, int ldb, float *s, float rcond, int *rank, int *info);
extern void cgelss(int m, int n, int nrhs, complex *a, int lda, complex *b, int ldb, float *sing, float rcond, int *irank, int *info);
extern void cgelsx(int m, int n, int nrhs, complex *a, int lda, complex *b, int ldb, int *jpivot, float rcond, int *irank, int *info);
extern void cgelsy(int m, int n, int nrhs, complex *a, int lda, complex *b, int ldb, int *jpvt, float rcond, int *rank, int *info);
extern void cgeqlf(int m, int n, complex *a, int lda, complex *tau, int *info);
extern void cgeqp3(int m, int n, complex *a, int lda, int *jpvt, complex *tau, int *info);
extern void cgeqpf(int m, int n, complex *a, int lda, int *jpvt, complex *tau, int *info);
extern void cgeqrf(int m, int n, complex *a, int lda, complex *tau, int *info);
extern void cgerfs(char transa, int n, int nrhs, complex *a, int lda, complex *af, int ldaf, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void cgerqf(int m, int n, complex *a, int lda, complex *tau, int *info);
extern void cgesdd(char jobz, int m, int n, complex *a, int lda, float *s, complex *u, int ldu, complex *vt, int ldvt, int *info);
extern void cgesv(int n, int nrhs, complex *a, int lda, int *ipiv, complex *b, int ldb, int *info);
extern void cgesvd(char jobu, char jobvt, int m, int n, complex *a, int lda, float *sing, complex *u, int ldu, complex *vt, int ldvt, int *info);

/* N.B. In this C interface to cgesvx, rpvgrw returns what is returned in
   the first element of RWORK by the FORTRAN interface */
extern void cgesvx(char fact, char transa, int n, int nrhs, complex *a, int lda, complex *af, int ldaf, int *ipiv, char equed, float *rowsc, float *colsc, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, float *rpvgrw, int *info);

extern void cgetrf(int m, int n, complex *a, int lda, int *ipiv, int *info);
extern void cgetri(int n, complex *a, int lda, int *ipiv, int *info);
extern void cgetrs(char transa, int n, int nrhs, complex *a, int lda, int *ipiv, complex *b, int ldb, int *info);
extern void cggbak(char job, char side, int n, int ilo, int ihi, float *lscale, float *rscale, int m, complex *v, int ldv, int *info);
extern void cggbal(char job, int n, complex *a, int lda, complex *b, int ldb, int *ilo, int *ihi, float *lscale, float *rscale, int *info);
extern void cgges(char jobvsl, char jobvsr, char sort, ACML_CGGES_SELCTG selctg, int n, complex *a, int lda, complex *b, int ldb, int *sdim, complex *alpha, complex *beta, complex *vsl, int ldvsl, complex *vsr, int ldvsr, int *info);
extern void cggesx(char jobvsl, char jobvsr, char sort, ACML_CGGESX_SELCTG selctg, char sense, int n, complex *a, int lda, complex *b, int ldb, int *sdim, complex *alpha, complex *beta, complex *vsl, int ldvsl, complex *vsr, int ldvsr, float *rconde, float *rcondv, int *info);
extern void cggev(char jobvl, char jobvr, int n, complex *a, int lda, complex *b, int ldb, complex *alpha, complex *beta, complex *vl, int ldvl, complex *vr, int ldvr, int *info);
extern void cggevx(char balanc, char jobvl, char jobvr, char sense, int n, complex *a, int lda, complex *b, int ldb, complex *alpha, complex *beta, complex *vl, int ldvl, complex *vr, int ldvr, int *ilo, int *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv, int *info);
extern void cggglm(int n, int m, int p, complex *a, int lda, complex *b, int ldb, complex *d, complex *x, complex *y, int *info);
extern void cgghrd(char compq, char compz, int n, int ilo, int ihi, complex *a, int lda, complex *b, int ldb, complex *q, int ldq, complex *z, int ldz, int *info);
extern void cgglse(int m, int n, int p, complex *a, int lda, complex *b, int ldb, complex *c, complex *d, complex *x, int *info);
extern void cggqrf(int n, int m, int p, complex *a, int lda, complex *taua, complex *b, int ldb, complex *taub, int *info);
extern void cggrqf(int m, int p, int n, complex *a, int lda, complex *taua, complex *b, int ldb, complex *taub, int *info);
extern void cggsvd(char jobu, char jobv, char jobq, int m, int n, int p, int *k, int *l, complex *a, int lda, complex *b, int ldb, float *alpha, float *beta, complex *u, int ldu, complex *v, int ldv, complex *q, int ldq, int *iwork, int *info);
extern void cggsvp(char jobu, char jobv, char jobq, int m, int p, int n, complex *a, int lda, complex *b, int ldb, float tola, float tolb, int *k, int *l, complex *u, int ldu, complex *v, int ldv, complex *q, int ldq, int *info);
extern void cgtcon(char norm, int n, complex *low, complex *diag, complex *up1, complex *up2, int *ipiv, float anorm, float *rcond, int *info);
extern void cgtrfs(char transa, int n, int nrhs, complex *low, complex *diag, complex *up, complex *lowf, complex *diagf, complex *upf1, complex *upf2, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void cgtsv(int n, int nrhs, complex *low, complex *diag, complex *up, complex *b, int ldb, int *info);
extern void cgtsvx(char fact, char transa, int n, int nrhs, complex *low, complex *diag, complex *up, complex *lowf, complex *diagf, complex *upf1, complex *upf2, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void cgttrf(int n, complex *low, complex *diag, complex *up1, complex *up2, int *ipiv, int *info);
extern void cgttrs(char transa, int n, int nrhs, complex *low, complex *diag, complex *up1, complex *up2, int *ipiv, complex *b, int ldb, int *info);
extern void chbev(char jobz, char uplo, int n, int ndiag, complex *a, int lda, float *w, complex *z, int ldz, int *info);
extern void chbevd(char jobz, char uplo, int n, int kd, complex *ab, int ldab, float *w, complex *z, int ldz, int *info);
extern void chbevx(char jobz, char range, char uplo, int n, int ndiag, complex *a, int lda, complex *q, int ldq, float vl, float vu, int il, int iu, float abtol, int *nfound, float *w, complex *z, int ldz, int *ifail, int *info);
extern void chbgst(char vect, char uplo, int n, int ka, int kb, complex *ab, int ldab, complex *bb, int ldbb, complex *x, int ldx, int *info);
extern void chbgv(char jobz, char uplo, int n, int ka, int kb, complex *ab, int ldab, complex *bb, int ldbb, float *w, complex *z, int ldz, int *info);
extern void chbgvd(char jobz, char uplo, int n, int ka, int kb, complex *ab, int ldab, complex *bb, int ldbb, float *w, complex *z, int ldz, int *info);
extern void chbgvx(char jobz, char range, char uplo, int n, int ka, int kb, complex *ab, int ldab, complex *bb, int ldbb, complex *q, int ldq, float vl, float vu, int il, int iu, float abstol, int *m, float *w, complex *z, int ldz, int *ifail, int *info);
extern void chbtrd(char vect, char uplo, int n, int kd, complex *ab, int ldab, float *d, float *e, complex *q, int ldq, int *info);
extern void checon(char uplo, int n, complex *a, int lda, int *ipiv, float anorm, float *rcond, int *info);
extern void cheev(char jobz, char uplo, int n, complex *a, int lda, float *w, int *info);
extern void cheevd(char jobz, char uplo, int n, complex *a, int lda, float *w, int *info);
extern void cheevr(char jobz, char range, char uplo, int n, complex *a, int lda, float vl, float vu, int il, int iu, float abstol, int *m, float *w, complex *z, int ldz, int *isuppz, int *info);
extern void cheevx(char jobz, char range, char uplo, int n, complex *a, int lda, float vl, float vu, int il, int iu, float abtol, int *nfound, float *w, complex *z, int ldz, int *ifail, int *info);
extern void chegst(int itype, char uplo, int n, complex *a, int lda, complex *b, int ldb, int *info);
extern void chegv(int itype, char jobz, char uplo, int n, complex *a, int lda, complex *b, int ldb, float *w, int *info);
extern void chegvd(int itype, char jobz, char uplo, int n, complex *a, int lda, complex *b, int ldb, float *w, int *info);
extern void chegvx(int itype, char jobz, char range, char uplo, int n, complex *a, int lda, complex *b, int ldb, float vl, float vu, int il, int iu, float abstol, int *m, float *w, complex *z, int ldz, int *ifail, int *info);
extern void cherfs(char uplo, int n, int nrhs, complex *a, int lda, complex *af, int ldaf, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void chesv(char uplo, int n, int nrhs, complex *a, int lda, int *ipiv, complex *b, int ldb, int *info);
extern void chesvx(char fact, char uplo, int n, int nrhs, complex *a, int lda, complex *af, int ldaf, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void chetrd(char uplo, int n, complex *a, int lda, float *d, float *e, complex *tau, int *info);
extern void chetrf(char uplo, int n, complex *a, int lda, int *ipiv, int *info);
extern void chetri(char uplo, int n, complex *a, int lda, int *ipiv, int *info);
extern void chetrs(char uplo, int n, int nrhs, complex *a, int lda, int *ipiv, complex *b, int ldb, int *info);
extern void chgeqz(char job, char compq, char compz, int n, int ilo, int ihi, complex *a, int lda, complex *b, int ldb, complex *alpha, complex *beta, complex *q, int ldq, complex *z, int ldz, int *info);
extern void chpcon(char uplo, int n, complex *a, int *ipiv, float anorm, float *rcond, int *info);
extern void chpev(char jobz, char uplo, int n, complex *a, float *w, complex *z, int ldz, int *info);
extern void chpevd(char jobz, char uplo, int n, complex *ap, float *w, complex *z, int ldz, int *info);
extern void chpevx(char jobz, char range, char uplo, int n, complex *a, float vl, float vu, int il, int iu, float abtol, int *nfound, float *w, complex *z, int ldz, int *ifail, int *info);
extern void chpgst(int itype, char uplo, int n, complex *ap, complex *bp, int *info);
extern void chpgv(int itype, char jobz, char uplo, int n, complex *a, complex *b, float *w, complex *z, int ldz, int *info);
extern void chpgvd(int itype, char jobz, char uplo, int n, complex *ap, complex *bp, float *w, complex *z, int ldz, int *info);
extern void chpgvx(int itype, char jobz, char range, char uplo, int n, complex *ap, complex *bp, float vl, float vu, int il, int iu, float abstol, int *m, float *w, complex *z, int ldz, int *ifail, int *info);
extern void chprfs(char uplo, int n, int nrhs, complex *a, complex *af, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void chpsv(char uplo, int n, int nrhs, complex *a, int *ipiv, complex *b, int ldb, int *info);
extern void chpsvx(char fact, char uplo, int n, int nrhs, complex *a, complex *af, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void chptrd(char uplo, int n, complex *ap, float *d, float *e, complex *tau, int *info);
extern void chptrf(char uplo, int n, complex *a, int *ipiv, int *info);
extern void chptri(char uplo, int n, complex *a, int *ipiv, int *info);
extern void chptrs(char uplo, int n, int nrhs, complex *a, int *ipiv, complex *b, int ldb, int *info);
extern void chsein(char side, char eigsrc, char initv, int *select, int n, complex *h, int ldh, complex *w, complex *vl, int ldvl, complex *vr, int ldvr, int mm, int *m, int *ifaill, int *ifailr, int *info);
extern void chseqr(char job, char compz, int n, int ilo, int ihi, complex *h, int ldh, complex *w, complex *z, int ldz, int *info);
extern void cpbcon(char uplo, int n, int ndiag, complex *a, int lda, float anorm, float *rcond, int *info);
extern void cpbequ(char uplo, int n, int ndiag, complex *a, int lda, float *scale, float *scond, float *amax, int *info);
extern void cpbrfs(char uplo, int n, int ndiag, int nrhs, complex *a, int lda, complex *af, int ldaf, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void cpbstf(char uplo, int n, int kd, complex *ab, int ldab, int *info);
extern void cpbsv(char uplo, int n, int ndiag, int nrhs, complex *a, int lda, complex *b, int ldb, int *info);
extern void cpbsvx(char fact, char uplo, int n, int ndiag, int nrhs, complex *a, int lda, complex *af, int ldaf, char equed, float *scale, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void cpbtrf(char uplo, int n, int ndiag, complex *a, int lda, int *info);
extern void cpbtrs(char uplo, int n, int ndiag, int nrhs, complex *a, int lda, complex *b, int ldb, int *info);
extern void cpocon(char uplo, int n, complex *a, int lda, float anorm, float *rcond, int *info);
extern void cpoequ(int n, complex *a, int lda, float *scale, float *scond, float *amax, int *info);
extern void cporfs(char uplo, int n, int nrhs, complex *a, int lda, complex *af, int ldaf, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void cposv(char uplo, int n, int nrhs, complex *a, int lda, complex *b, int ldb, int *info);
extern void cposvx(char fact, char uplo, int n, int nrhs, complex *a, int lda, complex *af, int ldaf, char equed, float *scale, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void cpotrf(char uplo, int n, complex *a, int lda, int *info);
extern void cpotri(char uplo, int n, complex *a, int lda, int *info);
extern void cpotrs(char uplo, int n, int nrhs, complex *a, int lda, complex *b, int ldb, int *info);
extern void cppcon(char uplo, int n, complex *a, float anorm, float *rcond, int *info);
extern void cppequ(char uplo, int n, complex *a, float *scale, float *scond, float *amax, int *info);
extern void cpprfs(char uplo, int n, int nrhs, complex *a, complex *af, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void cppsv(char uplo, int n, int nrhs, complex *a, complex *b, int ldb, int *info);
extern void cppsvx(char fact, char uplo, int n, int nrhs, complex *a, complex *af, char equed, float *scale, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void cpptrf(char uplo, int n, complex *a, int *info);
extern void cpptri(char uplo, int n, complex *a, int *info);
extern void cpptrs(char uplo, int n, int nrhs, complex *a, complex *b, int ldb, int *info);
extern void cptcon(int n, float *diag, complex *offd, float anorm, float *rcond, int *info);
extern void cpteqr(char compz, int n, float *d, float *e, complex *z, int ldz, int *info);
extern void cptrfs(char uplo, int n, int nrhs, float *diag, complex *offd, float *diagf, complex *offdf, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void cptsv(int n, int nrhs, float *diag, complex *sub, complex *b, int ldb, int *info);
extern void cptsvx(char fact, int n, int nrhs, float *diag, complex *sub, float *diagf, complex *subf, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void cpttrf(int n, float *diag, complex *offd, int *info);
extern void cpttrs(char uplo, int n, int nrhs, float *diag, complex *offd, complex *b, int ldb, int *info);
extern void cspcon(char uplo, int n, complex *a, int *ipiv, float anorm, float *rcond, int *info);
extern void csprfs(char uplo, int n, int nrhs, complex *a, complex *af, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void cspsv(char uplo, int n, int nrhs, complex *a, int *ipiv, complex *b, int ldb, int *info);
extern void cspsvx(char fact, char uplo, int n, int nrhs, complex *a, complex *af, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void csptrf(char uplo, int n, complex *a, int *ipiv, int *info);
extern void csptri(char uplo, int n, complex *a, int *ipiv, int *info);
extern void csptrs(char uplo, int n, int nrhs, complex *a, int *ipiv, complex *b, int ldb, int *info);
extern void cstedc(char compz, int n, float *d, float *e, complex *z, int ldz, int *info);
extern void cstegr(char jobz, char range, int n, float *d, float *e, float vl, float vu, int il, int iu, float abstol, int *m, float *w, complex *z, int ldz, int *isuppz, int *info);
extern void cstein(int n, float *d, float *e, int m, float *w, int *iblock, int *isplit, complex *z, int ldz, int *ifail, int *info);
extern void csteqr(char compz, int n, float *d, float *e, complex *z, int ldz, int *info);
extern void csycon(char uplo, int n, complex *a, int lda, int *ipiv, float anorm, float *rcond, int *info);
extern void csyrfs(char uplo, int n, int nrhs, complex *a, int lda, complex *af, int ldaf, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void csysv(char uplo, int n, int nrhs, complex *a, int lda, int *ipiv, complex *b, int ldb, int *info);
extern void csysvx(char fact, char uplo, int n, int nrhs, complex *a, int lda, complex *af, int ldaf, int *ipiv, complex *b, int ldb, complex *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void csytrf(char uplo, int n, complex *a, int lda, int *ipiv, int *info);
extern void csytri(char uplo, int n, complex *a, int lda, int *ipiv, int *info);
extern void csytrs(char uplo, int n, int nrhs, complex *a, int lda, int *ipiv, complex *b, int ldb, int *info);
extern void ctbcon(char norm, char uplo, char diag, int n, int ndiag, complex *a, int lda, float *rcond, int *info);
extern void ctbrfs(char uplo, char transa, char diag, int n, int ndiag, int nrhs, complex *a, int lda, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void ctbtrs(char uplo, char transa, char diag, int n, int ndiag, int nrhs, complex *a, int lda, complex *b, int ldb, int *info);
extern void ctgevc(char side, char howmny, int *select, int n, complex *a, int lda, complex *b, int ldb, complex *vl, int ldvl, complex *vr, int ldvr, int mm, int *m, int *info);
extern void ctgexc(int wantq, int wantz, int n, complex *a, int lda, complex *b, int ldb, complex *q, int ldq, complex *z, int ldz, int *ifst, int *ilst, int *info);
extern void ctgsen(int ijob, int wantq, int wantz, int *select, int n, complex *a, int lda, complex *b, int ldb, complex *alpha, complex *beta, complex *q, int ldq, complex *z, int ldz, int *m, float *pl, float *pr, float *dif, int *info);
extern void ctgsja(char jobu, char jobv, char jobq, int m, int p, int n, int k, int l, complex *a, int lda, complex *b, int ldb, float tola, float tolb, float *alpha, float *beta, complex *u, int ldu, complex *v, int ldv, complex *q, int ldq, int *ncycle, int *info);
extern void ctgsna(char job, char howmnt, int *select, int n, complex *a, int lda, complex *b, int ldb, complex *vl, int ldvl, complex *vr, int ldvr, float *s, float *dif, int mm, int *m, int *info);
extern void ctgsyl(char trans, int ijob, int m, int n, complex *a, int lda, complex *b, int ldb, complex *c, int ldc, complex *d, int ldd, complex *e, int lde, complex *f, int ldf, float *scale, float *dif, int *info);
extern void ctpcon(char norm, char uplo, char diag, int n, complex *a, float *rcond, int *info);
extern void ctprfs(char uplo, char transa, char diag, int n, int nrhs, complex *a, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void ctptri(char uplo, char diag, int n, complex *a, int *info);
extern void ctptrs(char uplo, char transa, char diag, int n, int nrhs, complex *a, complex *b, int ldb, int *info);
extern void ctrcon(char norm, char uplo, char diag, int n, complex *a, int lda, float *rcond, int *info);
extern void ctrevc(char side, char howmny, int *select, int n, complex *t, int ldt, complex *vl, int ldvl, complex *vr, int ldvr, int mm, int *m, int *info);
extern void ctrexc(char compq, int n, complex *t, int ldt, complex *q, int ldq, int ifst, int ilst, int *info);
extern void ctrrfs(char uplo, char transa, char diag, int n, int nrhs, complex *a, int lda, complex *b, int ldb, complex *x, int ldx, float *ferr, float *berr, int *info);
extern void ctrsen(char job, char compq, int *select, int n, complex *t, int ldt, complex *q, int ldq, complex *w, int *m, float *s, float *sep, int *info);
extern void ctrsna(char job, char howmny, int *select, int n, complex *t, int ldt, complex *vl, int ldvl, complex *vr, int ldvr, float *s, float *sep, int mm, int *m, int *info);
extern void ctrsyl(char trana, char tranb, int isgn, int m, int n, complex *a, int lda, complex *b, int ldb, complex *c, int ldc, float *scale, int *info);
extern void ctrtri(char uplo, char diag, int n, complex *a, int lda, int *info);
extern void ctrtrs(char uplo, char transa, char diag, int n, int nrhs, complex *a, int lda, complex *b, int ldb, int *info);
extern void ctzrqf(int m, int n, complex *a, int lda, complex *tau, int *info);
extern void ctzrzf(int m, int n, complex *a, int lda, complex *tau, int *info);
extern void cungbr(char vect, int m, int n, int k, complex *a, int lda, complex *tau, int *info);
extern void cunghr(int n, int ilo, int ihi, complex *a, int lda, complex *tau, int *info);
extern void cunglq(int m, int n, int k, complex *a, int lda, complex *tau, int *info);
extern void cungql(int m, int n, int k, complex *a, int lda, complex *tau, int *info);
extern void cungqr(int m, int n, int k, complex *a, int lda, complex *tau, int *info);
extern void cungrq(int m, int n, int k, complex *a, int lda, complex *tau, int *info);
extern void cungtr(char uplo, int n, complex *a, int lda, complex *tau, int *info);
extern void cunmbr(char vect, char side, char trans, int m, int n, int k, complex *a, int lda, complex *tau, complex *c, int ldc, int *info);
extern void cunmhr(char side, char trans, int m, int n, int ilo, int ihi, complex *a, int lda, complex *tau, complex *c, int ldc, int *info);
extern void cunmlq(char side, char trans, int m, int n, int k, complex *a, int lda, complex *tau, complex *c, int ldc, int *info);
extern void cunmql(char side, char trans, int m, int n, int k, complex *a, int lda, complex *tau, complex *c, int ldc, int *info);
extern void cunmqr(char side, char trans, int m, int n, int k, complex *a, int lda, complex *tau, complex *c, int ldc, int *info);
extern void cunmrq(char side, char trans, int m, int n, int k, complex *a, int lda, complex *tau, complex *c, int ldc, int *info);
extern void cunmrz(char side, char trans, int m, int n, int k, int l, complex *a, int lda, complex *tau, complex *c, int ldc, int *info);
extern void cunmtr(char side, char uplo, char trans, int m, int n, complex *a, int lda, complex *tau, complex *c, int ldc, int *info);
extern void cupgtr(char uplo, int n, complex *ap, complex *tau, complex *q, int ldq, int *info);
extern void cupmtr(char side, char uplo, char trans, int m, int n, complex *ap, complex *tau, complex *c, int ldc, int *info);
extern void dbdsdc(char uplo, char compq, int n, double *d, double *e, double *u, int ldu, double *vt, int ldvt, double *q, int *iq, int *info);
extern void dbdsqr(char uplo, int n, int ncvt, int nru, int ncc, double *d, double *e, double *vt, int ldvt, double *u, int ldu, double *c, int ldc, int *info);
extern void ddisna(char job, int m, int n, double *d, double *sep, int *info);
extern void dgbbrd(char vect, int m, int n, int ncc, int kl, int ku, double *ab, int ldab, double *d, double *e, double *q, int ldq, double *pt, int ldpt, double *c, int ldc, int *info);
extern void dgbcon(char norm, int n, int nsub, int nsuper, double *a, int lda, int *ipiv, double anorm, double *rcond, int *info);
extern void dgbequ(int m, int n, int nsub, int nsuper, double *a, int lda, double *rowsc, double *colsc, double *rowcn, double *colcn, double *amax, int *info);
extern void dgbrfs(char transa, int n, int nsub, int nsuper, int nrhs, double *a, int lda, double *af, int ldaf, int *ipiv, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dgbsv(int n, int nsub, int nsuper, int nrhs, double *a, int lda, int *ipiv, double *b, int ldb, int *info);

/* N.B. In this C interface to dgbsvx, rpvgrw returns what is returned in
   the first element of WORK by the FORTRAN interface */
extern void dgbsvx(char fact, char transa, int n, int nsub, int nsuper, int nrhs, double *a, int lda, double *af, int ldaf, int *ipiv, char equed, double *rowsc, double *colsc, double *b, int ldb, double *x, int ldx, double *rcond, double *ferr, double *berr, double *rpvgrw, int *info);

extern void dgbtrf(int m, int n, int nsub, int nsuper, double *a, int lda, int *ipiv, int *info);
extern void dgbtrs(char transa, int n, int nsub, int nsuper, int nrhs, double *a, int lda, int *ipiv, double *b, int ldb, int *info);
extern void dgebak(char job, char side, int n, int ilo, int ihi, double *scale, int m, double *v, int ldv, int *info);
extern void dgebal(char job, int n, double *a, int lda, int *ilo, int *ihi, double *scale, int *info);
extern void dgebrd(int m, int n, double *a, int lda, double *d, double *e, double *tauq, double *taup, int *info);
extern void dgecon(char norm, int n, double *a, int lda, double anorm, double *rcond, int *info);
extern void dgeequ(int m, int n, double *a, int lda, double *rowsc, double *colsc, double *rowcn, double *colcn, double *amax, int *info);
extern void dgees(char jobz, char sortev, ACML_DGEES_SELECT select, int n, double *a, int lda, int *nout, double *wr, double *wi, double *z, int ldz, int *info);
extern void dgeesx(char jobz, char sortev, ACML_DGEESX_SELECT select, char sense, int n, double *a, int lda, int *nout, double *wr, double *wi, double *z, int ldz, double *srcone, double *rconv, int *info);
extern void dgeev(char jobvl, char jobvr, int n, double *a, int lda, double *wr, double *wi, double *vl, int ldvl, double *vr, int ldvr, int *info);
extern void dgeevx(char balanc, char jobvl, char jobvr, char sense, int n, double *a, int lda, double *wr, double *wi, double *vl, int ldvl, double *vr, int ldvr, int *ilo, int *ihi, double *scale, double *abnrm, double *rcone, double *rconv, int *info);
extern void dgegs(char jobvsl, char jobvsr, int n, double *a, int lda, double *b, int ldb, double *alphar, double *alphai, double *beta, double *vsl, int ldvsl, double *vsr, int ldvsr, int *info);
extern void dgegv(char jobvl, char jobvr, int n, double *a, int lda, double *b, int ldb, double *alphar, double *alphai, double *beta, double *vl, int ldvl, double *vr, int ldvr, int *info);
extern void dgehrd(int n, int ilo, int ihi, double *a, int lda, double *tau, int *info);
extern void dgelqf(int m, int n, double *a, int lda, double *tau, int *info);
extern void dgels(char trans, int m, int n, int nrhs, double *a, int lda, double *b, int ldb, int *info);
extern void dgelsd(int m, int n, int nrhs, double *a, int lda, double *b, int ldb, double *s, double rcond, int *rank, int *info);
extern void dgelss(int m, int n, int nrhs, double *a, int lda, double *b, int ldb, double *sing, double rcond, int *irank, int *info);
extern void dgelsx(int m, int n, int nrhs, double *a, int lda, double *b, int ldb, int *jpivot, double rcond, int *irank, int *info);
extern void dgelsy(int m, int n, int nrhs, double *a, int lda, double *b, int ldb, int *jpvt, double rcond, int *rank, int *info);
extern void dgeqlf(int m, int n, double *a, int lda, double *tau, int *info);
extern void dgeqp3(int m, int n, double *a, int lda, int *jpvt, double *tau, int *info);
extern void dgeqpf(int m, int n, double *a, int lda, int *jpivot, double *tau, int *info);
extern void dgeqrf(int m, int n, double *a, int lda, double *tau, int *info);
extern void dgerfs(char transa, int n, int nrhs, double *a, int lda, double *af, int ldaf, int *ipiv, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dgerqf(int m, int n, double *a, int lda, double *tau, int *info);
extern void dgesdd(char jobz, int m, int n, double *a, int lda, double *s, double *u, int ldu, double *vt, int ldvt, int *info);
extern void dgesv(int n, int nrhs, double *a, int lda, int *ipiv, double *b, int ldb, int *info);
extern void dgesvd(char jobu, char jobvt, int m, int n, double *a, int lda, double *sing, double *u, int ldu, double *vt, int ldvt, int *info);

/* N.B. In this C interface to dgesvx, rpvgrw returns what is returned in
   the first element of WORK by the FORTRAN interface */
extern void dgesvx(char fact, char transa, int n, int nrhs, double *a, int lda, double *af, int ldaf, int *ipiv, char equed, double *rowsc, double *colsc, double *b, int ldb, double *x, int ldx, double *rcond, double *ferr, double *berr, double *rpvgrw, int *info);

extern void dgetrf(int m, int n, double *a, int lda, int *ipiv, int *info);
extern void dgetri(int n, double *a, int lda, int *ipiv, int *info);
extern void dgetrs(char transa, int n, int nrhs, double *a, int lda, int *ipiv, double *b, int ldb, int *info);
extern void dggbak(char job, char side, int n, int ilo, int ihi, double *lscale, double *rscale, int m, double *v, int ldv, int *info);
extern void dggbal(char job, int n, double *a, int lda, double *b, int ldb, int *ilo, int *ihi, double *lscale, double *rscale, int *info);
extern void dgges(char jobvsl, char jobvsr, char sort, ACML_DGGES_DELCTG delctg, int n, double *a, int lda, double *b, int ldb, int *sdim, double *alphar, double *alphai, double *beta, double *vsl, int ldvsl, double *vsr, int ldvsr, int *info);
extern void dggesx(char jobvsl, char jobvsr, char sort, ACML_DGGESX_DELCTG delctg, char sense, int n, double *a, int lda, double *b, int ldb, int *sdim, double *alphar, double *alphai, double *beta, double *vsl, int ldvsl, double *vsr, int ldvsr, double *rconde, double *rcondv, int *info);
extern void dggev(char jobvl, char jobvr, int n, double *a, int lda, double *b, int ldb, double *alphar, double *alphai, double *beta, double *vl, int ldvl, double *vr, int ldvr, int *info);
extern void dggevx(char balanc, char jobvl, char jobvr, char sense, int n, double *a, int lda, double *b, int ldb, double *alphar, double *alphai, double *beta, double *vl, int ldvl, double *vr, int ldvr, int *ilo, int *ihi, double *lscale, double *rscale, double *abnrm, double *bbnrm, double *rconde, double *rcondv, int *info);
extern void dggglm(int n, int m, int p, double *a, int lda, double *b, int ldb, double *d, double *x, double *y, int *info);
extern void dgghrd(char compq, char compz, int n, int ilo, int ihi, double *a, int lda, double *b, int ldb, double *q, int ldq, double *z, int ldz, int *info);
extern void dgglse(int m, int n, int p, double *a, int lda, double *b, int ldb, double *c, double *d, double *x, int *info);
extern void dggqrf(int n, int m, int p, double *a, int lda, double *taua, double *b, int ldb, double *taub, int *info);
extern void dggrqf(int m, int p, int n, double *a, int lda, double *taua, double *b, int ldb, double *taub, int *info);
extern void dggsvd(char jobu, char jobv, char jobq, int m, int n, int p, int *k, int *l, double *a, int lda, double *b, int ldb, double *alpha, double *beta, double *u, int ldu, double *v, int ldv, double *q, int ldq, int *iwork3, int *info);
extern void dggsvp(char jobu, char jobv, char jobq, int m, int p, int n, double *a, int lda, double *b, int ldb, double tola, double tolb, int *k, int *l, double *u, int ldu, double *v, int ldv, double *q, int ldq, int *info);
extern void dgtcon(char norm, int n, double *low, double *diag, double *up1, double *up2, int *ipiv, double anorm, double *rcond, int *info);
extern void dgtrfs(char transa, int n, int nrhs, double *low, double *diag, double *up, double *lowf, double *diagf, double *upf1, double *upf2, int *ipiv, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dgtsv(int n, int nrhs, double *low, double *diag, double *up, double *b, int ldb, int *info);
extern void dgtsvx(char fact, char transa, int n, int nrhs, double *low, double *diag, double *up, double *lowf, double *diagf, double *upf1, double *upf2, int *ipiv, double *b, int ldb, double *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void dgttrf(int n, double *low, double *diag, double *up1, double *up2, int *ipiv, int *info);
extern void dgttrs(char transa, int n, int nrhs, double *low, double *diag, double *up1, double *up2, int *ipiv, double *b, int ldb, int *info);
extern void dhgeqz(char job, char compq, char compz, int n, int ilo, int ihi, double *a, int lda, double *b, int ldb, double *alphar, double *alphai, double *beta, double *q, int ldq, double *z, int ldz, int *info);
extern void dhsein(char side, char eigsrc, char initv, int *select, int n, double *h, int ldh, double *wr, double *wi, double *vl, int ldvl, double *vr, int ldvr, int mm, int *m, int *ifaill, int *ifailr, int *info);
extern void dhseqr(char job, char compz, int n, int ilo, int ihi, double *h, int ldh, double *wr, double *wi, double *z, int ldz, int *info);
extern double dlamch(char cmach);
extern void dopgtr(char uplo, int n, double *ap, double *tau, double *q, int ldq, int *info);
extern void dopmtr(char side, char uplo, char trans, int m, int n, double *ap, double *tau, double *c, int ldc, int *info);
extern void dorgbr(char vect, int m, int n, int k, double *a, int lda, double *tau, int *info);
extern void dorghr(int n, int ilo, int ihi, double *a, int lda, double *tau, int *info);
extern void dorglq(int m, int n, int k, double *a, int lda, double *tau, int *info);
extern void dorgql(int m, int n, int k, double *a, int lda, double *tau, int *info);
extern void dorgqr(int m, int n, int k, double *a, int lda, double *tau, int *info);
extern void dorgrq(int m, int n, int k, double *a, int lda, double *tau, int *info);
extern void dorgtr(char uplo, int n, double *a, int lda, double *tau, int *info);
extern void dormbr(char vect, char side, char trans, int m, int n, int k, double *a, int lda, double *tau, double *c, int ldc, int *info);
extern void dormhr(char side, char trans, int m, int n, int ilo, int ihi, double *a, int lda, double *tau, double *c, int ldc, int *info);
extern void dormlq(char side, char trans, int m, int n, int k, double *a, int lda, double *tau, double *c, int ldc, int *info);
extern void dormql(char side, char trans, int m, int n, int k, double *a, int lda, double *tau, double *c, int ldc, int *info);
extern void dormqr(char side, char trans, int m, int n, int k, double *a, int lda, double *tau, double *c, int ldc, int *info);
extern void dormr3(char side, char trans, int m, int n, int k, int l, double *a, int lda, double *tau, double *c, int ldc, int *info);
extern void dormrq(char side, char trans, int m, int n, int k, double *a, int lda, double *tau, double *c, int ldc, int *info);
extern void dormrz(char side, char trans, int m, int n, int k, int l, double *a, int lda, double *tau, double *c, int ldc, int *info);
extern void dormtr(char side, char uplo, char trans, int m, int n, double *a, int lda, double *tau, double *c, int ldc, int *info);
extern void dpbcon(char uplo, int n, int ndiag, double *a, int lda, double anorm, double *rcond, int *info);
extern void dpbequ(char uplo, int n, int ndiag, double *a, int lda, double *scale, double *scond, double *amax, int *info);
extern void dpbrfs(char uplo, int n, int ndiag, int nrhs, double *a, int lda, double *af, int ldaf, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dpbstf(char uplo, int n, int kd, double *ab, int ldab, int *info);
extern void dpbsv(char uplo, int n, int ndiag, int nrhs, double *a, int lda, double *b, int ldb, int *info);
extern void dpbsvx(char fact, char uplo, int n, int ndiag, int nrhs, double *a, int lda, double *af, int ldaf, char equed, double *scale, double *b, int ldb, double *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void dpbtrf(char uplo, int n, int ndiag, double *a, int lda, int *info);
extern void dpbtrs(char uplo, int n, int ndiag, int nrhs, double *a, int lda, double *b, int ldb, int *info);
extern void dpocon(char uplo, int n, double *a, int lda, double anorm, double *rcond, int *info);
extern void dpoequ(int n, double *a, int lda, double *scale, double *scond, double *amax, int *info);
extern void dporfs(char uplo, int n, int nrhs, double *a, int lda, double *af, int ldaf, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dposv(char uplo, int n, int nrhs, double *a, int lda, double *b, int ldb, int *info);
extern void dposvx(char fact, char uplo, int n, int nrhs, double *a, int lda, double *af, int ldaf, char equed, double *scale, double *b, int ldb, double *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void dpotrf(char uplo, int n, double *a, int lda, int *info);
extern void dpotri(char uplo, int n, double *a, int lda, int *info);
extern void dpotrs(char uplo, int n, int nrhs, double *a, int lda, double *b, int ldb, int *info);
extern void dppcon(char uplo, int n, double *a, double anorm, double *rcond, int *info);
extern void dppequ(char uplo, int n, double *a, double *scale, double *scond, double *amax, int *info);
extern void dpprfs(char uplo, int n, int nrhs, double *a, double *af, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dppsv(char uplo, int n, int nrhs, double *a, double *b, int ldb, int *info);
extern void dppsvx(char fact, char uplo, int n, int nrhs, double *a, double *af, char equed, double *scale, double *b, int ldb, double *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void dpptrf(char uplo, int n, double *a, int *info);
extern void dpptri(char uplo, int n, double *a, int *info);
extern void dpptrs(char uplo, int n, int nrhs, double *a, double *b, int ldb, int *info);
extern void dptcon(int n, double *diag, double *offd, double anorm, double *rcond, int *info);
extern void dpteqr(char compz, int n, double *d, double *e, double *z, int ldz, int *info);
extern void dptrfs(int n, int nrhs, double *diag, double *offd, double *diagf, double *offdf, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dptsv(int n, int nrhs, double *diag, double *sub, double *b, int ldb, int *info);
extern void dptsvx(char fact, int n, int nrhs, double *diag, double *sub, double *diagf, double *subf, double *b, int ldb, double *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void dpttrf(int n, double *diag, double *offd, int *info);
extern void dpttrs(int n, int nrhs, double *diag, double *offd, double *b, int ldb, int *info);
extern void dsbev(char jobz, char uplo, int n, int ndiag, double *a, int lda, double *w, double *z, int ldz, int *info);
extern void dsbevd(char jobz, char uplo, int n, int kd, double *ab, int ldab, double *w, double *z, int ldz, int *info);
extern void dsbevx(char jobz, char range, char uplo, int n, int ndiag, double *a, int lda, double *q, int ldq, double vl, double vu, int il, int iu, double abtol, int *nfound, double *w, double *z, int ldz, int *ifail, int *info);
extern void dsbgst(char vect, char uplo, int n, int ka, int kb, double *ab, int ldab, double *bb, int ldbb, double *x, int ldx, int *info);
extern void dsbgv(char jobz, char uplo, int n, int ka, int kb, double *ab, int ldab, double *bb, int ldbb, double *w, double *z, int ldz, int *info);
extern void dsbgvd(char jobz, char uplo, int n, int ka, int kb, double *ab, int ldab, double *bb, int ldbb, double *w, double *z, int ldz, int *info);
extern void dsbgvx(char jobz, char range, char uplo, int n, int ka, int kb, double *ab, int ldab, double *bb, int ldbb, double *q, int ldq, double vl, double vu, int il, int iu, double abstol, int *m, double *w, double *z, int ldz, int *ifail, int *info);
extern void dsbtrd(char vect, char uplo, int n, int kd, double *ab, int ldab, double *d, double *e, double *q, int ldq, int *info);
extern double dsecnd(void);
extern void dspcon(char uplo, int n, double *a, int *ipiv, double anorm, double *rcond, int *info);
extern void dspev(char jobz, char uplo, int n, double *a, double *w, double *z, int ldz, int *info);
extern void dspevd(char jobz, char uplo, int n, double *ap, double *w, double *z, int ldz, int *info);
extern void dspevx(char jobz, char range, char uplo, int n, double *a, double vl, double vu, int il, int iu, double abtol, int *nfound, double *w, double *z, int ldz, int *ifail, int *info);
extern void dspgst(int itype, char uplo, int n, double *ap, double *bp, int *info);
extern void dspgv(int itype, char jobz, char uplo, int n, double *a, double *b, double *w, double *z, int ldz, int *info);
extern void dspgvd(int itype, char jobz, char uplo, int n, double *ap, double *bp, double *w, double *z, int ldz, int *info);
extern void dspgvx(int itype, char jobz, char range, char uplo, int n, double *ap, double *bp, double vl, double vu, int il, int iu, double abstol, int *m, double *w, double *z, int ldz, int *ifail, int *info);
extern void dsprfs(char uplo, int n, int nrhs, double *a, double *af, int *ipiv, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dspsv(char uplo, int n, int nrhs, double *a, int *ipiv, double *b, int ldb, int *info);
extern void dspsvx(char fact, char uplo, int n, int nrhs, double *a, double *af, int *ipiv, double *b, int ldb, double *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void dsptrd(char uplo, int n, double *ap, double *d, double *e, double *tau, int *info);
extern void dsptrf(char uplo, int n, double *a, int *ipiv, int *info);
extern void dsptri(char uplo, int n, double *a, int *ipiv, int *info);
extern void dsptrs(char uplo, int n, int nrhs, double *a, int *ipiv, double *b, int ldb, int *info);
extern void dstebz(char range, char order, int n, double vl, double vu, int il, int iu, double abstol, double *d, double *e, int *m, int *nsplit, double *w, int *iblock, int *isplit, int *info);
extern void dstedc(char compz, int n, double *d, double *e, double *z, int ldz, int *info);
extern void dstegr(char jobz, char range, int n, double *d, double *e, double vl, double vu, int il, int iu, double abstol, int *m, double *w, double *z, int ldz, int *isuppz, int *info);
extern void dstein(int n, double *d, double *e, int m, double *w, int *iblock, int *isplit, double *z, int ldz, int *ifail, int *info);
extern void dsteqr(char compz, int n, double *d, double *e, double *z, int ldz, int *info);
extern void dsterf(int n, double *d, double *e, int *info);
extern void dstev(char jobz, int n, double *diag, double *offd, double *z, int ldz, int *info);
extern void dstevd(char jobz, int n, double *d, double *e, double *z, int ldz, int *info);
extern void dstevr(char jobz, char range, int n, double *d, double *e, double vl, double vu, int il, int iu, double abstol, int *m, double *w, double *z, int ldz, int *isuppz, int *info);
extern void dstevx(char jobz, char range, int n, double *diag, double *offd, double vl, double vu, int il, int iu, double abtol, int *nfound, double *w, double *z, int ldz, int *ifail, int *info);
extern void dsycon(char uplo, int n, double *a, int lda, int *ipiv, double anorm, double *rcond, int *info);
extern void dsyev(char jobz, char uplo, int n, double *a, int lda, double *w, int *info);
extern void dsyevd(char jobz, char uplo, int n, double *a, int lda, double *w, int *info);
extern void dsyevr(char jobz, char range, char uplo, int n, double *a, int lda, double vl, double vu, int il, int iu, double abstol, int *m, double *w, double *z, int ldz, int *isuppz, int *info);
extern void dsyevx(char jobz, char range, char uplo, int n, double *a, int lda, double vl, double vu, int il, int iu, double abtol, int *nfound, double *w, double *z, int ldz, int *ifail, int *info);
extern void dsygst(int itype, char uplo, int n, double *a, int lda, double *b, int ldb, int *info);
extern void dsygv(int itype, char jobz, char uplo, int n, double *a, int lda, double *b, int ldb, double *w, int *info);
extern void dsygvd(int itype, char jobz, char uplo, int n, double *a, int lda, double *b, int ldb, double *w, int *info);
extern void dsygvx(int itype, char jobz, char range, char uplo, int n, double *a, int lda, double *b, int ldb, double vl, double vu, int il, int iu, double abstol, int *m, double *w, double *z, int ldz, int *ifail, int *info);
extern void dsyrfs(char uplo, int n, int nrhs, double *a, int lda, double *af, int ldaf, int *ipiv, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dsysv(char uplo, int n, int nrhs, double *a, int lda, int *ipiv, double *b, int ldb, int *info);
extern void dsysvx(char fact, char uplo, int n, int nrhs, double *a, int lda, double *af, int ldaf, int *ipiv, double *b, int ldb, double *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void dsytrd(char uplo, int n, double *a, int lda, double *d, double *e, double *tau, int *info);
extern void dsytrf(char uplo, int n, double *a, int lda, int *ipiv, int *info);
extern void dsytri(char uplo, int n, double *a, int lda, int *ipiv, int *info);
extern void dsytrs(char uplo, int n, int nrhs, double *a, int lda, int *ipiv, double *b, int ldb, int *info);
extern void dtbcon(char norm, char uplo, char diag, int n, int ndiag, double *a, int lda, double *rcond, int *info);
extern void dtbrfs(char uplo, char transa, char diag, int n, int ndiag, int nrhs, double *a, int lda, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dtbtrs(char uplo, char transa, char diag, int n, int ndiag, int nrhs, double *a, int lda, double *b, int ldb, int *info);
extern void dtgevc(char side, char howmny, int *select, int n, double *a, int lda, double *b, int ldb, double *vl, int ldvl, double *vr, int ldvr, int mm, int *m, int *info);
extern void dtgexc(int wantq, int wantz, int n, double *a, int lda, double *b, int ldb, double *q, int ldq, double *z, int ldz, int *ifst, int *ilst, int *info);
extern void dtgsen(int ijob, int wantq, int wantz, int *select, int n, double *a, int lda, double *b, int ldb, double *alphar, double *alphai, double *beta, double *q, int ldq, double *z, int ldz, int *m, double *pl, double *pr, double *dif, int *info);
extern void dtgsja(char jobu, char jobv, char jobq, int m, int p, int n, int k, int l, double *a, int lda, double *b, int ldb, double tola, double tolb, double *alpha, double *beta, double *u, int ldu, double *v, int ldv, double *q, int ldq, int *ncycle, int *info);
extern void dtgsna(char job, char howmnt, int *select, int n, double *a, int lda, double *b, int ldb, double *vl, int ldvl, double *vr, int ldvr, double *s, double *dif, int mm, int *m, int *info);
extern void dtgsyl(char trans, int ijob, int m, int n, double *a, int lda, double *b, int ldb, double *c, int ldc, double *d, int ldd, double *e, int lde, double *f, int ldf, double *scale, double *dif, int *info);
extern void dtpcon(char norm, char uplo, char diag, int n, double *a, double *rcond, int *info);
extern void dtprfs(char uplo, char transa, char diag, int n, int nrhs, double *a, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dtptri(char uplo, char diag, int n, double *a, int *info);
extern void dtptrs(char uplo, char transa, char diag, int n, int nrhs, double *a, double *b, int ldb, int *info);
extern void dtrcon(char norm, char uplo, char diag, int n, double *a, int lda, double *rcond, int *info);
extern void dtrevc(char side, char howmny, int *select, int n, double *t, int ldt, double *vl, int ldvl, double *vr, int ldvr, int mm, int *m, int *info);
extern void dtrexc(char compq, int n, double *t, int ldt, double *q, int ldq, int *ifst, int *ilst, int *info);
extern void dtrrfs(char uplo, char transa, char diag, int n, int nrhs, double *a, int lda, double *b, int ldb, double *x, int ldx, double *ferr, double *berr, int *info);
extern void dtrsen(char job, char compq, int *select, int n, double *t, int ldt, double *q, int ldq, double *wr, double *wi, int *m, double *s, double *sep, int *info);
extern void dtrsna(char job, char howmny, int *select, int n, double *t, int ldt, double *vl, int ldvl, double *vr, int ldvr, double *s, double *sep, int mm, int *m, int *info);
extern void dtrsyl(char trana, char tranb, int isgn, int m, int n, double *a, int lda, double *b, int ldb, double *c, int ldc, double *scale, int *info);
extern void dtrtri(char uplo, char diag, int n, double *a, int lda, int *info);
extern void dtrtrs(char uplo, char transa, char diag, int n, int nrhs, double *a, int lda, double *b, int ldb, int *info);
extern void dtzrqf(int m, int n, double *a, int lda, double *tau, int *info);
extern void dtzrzf(int m, int n, double *a, int lda, double *tau, int *info);
extern void sbdsdc(char uplo, char compq, int n, float *d, float *e, float *u, int ldu, float *vt, int ldvt, float *q, int *iq, int *info);
extern void sbdsqr(char uplo, int n, int ncvt, int nru, int ncc, float *d, float *e, float *vt, int ldvt, float *u, int ldu, float *c, int ldc, int *info);
extern void sdisna(char job, int m, int n, float *d, float *sep, int *info);
extern float second(void);
extern void sgbbrd(char vect, int m, int n, int ncc, int kl, int ku, float *ab, int ldab, float *d, float *e, float *q, int ldq, float *pt, int ldpt, float *c, int ldc, int *info);
extern void sgbcon(char norm, int n, int nsub, int nsuper, float *a, int lda, int *ipiv, float anorm, float *rcond, int *info);
extern void sgbequ(int m, int n, int nsub, int nsuper, float *a, int lda, float *rowsc, float *colsc, float *rowcn, float *colcn, float *amax, int *info);
extern void sgbrfs(char transa, int n, int nsub, int nsuper, int nrhs, float *a, int lda, float *af, int ldaf, int *ipiv, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void sgbsv(int n, int nsub, int nsuper, int nrhs, float *a, int lda, int *ipiv, float *b, int ldb, int *info);

/* N.B. In this C interface to sgbsvx, rpvgrw returns what is returned in
   the first element of WORK by the FORTRAN interface */
extern void sgbsvx(char fact, char transa, int n, int nsub, int nsuper, int nrhs, float *a, int lda, float *af, int ldaf, int *ipiv, char equed, float *rowsc, float *colsc, float *b, int ldb, float *x, int ldx, float *rcond, float *ferr, float *berr, float *rpvgrw, int *info);

extern void sgbtrf(int m, int n, int nsub, int nsuper, float *a, int lda, int *ipiv, int *info);
extern void sgbtrs(char transa, int n, int nsub, int nsuper, int nrhs, float *a, int lda, int *ipiv, float *b, int ldb, int *info);
extern void sgebak(char job, char side, int n, int ilo, int ihi, float *scale, int m, float *v, int ldv, int *info);
extern void sgebal(char job, int n, float *a, int lda, int *ilo, int *ihi, float *scale, int *info);
extern void sgebrd(int m, int n, float *a, int lda, float *d, float *e, float *tauq, float *taup, int *info);
extern void sgecon(char norm, int n, float *a, int lda, float anorm, float *rcond, int *info);
extern void sgeequ(int m, int n, float *a, int lda, float *rowsc, float *colsc, float *rowcn, float *colcn, float *amax, int *info);
extern void sgees(char jobz, char sortev, ACML_SGEES_SELECT select, int n, float *a, int lda, int *nout, float *wr, float *wi, float *z, int ldz, int *info);
extern void sgeesx(char jobz, char sortev, ACML_SGEESX_SELECT select, char sense, int n, float *a, int lda, int *nout, float *wr, float *wi, float *z, int ldz, float *srcone, float *rconv, int *info);
extern void sgeev(char jobvl, char jobvr, int n, float *a, int lda, float *wr, float *wi, float *vl, int ldvl, float *vr, int ldvr, int *info);
extern void sgeevx(char balanc, char jobvl, char jobvr, char sense, int n, float *a, int lda, float *wr, float *wi, float *vl, int ldvl, float *vr, int ldvr, int *ilo, int *ihi, float *scale, float *abnrm, float *rcone, float *rconv, int *info);
extern void sgegs(char jobvsl, char jobvsr, int n, float *a, int lda, float *b, int ldb, float *alphar, float *alphai, float *beta, float *vsl, int ldvsl, float *vsr, int ldvsr, int *info);
extern void sgegv(char jobvl, char jobvr, int n, float *a, int lda, float *b, int ldb, float *alphar, float *alphai, float *beta, float *vl, int ldvl, float *vr, int ldvr, int *info);
extern void sgehrd(int n, int ilo, int ihi, float *a, int lda, float *tau, int *info);
extern void sgelqf(int m, int n, float *a, int lda, float *tau, int *info);
extern void sgels(char trans, int m, int n, int nrhs, float *a, int lda, float *b, int ldb, int *info);
extern void sgelsd(int m, int n, int nrhs, float *a, int lda, float *b, int ldb, float *s, float rcond, int *rank, int *info);
extern void sgelss(int m, int n, int nrhs, float *a, int lda, float *b, int ldb, float *sing, float rcond, int *irank, int *info);
extern void sgelsx(int m, int n, int nrhs, float *a, int lda, float *b, int ldb, int *jpivot, float rcond, int *irank, int *info);
extern void sgelsy(int m, int n, int nrhs, float *a, int lda, float *b, int ldb, int *jpvt, float rcond, int *rank, int *info);
extern void sgeqlf(int m, int n, float *a, int lda, float *tau, int *info);
extern void sgeqp3(int m, int n, float *a, int lda, int *jpvt, float *tau, int *info);
extern void sgeqpf(int m, int n, float *a, int lda, int *jpivot, float *tau, int *info);
extern void sgeqrf(int m, int n, float *a, int lda, float *tau, int *info);
extern void sgerfs(char transa, int n, int nrhs, float *a, int lda, float *af, int ldaf, int *ipiv, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void sgerqf(int m, int n, float *a, int lda, float *tau, int *info);
extern void sgesdd(char jobz, int m, int n, float *a, int lda, float *s, float *u, int ldu, float *vt, int ldvt, int *info);
extern void sgesv(int n, int nrhs, float *a, int lda, int *ipiv, float *b, int ldb, int *info);
extern void sgesvd(char jobu, char jobvt, int m, int n, float *a, int lda, float *sing, float *u, int ldu, float *vt, int ldvt, int *info);

/* N.B. In this C interface to sgesvx, rpvgrw returns what is returned in
   the first element of WORK by the FORTRAN interface */
extern void sgesvx(char fact, char transa, int n, int nrhs, float *a, int lda, float *af, int ldaf, int *ipiv, char equed, float *rowsc, float *colsc, float *b, int ldb, float *x, int ldx, float *rcond, float *ferr, float *berr, float *rpvgrw, int *info);

extern void sgetrf(int m, int n, float *a, int lda, int *ipiv, int *info);
extern void sgetri(int n, float *a, int lda, int *ipiv, int *info);
extern void sgetrs(char transa, int n, int nrhs, float *a, int lda, int *ipiv, float *b, int ldb, int *info);
extern void sggbak(char job, char side, int n, int ilo, int ihi, float *lscale, float *rscale, int m, float *v, int ldv, int *info);
extern void sggbal(char job, int n, float *a, int lda, float *b, int ldb, int *ilo, int *ihi, float *lscale, float *rscale, int *info);
extern void sgges(char jobvsl, char jobvsr, char sort, ACML_SGGES_SELCTG selctg, int n, float *a, int lda, float *b, int ldb, int *sdim, float *alphar, float *alphai, float *beta, float *vsl, int ldvsl, float *vsr, int ldvsr, int *info);
extern void sggesx(char jobvsl, char jobvsr, char sort, ACML_SGGESX_SELCTG selctg, char sense, int n, float *a, int lda, float *b, int ldb, int *sdim, float *alphar, float *alphai, float *beta, float *vsl, int ldvsl, float *vsr, int ldvsr, float *rconde, float *rcondv, int *info);
extern void sggev(char jobvl, char jobvr, int n, float *a, int lda, float *b, int ldb, float *alphar, float *alphai, float *beta, float *vl, int ldvl, float *vr, int ldvr, int *info);
extern void sggevx(char balanc, char jobvl, char jobvr, char sense, int n, float *a, int lda, float *b, int ldb, float *alphar, float *alphai, float *beta, float *vl, int ldvl, float *vr, int ldvr, int *ilo, int *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv, int *info);
extern void sggglm(int n, int m, int p, float *a, int lda, float *b, int ldb, float *d, float *x, float *y, int *info);
extern void sgghrd(char compq, char compz, int n, int ilo, int ihi, float *a, int lda, float *b, int ldb, float *q, int ldq, float *z, int ldz, int *info);
extern void sgglse(int m, int n, int p, float *a, int lda, float *b, int ldb, float *c, float *d, float *x, int *info);
extern void sggqrf(int n, int m, int p, float *a, int lda, float *taua, float *b, int ldb, float *taub, int *info);
extern void sggrqf(int m, int p, int n, float *a, int lda, float *taua, float *b, int ldb, float *taub, int *info);
extern void sggsvd(char jobu, char jobv, char jobq, int m, int n, int p, int *k, int *l, float *a, int lda, float *b, int ldb, float *alpha, float *beta, float *u, int ldu, float *v, int ldv, float *q, int ldq, int *iwork3, int *info);
extern void sggsvp(char jobu, char jobv, char jobq, int m, int p, int n, float *a, int lda, float *b, int ldb, float tola, float tolb, int *k, int *l, float *u, int ldu, float *v, int ldv, float *q, int ldq, int *info);
extern void sgtcon(char norm, int n, float *low, float *diag, float *up1, float *up2, int *ipiv, float anorm, float *rcond, int *info);
extern void sgtrfs(char transa, int n, int nrhs, float *low, float *diag, float *up, float *lowf, float *diagf, float *upf1, float *upf2, int *ipiv, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void sgtsv(int n, int nrhs, float *low, float *diag, float *up, float *b, int ldb, int *info);
extern void sgtsvx(char fact, char transa, int n, int nrhs, float *low, float *diag, float *up, float *lowf, float *diagf, float *upf1, float *upf2, int *ipiv, float *b, int ldb, float *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void sgttrf(int n, float *low, float *diag, float *up1, float *up2, int *ipiv, int *info);
extern void sgttrs(char transa, int n, int nrhs, float *low, float *diag, float *up1, float *up2, int *ipiv, float *b, int ldb, int *info);
extern void shgeqz(char job, char compq, char compz, int n, int ilo, int ihi, float *a, int lda, float *b, int ldb, float *alphar, float *alphai, float *beta, float *q, int ldq, float *z, int ldz, int *info);
extern void shsein(char side, char eigsrc, char initv, int *select, int n, float *h, int ldh, float *wr, float *wi, float *vl, int ldvl, float *vr, int ldvr, int mm, int *m, int *ifaill, int *ifailr, int *info);
extern void shseqr(char job, char compz, int n, int ilo, int ihi, float *h, int ldh, float *wr, float *wi, float *z, int ldz, int *info);
extern float slamch(char cmach);
extern void sopgtr(char uplo, int n, float *ap, float *tau, float *q, int ldq, int *info);
extern void sopmtr(char side, char uplo, char trans, int m, int n, float *ap, float *tau, float *c, int ldc, int *info);
extern void sorgbr(char vect, int m, int n, int k, float *a, int lda, float *tau, int *info);
extern void sorghr(int n, int ilo, int ihi, float *a, int lda, float *tau, int *info);
extern void sorglq(int m, int n, int k, float *a, int lda, float *tau, int *info);
extern void sorgql(int m, int n, int k, float *a, int lda, float *tau, int *info);
extern void sorgqr(int m, int n, int k, float *a, int lda, float *tau, int *info);
extern void sorgrq(int m, int n, int k, float *a, int lda, float *tau, int *info);
extern void sorgtr(char uplo, int n, float *a, int lda, float *tau, int *info);
extern void sormbr(char vect, char side, char trans, int m, int n, int k, float *a, int lda, float *tau, float *c, int ldc, int *info);
extern void sormhr(char side, char trans, int m, int n, int ilo, int ihi, float *a, int lda, float *tau, float *c, int ldc, int *info);
extern void sormlq(char side, char trans, int m, int n, int k, float *a, int lda, float *tau, float *c, int ldc, int *info);
extern void sormql(char side, char trans, int m, int n, int k, float *a, int lda, float *tau, float *c, int ldc, int *info);
extern void sormqr(char side, char trans, int m, int n, int k, float *a, int lda, float *tau, float *c, int ldc, int *info);
extern void sormr3(char side, char trans, int m, int n, int k, int l, float *a, int lda, float *tau, float *c, int ldc, int *info);
extern void sormrq(char side, char trans, int m, int n, int k, float *a, int lda, float *tau, float *c, int ldc, int *info);
extern void sormrz(char side, char trans, int m, int n, int k, int l, float *a, int lda, float *tau, float *c, int ldc, int *info);
extern void sormtr(char side, char uplo, char trans, int m, int n, float *a, int lda, float *tau, float *c, int ldc, int *info);
extern void spbcon(char uplo, int n, int ndiag, float *a, int lda, float anorm, float *rcond, int *info);
extern void spbequ(char uplo, int n, int ndiag, float *a, int lda, float *scale, float *scond, float *amax, int *info);
extern void spbrfs(char uplo, int n, int ndiag, int nrhs, float *a, int lda, float *af, int ldaf, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void spbstf(char uplo, int n, int kd, float *ab, int ldab, int *info);
extern void spbsv(char uplo, int n, int ndiag, int nrhs, float *a, int lda, float *b, int ldb, int *info);
extern void spbsvx(char fact, char uplo, int n, int ndiag, int nrhs, float *a, int lda, float *af, int ldaf, char equed, float *scale, float *b, int ldb, float *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void spbtrf(char uplo, int n, int ndiag, float *a, int lda, int *info);
extern void spbtrs(char uplo, int n, int ndiag, int nrhs, float *a, int lda, float *b, int ldb, int *info);
extern void spocon(char uplo, int n, float *a, int lda, float anorm, float *rcond, int *info);
extern void spoequ(int n, float *a, int lda, float *scale, float *scond, float *amax, int *info);
extern void sporfs(char uplo, int n, int nrhs, float *a, int lda, float *af, int ldaf, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void sposv(char uplo, int n, int nrhs, float *a, int lda, float *b, int ldb, int *info);
extern void sposvx(char fact, char uplo, int n, int nrhs, float *a, int lda, float *af, int ldaf, char equed, float *scale, float *b, int ldb, float *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void spotrf(char uplo, int n, float *a, int lda, int *info);
extern void spotri(char uplo, int n, float *a, int lda, int *info);
extern void spotrs(char uplo, int n, int nrhs, float *a, int lda, float *b, int ldb, int *info);
extern void sppcon(char uplo, int n, float *a, float anorm, float *rcond, int *info);
extern void sppequ(char uplo, int n, float *a, float *scale, float *scond, float *amax, int *info);
extern void spprfs(char uplo, int n, int nrhs, float *a, float *af, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void sppsv(char uplo, int n, int nrhs, float *a, float *b, int ldb, int *info);
extern void sppsvx(char fact, char uplo, int n, int nrhs, float *a, float *af, char equed, float *scale, float *b, int ldb, float *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void spptrf(char uplo, int n, float *a, int *info);
extern void spptri(char uplo, int n, float *a, int *info);
extern void spptrs(char uplo, int n, int nrhs, float *a, float *b, int ldb, int *info);
extern void sptcon(int n, float *diag, float *offd, float anorm, float *rcond, int *info);
extern void spteqr(char compz, int n, float *d, float *e, float *z, int ldz, int *info);
extern void sptrfs(int n, int nrhs, float *diag, float *offd, float *diagf, float *offdf, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void sptsv(int n, int nrhs, float *diag, float *sub, float *b, int ldb, int *info);
extern void sptsvx(char fact, int n, int nrhs, float *diag, float *sub, float *diagf, float *subf, float *b, int ldb, float *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void spttrf(int n, float *diag, float *offd, int *info);
extern void spttrs(int n, int nrhs, float *diag, float *offd, float *b, int ldb, int *info);
extern void ssbev(char jobz, char uplo, int n, int ndiag, float *a, int lda, float *w, float *z, int ldz, int *info);
extern void ssbevd(char jobz, char uplo, int n, int kd, float *ab, int ldab, float *w, float *z, int ldz, int *info);
extern void ssbevx(char jobz, char range, char uplo, int n, int ndiag, float *a, int lda, float *q, int ldq, float vl, float vu, int il, int iu, float abtol, int *nfound, float *w, float *z, int ldz, int *ifail, int *info);
extern void ssbgst(char vect, char uplo, int n, int ka, int kb, float *ab, int ldab, float *bb, int ldbb, float *x, int ldx, int *info);
extern void ssbgv(char jobz, char uplo, int n, int ka, int kb, float *ab, int ldab, float *bb, int ldbb, float *w, float *z, int ldz, int *info);
extern void ssbgvd(char jobz, char uplo, int n, int ka, int kb, float *ab, int ldab, float *bb, int ldbb, float *w, float *z, int ldz, int *info);
extern void ssbgvx(char jobz, char range, char uplo, int n, int ka, int kb, float *ab, int ldab, float *bb, int ldbb, float *q, int ldq, float vl, float vu, int il, int iu, float abstol, int *m, float *w, float *z, int ldz, int *ifail, int *info);
extern void ssbtrd(char vect, char uplo, int n, int kd, float *ab, int ldab, float *d, float *e, float *q, int ldq, int *info);
extern void sspcon(char uplo, int n, float *a, int *ipiv, float anorm, float *rcond, int *info);
extern void sspev(char jobz, char uplo, int n, float *a, float *w, float *z, int ldz, int *info);
extern void sspevd(char jobz, char uplo, int n, float *ap, float *w, float *z, int ldz, int *info);
extern void sspevx(char jobz, char range, char uplo, int n, float *a, float vl, float vu, int il, int iu, float abtol, int *nfound, float *w, float *z, int ldz, int *ifail, int *info);
extern void sspgst(int itype, char uplo, int n, float *ap, float *bp, int *info);
extern void sspgv(int itype, char jobz, char uplo, int n, float *a, float *b, float *w, float *z, int ldz, int *info);
extern void sspgvd(int itype, char jobz, char uplo, int n, float *ap, float *bp, float *w, float *z, int ldz, int *info);
extern void sspgvx(int itype, char jobz, char range, char uplo, int n, float *ap, float *bp, float vl, float vu, int il, int iu, float abstol, int *m, float *w, float *z, int ldz, int *ifail, int *info);
extern void ssprfs(char uplo, int n, int nrhs, float *a, float *af, int *ipiv, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void sspsv(char uplo, int n, int nrhs, float *a, int *ipiv, float *b, int ldb, int *info);
extern void sspsvx(char fact, char uplo, int n, int nrhs, float *a, float *af, int *ipiv, float *b, int ldb, float *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void ssptrd(char uplo, int n, float *ap, float *d, float *e, float *tau, int *info);
extern void ssptrf(char uplo, int n, float *a, int *ipiv, int *info);
extern void ssptri(char uplo, int n, float *a, int *ipiv, int *info);
extern void ssptrs(char uplo, int n, int nrhs, float *a, int *ipiv, float *b, int ldb, int *info);
extern void sstebz(char range, char order, int n, float vl, float vu, int il, int iu, float abstol, float *d, float *e, int *m, int *nsplit, float *w, int *iblock, int *isplit, int *info);
extern void sstedc(char compz, int n, float *d, float *e, float *z, int ldz, int *info);
extern void sstegr(char jobz, char range, int n, float *d, float *e, float vl, float vu, int il, int iu, float abstol, int *m, float *w, float *z, int ldz, int *isuppz, int *info);
extern void sstein(int n, float *d, float *e, int m, float *w, int *iblock, int *isplit, float *z, int ldz, int *ifail, int *info);
extern void ssteqr(char compz, int n, float *d, float *e, float *z, int ldz, int *info);
extern void ssterf(int n, float *d, float *e, int *info);
extern void sstev(char jobz, int n, float *diag, float *offd, float *z, int ldz, int *info);
extern void sstevd(char jobz, int n, float *d, float *e, float *z, int ldz, int *info);
extern void sstevr(char jobz, char range, int n, float *d, float *e, float vl, float vu, int il, int iu, float abstol, int *m, float *w, float *z, int ldz, int *isuppz, int *info);
extern void sstevx(char jobz, char range, int n, float *diag, float *offd, float vl, float vu, int il, int iu, float abtol, int *nfound, float *w, float *z, int ldz, int *ifail, int *info);
extern void ssycon(char uplo, int n, float *a, int lda, int *ipiv, float anorm, float *rcond, int *info);
extern void ssyev(char jobz, char uplo, int n, float *a, int lda, float *w, int *info);
extern void ssyevd(char jobz, char uplo, int n, float *a, int lda, float *w, int *info);
extern void ssyevr(char jobz, char range, char uplo, int n, float *a, int lda, float vl, float vu, int il, int iu, float abstol, int *m, float *w, float *z, int ldz, int *isuppz, int *info);
extern void ssyevx(char jobz, char range, char uplo, int n, float *a, int lda, float vl, float vu, int il, int iu, float abtol, int *nfound, float *w, float *z, int ldz, int *ifail, int *info);
extern void ssygst(int itype, char uplo, int n, float *a, int lda, float *b, int ldb, int *info);
extern void ssygv(int itype, char jobz, char uplo, int n, float *a, int lda, float *b, int ldb, float *w, int *info);
extern void ssygvd(int itype, char jobz, char uplo, int n, float *a, int lda, float *b, int ldb, float *w, int *info);
extern void ssygvx(int itype, char jobz, char range, char uplo, int n, float *a, int lda, float *b, int ldb, float vl, float vu, int il, int iu, float abstol, int *m, float *w, float *z, int ldz, int *ifail, int *info);
extern void ssyrfs(char uplo, int n, int nrhs, float *a, int lda, float *af, int ldaf, int *ipiv, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void ssysv(char uplo, int n, int nrhs, float *a, int lda, int *ipiv, float *b, int ldb, int *info);
extern void ssysvx(char fact, char uplo, int n, int nrhs, float *a, int lda, float *af, int ldaf, int *ipiv, float *b, int ldb, float *x, int ldx, float *rcond, float *ferr, float *berr, int *info);
extern void ssytrd(char uplo, int n, float *a, int lda, float *d, float *e, float *tau, int *info);
extern void ssytrf(char uplo, int n, float *a, int lda, int *ipiv, int *info);
extern void ssytri(char uplo, int n, float *a, int lda, int *ipiv, int *info);
extern void ssytrs(char uplo, int n, int nrhs, float *a, int lda, int *ipiv, float *b, int ldb, int *info);
extern void stbcon(char norm, char uplo, char diag, int n, int ndiag, float *a, int lda, float *rcond, int *info);
extern void stbrfs(char uplo, char transa, char diag, int n, int ndiag, int nrhs, float *a, int lda, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void stbtrs(char uplo, char transa, char diag, int n, int ndiag, int nrhs, float *a, int lda, float *b, int ldb, int *info);
extern void stgevc(char side, char howmny, int *select, int n, float *a, int lda, float *b, int ldb, float *vl, int ldvl, float *vr, int ldvr, int mm, int *m, int *info);
extern void stgexc(int wantq, int wantz, int n, float *a, int lda, float *b, int ldb, float *q, int ldq, float *z, int ldz, int *ifst, int *ilst, int *info);
extern void stgsen(int ijob, int wantq, int wantz, int *select, int n, float *a, int lda, float *b, int ldb, float *alphar, float *alphai, float *beta, float *q, int ldq, float *z, int ldz, int *m, float *pl, float *pr, float *dif, int *info);
extern void stgsja(char jobu, char jobv, char jobq, int m, int p, int n, int k, int l, float *a, int lda, float *b, int ldb, float tola, float tolb, float *alpha, float *beta, float *u, int ldu, float *v, int ldv, float *q, int ldq, int *ncycle, int *info);
extern void stgsna(char job, char howmnt, int *select, int n, float *a, int lda, float *b, int ldb, float *vl, int ldvl, float *vr, int ldvr, float *s, float *dif, int mm, int *m, int *info);
extern void stgsyl(char trans, int ijob, int m, int n, float *a, int lda, float *b, int ldb, float *c, int ldc, float *d, int ldd, float *e, int lde, float *f, int ldf, float *scale, float *dif, int *info);
extern void stpcon(char norm, char uplo, char diag, int n, float *a, float *rcond, int *info);
extern void stprfs(char uplo, char transa, char diag, int n, int nrhs, float *a, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void stptri(char uplo, char diag, int n, float *a, int *info);
extern void stptrs(char uplo, char transa, char diag, int n, int nrhs, float *a, float *b, int ldb, int *info);
extern void strcon(char norm, char uplo, char diag, int n, float *a, int lda, float *rcond, int *info);
extern void strevc(char side, char howmny, int *select, int n, float *t, int ldt, float *vl, int ldvl, float *vr, int ldvr, int mm, int *m, int *info);
extern void strexc(char compq, int n, float *t, int ldt, float *q, int ldq, int *ifst, int *ilst, int *info);
extern void strrfs(char uplo, char transa, char diag, int n, int nrhs, float *a, int lda, float *b, int ldb, float *x, int ldx, float *ferr, float *berr, int *info);
extern void strsen(char job, char compq, int *select, int n, float *t, int ldt, float *q, int ldq, float *wr, float *wi, int *m, float *s, float *sep, int *info);
extern void strsna(char job, char howmny, int *select, int n, float *t, int ldt, float *vl, int ldvl, float *vr, int ldvr, float *s, float *sep, int mm, int *m, int *info);
extern void strsyl(char trana, char tranb, int isgn, int m, int n, float *a, int lda, float *b, int ldb, float *c, int ldc, float *scale, int *info);
extern void strtri(char uplo, char diag, int n, float *a, int lda, int *info);
extern void strtrs(char uplo, char transa, char diag, int n, int nrhs, float *a, int lda, float *b, int ldb, int *info);
extern void stzrqf(int m, int n, float *a, int lda, float *tau, int *info);
extern void stzrzf(int m, int n, float *a, int lda, float *tau, int *info);
extern void zbdsqr(char uplo, int n, int ncvt, int nru, int ncc, double *d, double *e, doublecomplex *vt, int ldvt, doublecomplex *u, int ldu, doublecomplex *c, int ldc, int *info);
extern void zgbbrd(char vect, int m, int n, int ncc, int kl, int ku, doublecomplex *ab, int ldab, double *d, double *e, doublecomplex *q, int ldq, doublecomplex *pt, int ldpt, doublecomplex *c, int ldc, int *info);
extern void zgbcon(char norm, int n, int nsub, int nsuper, doublecomplex *a, int lda, int *ipiv, double anorm, double *rcond, int *info);
extern void zgbequ(int m, int n, int nsub, int nsuper, doublecomplex *a, int lda, double *rowsc, double *colsc, double *rowcn, double *colcn, double *amax, int *info);
extern void zgbrfs(char transa, int n, int nsub, int nsuper, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zgbsv(int n, int nsub, int nsuper, int nrhs, doublecomplex *a, int lda, int *ipiv, doublecomplex *b, int ldb, int *info);

/* N.B. In this C interface to zgbsvx, rpvgrw returns what is returned in
   the first element of RWORK by the FORTRAN interface */
extern void zgbsvx(char fact, char transa, int n, int nsub, int nsuper, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, int *ipiv, char equed, double *rowsc, double *colsc, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, double *rpvgrw, int *info);

extern void zgbtrf(int m, int n, int nsub, int nsuper, doublecomplex *a, int lda, int *ipiv, int *info);
extern void zgbtrs(char transa, int n, int nsub, int nsuper, int nrhs, doublecomplex *a, int lda, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zgebak(char job, char side, int n, int ilo, int ihi, double *scale, int m, doublecomplex *v, int ldv, int *info);
extern void zgebal(char job, int n, doublecomplex *a, int lda, int *ilo, int *ihi, double *scale, int *info);
extern void zgebrd(int m, int n, doublecomplex *a, int lda, double *d, double *e, doublecomplex *tauq, doublecomplex *taup, int *info);
extern void zgecon(char norm, int n, doublecomplex *a, int lda, double anorm, double *rcond, int *info);
extern void zgeequ(int m, int n, doublecomplex *a, int lda, double *rowsc, double *colsc, double *rowcn, double *colcn, double *amax, int *info);
extern void zgees(char jobvs, char sort, ACML_ZGEES_SELECT select, int n, doublecomplex *a, int lda, int *sdim, doublecomplex *w, doublecomplex *vs, int ldvs, int *info);
extern void zgeesx(char jobvs, char sort, ACML_ZGEESX_SELECT select, char sense, int n, doublecomplex *a, int lda, int *sdim, doublecomplex *w, doublecomplex *vs, int ldvs, double *rconde, double *rcondv, int *info);
extern void zgeev(char jobvl, char jobvr, int n, doublecomplex *a, int lda, doublecomplex *w, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, int *info);
extern void zgeevx(char balanc, char jobvl, char jobvr, char sense, int n, doublecomplex *a, int lda, doublecomplex *w, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, int *ilo, int *ihi, double *scale, double *abnrm, double *rconde, double *rcondv, int *info);
extern void zgegs(char jobvsl, char jobvsr, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vsl, int ldvsl, doublecomplex *vsr, int ldvsr, int *info);
extern void zgegv(char jobvl, char jobvr, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, int *info);
extern void zgehrd(int n, int ilo, int ihi, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zgelqf(int m, int n, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zgels(char trans, int m, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *info);
extern void zgelsd(int m, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, double *s, double rcond, int *rank, int *info);
extern void zgelss(int m, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, double *sing, double rcond, int *irank, int *info);
extern void zgelsx(int m, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *jpivot, double rcond, int *irank, int *info);
extern void zgelsy(int m, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *jpvt, double rcond, int *rank, int *info);
extern void zgeqlf(int m, int n, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zgeqp3(int m, int n, doublecomplex *a, int lda, int *jpvt, doublecomplex *tau, int *info);
extern void zgeqpf(int m, int n, doublecomplex *a, int lda, int *jpvt, doublecomplex *tau, int *info);
extern void zgeqrf(int m, int n, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zgerfs(char transa, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zgerqf(int m, int n, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zgesdd(char jobz, int m, int n, doublecomplex *a, int lda, double *s, doublecomplex *u, int ldu, doublecomplex *vt, int ldvt, int *info);
extern void zgesv(int n, int nrhs, doublecomplex *a, int lda, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zgesvd(char jobu, char jobvt, int m, int n, doublecomplex *a, int lda, double *sing, doublecomplex *u, int ldu, doublecomplex *vt, int ldvt, int *info);

/* N.B. In this C interface to zgesvx, rpvgrw returns what is returned in
   the first element of RWORK by the FORTRAN interface */
extern void zgesvx(char fact, char transa, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, int *ipiv, char equed, double *rowsc, double *colsc, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, double *rpvgrw, int *info);

extern void zgetrf(int m, int n, doublecomplex *a, int lda, int *ipiv, int *info);
extern void zgetri(int n, doublecomplex *a, int lda, int *ipiv, int *info);
extern void zgetrs(char transa, int n, int nrhs, doublecomplex *a, int lda, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zggbak(char job, char side, int n, int ilo, int ihi, double *lscale, double *rscale, int m, doublecomplex *v, int ldv, int *info);
extern void zggbal(char job, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *ilo, int *ihi, double *lscale, double *rscale, int *info);
extern void zgges(char jobvsl, char jobvsr, char sort, ACML_ZGGES_DELCTG delctg, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *sdim, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vsl, int ldvsl, doublecomplex *vsr, int ldvsr, int *info);
extern void zggesx(char jobvsl, char jobvsr, char sort, ACML_ZGGESX_DELCTG delctg, char sense, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *sdim, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vsl, int ldvsl, doublecomplex *vsr, int ldvsr, double *rconde, double *rcondv, int *info);
extern void zggev(char jobvl, char jobvr, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, int *info);
extern void zggevx(char balanc, char jobvl, char jobvr, char sense, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, int *ilo, int *ihi, double *lscale, double *rscale, double *abnrm, double *bbnrm, double *rconde, double *rcondv, int *info);
extern void zggglm(int n, int m, int p, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *d, doublecomplex *x, doublecomplex *y, int *info);
extern void zgghrd(char compq, char compz, int n, int ilo, int ihi, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *q, int ldq, doublecomplex *z, int ldz, int *info);
extern void zgglse(int m, int n, int p, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *c, doublecomplex *d, doublecomplex *x, int *info);
extern void zggqrf(int n, int m, int p, doublecomplex *a, int lda, doublecomplex *taua, doublecomplex *b, int ldb, doublecomplex *taub, int *info);
extern void zggrqf(int m, int p, int n, doublecomplex *a, int lda, doublecomplex *taua, doublecomplex *b, int ldb, doublecomplex *taub, int *info);
extern void zggsvd(char jobu, char jobv, char jobq, int m, int n, int p, int *k, int *l, doublecomplex *a, int lda, doublecomplex *b, int ldb, double *alpha, double *beta, doublecomplex *u, int ldu, doublecomplex *v, int ldv, doublecomplex *q, int ldq, int *iwork3, int *info);
extern void zggsvp(char jobu, char jobv, char jobq, int m, int p, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, double tola, double tolb, int *k, int *l, doublecomplex *u, int ldu, doublecomplex *v, int ldv, doublecomplex *q, int ldq, int *info);
extern void zgtcon(char norm, int n, doublecomplex *low, doublecomplex *diag, doublecomplex *up1, doublecomplex *up2, int *ipiv, double anorm, double *rcond, int *info);
extern void zgtrfs(char transa, int n, int nrhs, doublecomplex *low, doublecomplex *diag, doublecomplex *up, doublecomplex *lowf, doublecomplex *diagf, doublecomplex *upf1, doublecomplex *upf2, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zgtsv(int n, int nrhs, doublecomplex *low, doublecomplex *diag, doublecomplex *up, doublecomplex *b, int ldb, int *info);
extern void zgtsvx(char fact, char transa, int n, int nrhs, doublecomplex *low, doublecomplex *diag, doublecomplex *up, doublecomplex *lowf, doublecomplex *diagf, doublecomplex *upf1, doublecomplex *upf2, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void zgttrf(int n, doublecomplex *low, doublecomplex *diag, doublecomplex *up1, doublecomplex *up2, int *ipiv, int *info);
extern void zgttrs(char transa, int n, int nrhs, doublecomplex *low, doublecomplex *diag, doublecomplex *up1, doublecomplex *up2, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zhbev(char jobz, char uplo, int n, int ndiag, doublecomplex *a, int lda, double *w, doublecomplex *z, int ldz, int *info);
extern void zhbevd(char jobz, char uplo, int n, int kd, doublecomplex *ab, int ldab, double *w, doublecomplex *z, int ldz, int *info);
extern void zhbevx(char jobz, char range, char uplo, int n, int ndiag, doublecomplex *a, int lda, doublecomplex *q, int ldq, double vl, double vu, int il, int iu, double abtol, int *nfound, double *w, doublecomplex *z, int ldz, int *ifail, int *info);
extern void zhbgst(char vect, char uplo, int n, int ka, int kb, doublecomplex *ab, int ldab, doublecomplex *bb, int ldbb, doublecomplex *x, int ldx, int *info);
extern void zhbgv(char jobz, char uplo, int n, int ka, int kb, doublecomplex *ab, int ldab, doublecomplex *bb, int ldbb, double *w, doublecomplex *z, int ldz, int *info);
extern void zhbgvd(char jobz, char uplo, int n, int ka, int kb, doublecomplex *ab, int ldab, doublecomplex *bb, int ldbb, double *w, doublecomplex *z, int ldz, int *info);
extern void zhbgvx(char jobz, char range, char uplo, int n, int ka, int kb, doublecomplex *ab, int ldab, doublecomplex *bb, int ldbb, doublecomplex *q, int ldq, double vl, double vu, int il, int iu, double abstol, int *m, double *w, doublecomplex *z, int ldz, int *ifail, int *info);
extern void zhbtrd(char vect, char uplo, int n, int kd, doublecomplex *ab, int ldab, double *d, double *e, doublecomplex *q, int ldq, int *info);
extern void zhecon(char uplo, int n, doublecomplex *a, int lda, int *ipiv, double anorm, double *rcond, int *info);
extern void zheev(char jobz, char uplo, int n, doublecomplex *a, int lda, double *w, int *info);
extern void zheevd(char jobz, char uplo, int n, doublecomplex *a, int lda, double *w, int *info);
extern void zheevr(char jobz, char range, char uplo, int n, doublecomplex *a, int lda, double vl, double vu, int il, int iu, double abstol, int *m, double *w, doublecomplex *z, int ldz, int *isuppz, int *info);
extern void zheevx(char jobz, char range, char uplo, int n, doublecomplex *a, int lda, double vl, double vu, int il, int iu, double abtol, int *nfound, double *w, doublecomplex *z, int ldz, int *ifail, int *info);
extern void zhegst(int itype, char uplo, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *info);
extern void zhegv(int itype, char jobz, char uplo, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, double *w, int *info);
extern void zhegvd(int itype, char jobz, char uplo, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, double *w, int *info);
extern void zhegvx(int itype, char jobz, char range, char uplo, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, double vl, double vu, int il, int iu, double abstol, int *m, double *w, doublecomplex *z, int ldz, int *ifail, int *info);
extern void zherfs(char uplo, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zhesv(char uplo, int n, int nrhs, doublecomplex *a, int lda, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zhesvx(char fact, char uplo, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void zhetrd(char uplo, int n, doublecomplex *a, int lda, double *d, double *e, doublecomplex *tau, int *info);
extern void zhetrf(char uplo, int n, doublecomplex *a, int lda, int *ipiv, int *info);
extern void zhetri(char uplo, int n, doublecomplex *a, int lda, int *ipiv, int *info);
extern void zhetrs(char uplo, int n, int nrhs, doublecomplex *a, int lda, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zhgeqz(char job, char compq, char compz, int n, int ilo, int ihi, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *q, int ldq, doublecomplex *z, int ldz, int *info);
extern void zhpcon(char uplo, int n, doublecomplex *a, int *ipiv, double anorm, double *rcond, int *info);
extern void zhpev(char jobz, char uplo, int n, doublecomplex *a, double *w, doublecomplex *z, int ldz, int *info);
extern void zhpevd(char jobz, char uplo, int n, doublecomplex *ap, double *w, doublecomplex *z, int ldz, int *info);
extern void zhpevx(char jobz, char range, char uplo, int n, doublecomplex *a, double vl, double vu, int il, int iu, double abtol, int *nfound, double *w, doublecomplex *z, int ldz, int *ifail, int *info);
extern void zhpgst(int itype, char uplo, int n, doublecomplex *ap, doublecomplex *bp, int *info);
extern void zhpgv(int itype, char jobz, char uplo, int n, doublecomplex *a, doublecomplex *b, double *w, doublecomplex *z, int ldz, int *info);
extern void zhpgvd(int itype, char jobz, char uplo, int n, doublecomplex *ap, doublecomplex *bp, double *w, doublecomplex *z, int ldz, int *info);
extern void zhpgvx(int itype, char jobz, char range, char uplo, int n, doublecomplex *ap, doublecomplex *bp, double vl, double vu, int il, int iu, double abstol, int *m, double *w, doublecomplex *z, int ldz, int *ifail, int *info);
extern void zhprfs(char uplo, int n, int nrhs, doublecomplex *a, doublecomplex *af, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zhpsv(char uplo, int n, int nrhs, doublecomplex *a, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zhpsvx(char fact, char uplo, int n, int nrhs, doublecomplex *a, doublecomplex *af, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void zhptrd(char uplo, int n, doublecomplex *ap, double *d, double *e, doublecomplex *tau, int *info);
extern void zhptrf(char uplo, int n, doublecomplex *a, int *ipiv, int *info);
extern void zhptri(char uplo, int n, doublecomplex *a, int *ipiv, int *info);
extern void zhptrs(char uplo, int n, int nrhs, doublecomplex *a, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zhsein(char side, char eigsrc, char initv, int *select, int n, doublecomplex *h, int ldh, doublecomplex *w, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, int mm, int *m, int *ifaill, int *ifailr, int *info);
extern void zhseqr(char job, char compz, int n, int ilo, int ihi, doublecomplex *h, int ldh, doublecomplex *w, doublecomplex *z, int ldz, int *info);
extern void zpbcon(char uplo, int n, int ndiag, doublecomplex *a, int lda, double anorm, double *rcond, int *info);
extern void zpbequ(char uplo, int n, int ndiag, doublecomplex *a, int lda, double *scale, double *scond, double *amax, int *info);
extern void zpbrfs(char uplo, int n, int ndiag, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zpbstf(char uplo, int n, int kd, doublecomplex *ab, int ldab, int *info);
extern void zpbsv(char uplo, int n, int ndiag, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *info);
extern void zpbsvx(char fact, char uplo, int n, int ndiag, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, char equed, double *scale, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void zpbtrf(char uplo, int n, int ndiag, doublecomplex *a, int lda, int *info);
extern void zpbtrs(char uplo, int n, int ndiag, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *info);
extern void zpocon(char uplo, int n, doublecomplex *a, int lda, double anorm, double *rcond, int *info);
extern void zpoequ(int n, doublecomplex *a, int lda, double *scale, double *scond, double *amax, int *info);
extern void zporfs(char uplo, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zposv(char uplo, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *info);
extern void zposvx(char fact, char uplo, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, char equed, double *scale, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void zpotrf(char uplo, int n, doublecomplex *a, int lda, int *info);
extern void zpotri(char uplo, int n, doublecomplex *a, int lda, int *info);
extern void zpotrs(char uplo, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *info);
extern void zppcon(char uplo, int n, doublecomplex *a, double anorm, double *rcond, int *info);
extern void zppequ(char uplo, int n, doublecomplex *a, double *scale, double *scond, double *amax, int *info);
extern void zpprfs(char uplo, int n, int nrhs, doublecomplex *a, doublecomplex *af, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zppsv(char uplo, int n, int nrhs, doublecomplex *a, doublecomplex *b, int ldb, int *info);
extern void zppsvx(char fact, char uplo, int n, int nrhs, doublecomplex *a, doublecomplex *af, char equed, double *scale, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void zpptrf(char uplo, int n, doublecomplex *a, int *info);
extern void zpptri(char uplo, int n, doublecomplex *a, int *info);
extern void zpptrs(char uplo, int n, int nrhs, doublecomplex *a, doublecomplex *b, int ldb, int *info);
extern void zptcon(int n, double *diag, doublecomplex *offd, double anorm, double *rcond, int *info);
extern void zpteqr(char compz, int n, double *d, double *e, doublecomplex *z, int ldz, int *info);
extern void zptrfs(char uplo, int n, int nrhs, double *diag, doublecomplex *offd, double *diagf, doublecomplex *offdf, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zptsv(int n, int nrhs, double *diag, doublecomplex *sub, doublecomplex *b, int ldb, int *info);
extern void zptsvx(char fact, int n, int nrhs, double *diag, doublecomplex *sub, double *diagf, doublecomplex *subf, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void zpttrf(int n, double *diag, doublecomplex *offd, int *info);
extern void zpttrs(char uplo, int n, int nrhs, double *diag, doublecomplex *offd, doublecomplex *b, int ldb, int *info);
extern void zspcon(char uplo, int n, doublecomplex *a, int *ipiv, double anorm, double *rcond, int *info);
extern void zsprfs(char uplo, int n, int nrhs, doublecomplex *a, doublecomplex *af, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zspsv(char uplo, int n, int nrhs, doublecomplex *a, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zspsvx(char fact, char uplo, int n, int nrhs, doublecomplex *a, doublecomplex *af, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void zsptrf(char uplo, int n, doublecomplex *a, int *ipiv, int *info);
extern void zsptri(char uplo, int n, doublecomplex *a, int *ipiv, int *info);
extern void zsptrs(char uplo, int n, int nrhs, doublecomplex *a, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zstedc(char compz, int n, double *d, double *e, doublecomplex *z, int ldz, int *info);
extern void zstegr(char jobz, char range, int n, double *d, double *e, double vl, double vu, int il, int iu, double abstol, int *m, double *w, doublecomplex *z, int ldz, int *isuppz, int *info);
extern void zstein(int n, double *d, double *e, int m, double *w, int *iblock, int *isplit, doublecomplex *z, int ldz, int *ifail, int *info);
extern void zsteqr(char compz, int n, double *d, double *e, doublecomplex *z, int ldz, int *info);
extern void zsycon(char uplo, int n, doublecomplex *a, int lda, int *ipiv, double anorm, double *rcond, int *info);
extern void zsyrfs(char uplo, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void zsysv(char uplo, int n, int nrhs, doublecomplex *a, int lda, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void zsysvx(char fact, char uplo, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *af, int ldaf, int *ipiv, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *rcond, double *ferr, double *berr, int *info);
extern void zsytrf(char uplo, int n, doublecomplex *a, int lda, int *ipiv, int *info);
extern void zsytri(char uplo, int n, doublecomplex *a, int lda, int *ipiv, int *info);
extern void zsytrs(char uplo, int n, int nrhs, doublecomplex *a, int lda, int *ipiv, doublecomplex *b, int ldb, int *info);
extern void ztbcon(char norm, char uplo, char diag, int n, int ndiag, doublecomplex *a, int lda, double *rcond, int *info);
extern void ztbrfs(char uplo, char transa, char diag, int n, int ndiag, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void ztbtrs(char uplo, char transa, char diag, int n, int ndiag, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *info);
extern void ztgevc(char side, char howmny, int *select, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, int mm, int *m, int *info);
extern void ztgexc(int wantq, int wantz, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *q, int ldq, doublecomplex *z, int ldz, int *ifst, int *ilst, int *info);
extern void ztgsen(int ijob, int wantq, int wantz, int *select, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *q, int ldq, doublecomplex *z, int ldz, int *m, double *pl, double *pr, double *dif, int *info);
extern void ztgsja(char jobu, char jobv, char jobq, int m, int p, int n, int k, int l, doublecomplex *a, int lda, doublecomplex *b, int ldb, double tola, double tolb, double *alpha, double *beta, doublecomplex *u, int ldu, doublecomplex *v, int ldv, doublecomplex *q, int ldq, int *ncycle, int *info);
extern void ztgsna(char job, char howmnt, int *select, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, double *s, double *dif, int mm, int *m, int *info);
extern void ztgsyl(char trans, int ijob, int m, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *c, int ldc, doublecomplex *d, int ldd, doublecomplex *e, int lde, doublecomplex *f, int ldf, double *scale, double *dif, int *info);
extern void ztpcon(char norm, char uplo, char diag, int n, doublecomplex *a, double *rcond, int *info);
extern void ztprfs(char uplo, char transa, char diag, int n, int nrhs, doublecomplex *a, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void ztptri(char uplo, char diag, int n, doublecomplex *a, int *info);
extern void ztptrs(char uplo, char transa, char diag, int n, int nrhs, doublecomplex *a, doublecomplex *b, int ldb, int *info);
extern void ztrcon(char norm, char uplo, char diag, int n, doublecomplex *a, int lda, double *rcond, int *info);
extern void ztrevc(char side, char howmny, int *select, int n, doublecomplex *t, int ldt, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, int mm, int *m, int *info);
extern void ztrexc(char compq, int n, doublecomplex *t, int ldt, doublecomplex *q, int ldq, int ifst, int ilst, int *info);
extern void ztrrfs(char uplo, char transa, char diag, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *x, int ldx, double *ferr, double *berr, int *info);
extern void ztrsen(char job, char compq, int *select, int n, doublecomplex *t, int ldt, doublecomplex *q, int ldq, doublecomplex *w, int *m, double *s, double *sep, int *info);
extern void ztrsna(char job, char howmny, int *select, int n, doublecomplex *t, int ldt, doublecomplex *vl, int ldvl, doublecomplex *vr, int ldvr, double *s, double *sep, int mm, int *m, int *info);
extern void ztrsyl(char trana, char tranb, int isgn, int m, int n, doublecomplex *a, int lda, doublecomplex *b, int ldb, doublecomplex *c, int ldc, double *scale, int *info);
extern void ztrtri(char uplo, char diag, int n, doublecomplex *a, int lda, int *info);
extern void ztrtrs(char uplo, char transa, char diag, int n, int nrhs, doublecomplex *a, int lda, doublecomplex *b, int ldb, int *info);
extern void ztzrqf(int m, int n, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void ztzrzf(int m, int n, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zungbr(char vect, int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zunghr(int n, int ilo, int ihi, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zunglq(int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zungql(int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zungqr(int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zungrq(int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zungtr(char uplo, int n, doublecomplex *a, int lda, doublecomplex *tau, int *info);
extern void zunmbr(char vect, char side, char trans, int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, doublecomplex *c, int ldc, int *info);
extern void zunmhr(char side, char trans, int m, int n, int ilo, int ihi, doublecomplex *a, int lda, doublecomplex *tau, doublecomplex *c, int ldc, int *info);
extern void zunmlq(char side, char trans, int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, doublecomplex *c, int ldc, int *info);
extern void zunmql(char side, char trans, int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, doublecomplex *c, int ldc, int *info);
extern void zunmqr(char side, char trans, int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, doublecomplex *c, int ldc, int *info);
extern void zunmrq(char side, char trans, int m, int n, int k, doublecomplex *a, int lda, doublecomplex *tau, doublecomplex *c, int ldc, int *info);
extern void zunmrz(char side, char trans, int m, int n, int k, int l, doublecomplex *a, int lda, doublecomplex *tau, doublecomplex *c, int ldc, int *info);
extern void zunmtr(char side, char uplo, char trans, int m, int n, doublecomplex *a, int lda, doublecomplex *tau, doublecomplex *c, int ldc, int *info);
extern void zupgtr(char uplo, int n, doublecomplex *ap, doublecomplex *tau, doublecomplex *q, int ldq, int *info);
extern void zupmtr(char side, char uplo, char trans, int m, int n, doublecomplex *ap, doublecomplex *tau, doublecomplex *c, int ldc, int *info);

/* FFT C interfaces */
extern void cfft1d(int mode, int n, complex *x, complex *comm, int *info);
extern void cfft1dx(int mode, float scale, int inpl, int n, complex *x, int incx, complex *y, int incy, complex *comm, int *info);
extern void cfft1m(int mode, int nseq, int n, complex *x, complex *comm, int *info);
extern void cfft1mx(int mode, float scale, int inpl, int nseq, int n, complex *x, int incx1, int incx2, complex *y, int incy1, int incy2, complex *comm, int *info);
extern void cfft2d(int mode, int m, int n, complex *x, complex *comm, int *info);
extern void cfft2dx(int mode, float scale, int ltrans, int inpl, int m, int n, complex *x, int incx1, int incx2, complex *y, int incy1, int incy2, complex *comm, int *info);
extern void cfft3d(int mode, int l, int m, int n, complex *x, complex *comm, int *info);
extern void cfft3dx(int mode, float scale, int ltrans, int inpl, int l, int m, int n, complex *x, complex *y, complex *comm, int *info);
extern void cfft3dy(int mode, float scale, int inpl, int l, int m, int n, complex *x, int incx1, int incx2, int incx3, complex *y, int incy1, int incy2, int incy3, complex *comm, int lcomm, int *info);
extern void csfft(int mode, int n, float *x, float *comm, int *info);
extern void csfftm(int nseq, int n, float *x, float *comm, int *info);
extern void dzfft(int mode, int n, double *x, double *comm, int *info);
extern void dzfftm(int nseq, int n, double *x, double *comm, int *info);
extern void scfft(int mode, int n, float *x, float *comm, int *info);
extern void scfftm(int nseq, int n, float *x, float *comm, int *info);
extern void zdfft(int mode, int n, double *x, double *comm, int *info);
extern void zdfftm(int nseq, int n, double *x, double *comm, int *info);
extern void zfft1d(int mode, int n, doublecomplex *x, doublecomplex *comm, int *info);
extern void zfft1dx(int mode, double scale, int inpl, int n, doublecomplex *x, int incx, doublecomplex *y, int incy, doublecomplex *comm, int *info);
extern void zfft1m(int mode, int nseq, int n, doublecomplex *x, doublecomplex *comm, int *info);
extern void zfft1mx(int mode, double scale, int inpl, int nseq, int n, doublecomplex *x, int incx1, int incx2, doublecomplex *y, int incy1, int incy2, doublecomplex *comm, int *info);
extern void zfft2d(int mode, int m, int n, doublecomplex *x, doublecomplex *comm, int *info);
extern void zfft2dx(int mode, double scale, int ltrans, int inpl, int m, int n, doublecomplex *x, int incx1, int incx2, doublecomplex *y, int incy1, int incy2, doublecomplex *comm, int *info);
extern void zfft3d(int mode, int l, int m, int n, doublecomplex *x, doublecomplex *comm, int *info);
extern void zfft3dx(int mode, double scale, int ltrans, int inpl, int l, int m, int n, doublecomplex *x, doublecomplex *y, doublecomplex *comm, int *info);
extern void zfft3dy(int mode, double scale, int inpl, int l, int m, int n, doublecomplex *x, int incx1, int incx2, int incx3, doublecomplex *y, int incy1, int incy2, int incy3, doublecomplex *comm, int lcomm, int *info);

/* Sparse BLAS interfaces */
extern void caxpyi(int nz, complex a, complex *x, int *indx, complex *y);
extern complex cdotci(int nz, complex *x, int *indx, complex *y);
extern complex cdotui(int nz, complex *x, int *indx, complex *y);
extern void cgthr(int nz, complex *y, complex *x, int *indx);
extern void cgthrz(int nz, complex *y, complex *x, int *indx);
extern void csctr(int nz, complex *x, int *indx, complex *y);
extern void daxpyi(int nz, double a, double *x, int *indx, double *y);
extern double ddoti(int nz, double *x, int *indx, double *y);
extern void dgthr(int nz, double *y, double *x, int *indx);
extern void dgthrz(int nz, double *y, double *x, int *indx);
extern void droti(int nz, double *x, int *indx, double *y, double c, double s);
extern void dsctr(int nz, double *x, int *indx, double *y);
extern void saxpyi(int nz, float a, float *x, int *indx, float *y);
extern float sdoti(int nz, float *x, int *indx, float *y);
extern void sgthr(int nz, float *y, float *x, int *indx);
extern void sgthrz(int nz, float *y, float *x, int *indx);
extern void sroti(int nz, float *x, int *indx, float *y, float c, float s);
extern void ssctr(int nz, float *x, int *indx, float *y);
extern void zaxpyi(int nz, doublecomplex a, doublecomplex *x, int *indx, doublecomplex *y);
extern doublecomplex zdotci(int nz, doublecomplex *x, int *indx, doublecomplex *y);
extern doublecomplex zdotui(int nz, doublecomplex *x, int *indx, doublecomplex *y);
extern void zgthr(int nz, doublecomplex *y, doublecomplex *x, int *indx);
extern void zgthrz(int nz, doublecomplex *y, doublecomplex *x, int *indx);
extern void zsctr(int nz, doublecomplex *x, int *indx, doublecomplex *y);

/* Random number generators */
extern void drandbeta(int n, double a, double b, int *state, double *x, int *info);
extern void drandbinomial(int n, int m, double p, int *state, int *x, int *info);
extern void drandbinomialreference(int m, double p, double *ref, int *lref, int *info);
extern void drandblumblumshub(int n, int *state, double *x, int *info);
extern void drandcauchy(int n, double a, double b, int *state, double *x, int *info);
extern void drandchisquared(int n, int df, int *state, double *x, int *info);
extern void dranddiscreteuniform(int n, int a, int b, int *state, int *x, int *info);
extern void drandexponential(int n, double a, int *state, double *x, int *info);
extern void drandf(int n, int df1, int df2, int *state, double *x, int *info);
extern void drandgamma(int n, double a, double b, int *state, double *x, int *info);
extern void drandgaussian(int n, double xmu, double var, int *state, double *x, int *info);
extern void drandgeneraldiscrete(int n, double *ref, int *state, int *x, int *info);
extern void drandgeometric(int n, double p, int *state, int *x, int *info);
extern void drandgeometricreference(double p, double *ref, int *lref, int *info);
extern void drandhypergeometric(int n, int np, int ns, int m, int *state, int *x, int *info);
extern void drandhypergeometricreference(int np, int ns, int m, double *ref, int *lref, int *info);
extern void drandinitialize(int genid, int subid, int *seed, int *lseed, int *state, int *lstate, int *info);
extern void drandinitializebbs(int nbits, int lp, int *p, int lq, int *q, int ls, int *s, int *state, int *lstate, int *info);
extern void drandinitializeuser(ACML_DRANDINITIALIZEUSER_UINI uini, ACML_DRANDINITIALIZEUSER_UGEN ugen, int genid, int subid, int *seed, int *lseed, int *state, int *lstate, int *info);
extern void drandleapfrog(int n, int k, int *state, int *info);
extern void drandlogistic(int n, double a, double b, int *state, double *x, int *info);
extern void drandlognormal(int n, double xmu, double var, int *state, double *x, int *info);
extern void drandmultinomial(int n, int m, double *p, int k, int *state, int *x, int ldx, int *info);
extern void drandmultinormalreference(int m, double *xmu, double *c, int ldc, double *ref, int *lref, int *info);
extern void drandmultinormalr(int n, double *ref, int *state, double *x, int ldx, int *info);
extern void drandmultistudentstreference(int m, int df, double *xmu, double *c, int ldc, double *ref, int *lref, int *info);
extern void drandmultistudentstr(int n, double *ref, int *state, double *x, int ldx, int *info);
extern void drandnegativebinomial(int n, int m, double p, int *state, int *x, int *info);
extern void drandnegativebinomialreference(int m, double p, double *ref, int *lref, int *info);
extern void drandpoisson(int n, double lambda, int *state, int *x, int *info);
extern void drandpoissonreference(double lambda, double *ref, int *lref, int *info);
extern void drandskipahead(int n, int *state, int *info);
extern void drandstudentst(int n, int df, int *state, double *x, int *info);
extern void drandtriangular(int n, double xmin, double xmed, double xmax, int *state, double *x, int *info);
extern void dranduniform(int n, double a, double b, int *state, double *x, int *info);
extern void drandvonmises(int n, double vk, int *state, double *x, int *info);
extern void drandweibull(int n, double a, double b, int *state, double *x, int *info);
extern void drandmultinormal(int n, int m, double *xmu, double *c, int ldc, int *state, double *x, int ldx, int *info);
extern void drandmultistudentst(int n, int m, int df, double *xmu, double *c, int ldc, int *state, double *x, int ldx, int *info);
extern void srandbeta(int n, float a, float b, int *state, float *x, int *info);
extern void srandblumblumshub(int n, int *state, float *x, int *info);
extern void srandbinomial(int n, int m, float p, int *state, int *x, int *info);
extern void srandbinomialreference(int m, float p, float *ref, int *lref, int *info);
extern void srandcauchy(int n, float a, float b, int *state, float *x, int *info);
extern void srandchisquared(int n, int df, int *state, float *x, int *info);
extern void sranddiscreteuniform(int n, int a, int b, int *state, int *x, int *info);
extern void srandexponential(int n, float a, int *state, float *x, int *info);
extern void srandf(int n, int df1, int df2, int *state, float *x, int *info);
extern void srandgamma(int n, float a, float b, int *state, float *x, int *info);
extern void srandgaussian(int n, float xmu, float var, int *state, float *x, int *info);
extern void srandgeneraldiscrete(int n, float *ref, int *state, int *x, int *info);
extern void srandgeometric(int n, float p, int *state, int *x, int *info);
extern void srandgeometricreference(float p, float *ref, int *lref, int *info);
extern void srandhypergeometric(int n, int np, int ns, int m, int *state, int *x, int *info);
extern void srandhypergeometricreference(int np, int ns, int m, float *ref, int *lref, int *info);
extern void srandinitialize(int genid, int subid, int *seed, int *lseed, int *state, int *lstate, int *info);
extern void srandinitializebbs(int nbits, int lp, int *p, int lq, int *q, int ls, int *s, int *state, int *lstate, int *info);
extern void srandinitializeuser(ACML_SRANDINITIALIZEUSER_UINI uini, ACML_SRANDINITIALIZEUSER_UGEN ugen, int genid, int subid, int *seed, int *lseed, int *state, int *lstate, int *info);
extern void srandleapfrog(int n, int k, int *state, int *info);
extern void srandlogistic(int n, float a, float b, int *state, float *x, int *info);
extern void srandlognormal(int n, float xmu, float var, int *state, float *x, int *info);
extern void srandmultinomial(int n, int m, float *p, int k, int *state, int *x, int ldx, int *info);
extern void srandmultinormalreference(int m, float *xmu, float *c, int ldc, float *ref, int *lref, int *info);
extern void srandmultinormalr(int n, float *ref, int *state, float *x, int ldx, int *info);
extern void srandmultistudentstreference(int m, int df, float *xmu, float *c, int ldc, float *ref, int *lref, int *info);
extern void srandmultistudentstr(int n, float *ref, int *state, float *x, int ldx, int *info);
extern void srandnegativebinomial(int n, int m, float p, int *state, int *x, int *info);
extern void srandnegativebinomialreference(int m, float p, float *ref, int *lref, int *info);
extern void srandpoisson(int n, float lambda, int *state, int *x, int *info);
extern void srandpoissonreference(float lambda, float *ref, int *lref, int *info);
extern void srandskipahead(int n, int *state, int *info);
extern void srandstudentst(int n, int df, int *state, float *x, int *info);
extern void srandtriangular(int n, float xmin, float xmed, float xmax, int *state, float *x, int *info);
extern void sranduniform(int n, float a, float b, int *state, float *x, int *info);
extern void srandvonmises(int n, float vk, int *state, float *x, int *info);
extern void srandweibull(int n, float a, float b, int *state, float *x, int *info);
extern void srandmultinormal(int n, int m, float *xmu, float *c, int ldc, int *state, float *x, int ldx, int *info);
extern void srandmultistudentst(int n, int m, int df, float *xmu, float *c, int ldc, int *state, float *x, int ldx, int *info);

/* ACML version information */
extern void acmlversion(int *major, int *minor, int *patch);
extern void acmlinfo(void);

/* OMP interfaces */
extern int acmlgetmaxthreads(void);
extern int acmlgetnumthreads(void);
extern void acmlsetnumthreads(int numthreads);
extern int acmlgetnumprocs(void);

/* Basic complex arithmetic routines for C */
extern complex compose_complex(float x, float y);
extern float complex_real(complex z);
extern float complex_imag(complex z);
extern complex complex_negate(complex z);
extern complex complex_conjugate(complex z);
extern complex complex_add(complex z1, complex z2);
extern complex complex_subtract(complex z1, complex z2);
extern complex complex_multiply(complex z1, complex z2);
extern complex complex_divide(complex z1, complex z2);
extern int complex_equal(complex z1, complex z2);
extern int complex_notequal(complex z1, complex z2);
extern float complex_abs(complex z);

extern doublecomplex compose_doublecomplex(double x, double y);
extern double doublecomplex_real(doublecomplex z);
extern double doublecomplex_imag(doublecomplex z);
extern doublecomplex doublecomplex_negate(doublecomplex z);
extern doublecomplex doublecomplex_conjugate(doublecomplex z);
extern doublecomplex doublecomplex_add(doublecomplex z1, doublecomplex z2);
extern doublecomplex doublecomplex_subtract(doublecomplex z1, doublecomplex z2);
extern doublecomplex doublecomplex_multiply(doublecomplex z1, doublecomplex z2);
extern doublecomplex doublecomplex_divide(doublecomplex z1, doublecomplex z2);
extern int doublecomplex_equal(doublecomplex z1, doublecomplex z2);
extern int doublecomplex_notequal(doublecomplex z1, doublecomplex z2);
extern double doublecomplex_abs(doublecomplex z);

/* minmax utilities */
extern int acml_imax2(int a, int b);
extern int acml_imin2(int a, int b);
extern int acml_imax3(int a, int b, int c);
extern int acml_imin3(int a, int b, int c);
extern float acml_fmax2(float a, float b);
extern float acml_fmin2(float a, float b);
extern float acml_fmax3(float a, float b, float c);
extern float acml_fmin3(float a, float b, float c);
extern double acml_dmax2(double a, double b);
extern double acml_dmin2(double a, double b);
extern double acml_dmax3(double a, double b, double c);
extern double acml_dmin3(double a, double b, double c);

/* ------------------------------------------------------------------ */
/* Below are prototypes for the FORTRAN interfaces to ACML routines */

/*
   g77 promotes the return value of FORTRAN REAL functions to DOUBLE.
   Other compilers do not. If linking against a g77-compiled version
   of ACML, and calling such functions via their FORTRAN interface,
   (e.g. sasum_) the C programmer needs to be aware of this. Calling
   the C interface should always be safe. Functions affected in this
   way are the ACML BLAS functions:

          sasum_
          scasum_
          scnrm2_
          sdot_
          sdoti_
          sdsdot_
          snrm2_

   and the ACML LAPACK functions

          second_
          slamch_

   In order to get the expected behaviour, the macro
   FORTRAN_FLOAT_FUNCTIONS_RETURN_DOUBLE should be defined at the time
   of compilation of your C program which includes acml.h if you intend
   to call such a function from your C program and link to a g77-compiled
   version of ACML.

   N.B. the GNU gfortran compiler does NOT promote REAL to DOUBLE
   by default. Using the -ff2c flag with gfortran forces compatibility
   with g77 in this respect.
 */
/* BLAS routines */
extern void caxpy_(int *n, complex *alpha, complex *x, int *incx, complex *y, int *incy);
extern void ccopy_(int *n, complex *x, int *incx, complex *y, int *incy);
extern complex cdotc_(complex *ret_val, int *n, complex *x, int *incx, complex *y, int *incy);
extern void cdotcsub_(complex *ret_val, int *n, complex *x, int *incx, complex *y, int *incy);
extern complex cdotu_(complex *ret_val, int *n, complex *x, int *incx, complex *y, int *incy);
extern void cdotusub_(complex *ret_val, int *n, complex *x, int *incx, complex *y, int *incy);
extern void cgbmv_(char *trans, int *m, int *n, int *kl, int *ku, complex *alpha, complex *a, int *lda, complex *x, int *incx, complex *beta, complex *y, int *incy, int trans_len);
extern void cgemm_(char *transa, char *transb, int *m, int *n, int *k, complex *alpha, complex *a, int *lda, complex *b, int *ldb, complex *beta, complex *c, int *ldc, int transa_len, int transb_len);
extern void cgemv_(char *trans, int *m, int *n, complex *alpha, complex *a, int *lda, complex *x, int *incx, complex *beta, complex *y, int *incy, int trans_len);
extern void cgerc_(int *m, int *n, complex *alpha, complex *x, int *incx, complex *y, int *incy, complex *a, int *lda);
extern void cgeru_(int *m, int *n, complex *alpha, complex *x, int *incx, complex *y, int *incy, complex *a, int *lda);
extern void chbmv_(char *uplo, int *n, int *k, complex *alpha, complex *a, int *lda, complex *x, int *incx, complex *beta, complex *y, int *incy, int uplo_len);
extern void chemm_(char *side, char *uplo, int *m, int *n, complex *alpha, complex *a, int *lda, complex *b, int *ldb, complex *beta, complex *c, int *ldc, int side_len, int uplo_len);
extern void chemv_(char *uplo, int *n, complex *alpha, complex *a, int *lda, complex *x, int *incx, complex *beta, complex *y, int *incy, int uplo_len);
extern void cher_(char *uplo, int *n, float *alpha, complex *x, int *incx, complex *a, int *lda, int uplo_len);
extern void cher2_(char *uplo, int *n, complex *alpha, complex *x, int *incx, complex *y, int *incy, complex *a, int *lda, int uplo_len);
extern void cher2k_(char *uplo, char *trans, int *n, int *k, complex *alpha, complex *a, int *lda, complex *b, int *ldb, float *beta, complex *c, int *ldc, int uplo_len, int trans_len);
extern void cherk_(char *uplo, char *trans, int *n, int *k, float *alpha, complex *a, int *lda, float *beta, complex *c, int *ldc, int uplo_len, int trans_len);
extern void chpmv_(char *uplo, int *n, complex *alpha, complex *a, complex *x, int *incx, complex *beta, complex *y, int *incy, int uplo_len);
extern void chpr_(char *uplo, int *n, float *alpha, complex *x, int *incx, complex *a, int uplo_len);
extern void chpr2_(char *uplo, int *n, complex *alpha, complex *x, int *incx, complex *y, int *incy, complex *a, int uplo_len);
extern void crotg_(complex *a, complex *b, float *c, complex *s);
extern void cscal_(int *n, complex *alpha, complex *x, int *incx);
extern void csrot_(int *n, complex *x, int *incx, complex *y, int *incy, float *c, float *s);
extern void csscal_(int *n, float *alpha, complex *x, int *incx);
extern void cswap_(int *n, complex *x, int *incx, complex *y, int *incy);
extern void csymm_(char *side, char *uplo, int *m, int *n, complex *alpha, complex *a, int *lda, complex *b, int *ldb, complex *beta, complex *c, int *ldc, int side_len, int uplo_len);
extern void csyr2k_(char *uplo, char *trans, int *n, int *k, complex *alpha, complex *a, int *lda, complex *b, int *ldb, complex *beta, complex *c, int *ldc, int uplo_len, int trans_len);
extern void csyrk_(char *uplo, char *trans, int *n, int *k, complex *alpha, complex *a, int *lda, complex *beta, complex *c, int *ldc, int uplo_len, int trans_len);
extern void ctbmv_(char *uplo, char *trans, char *diag, int *n, int *k, complex *a, int *lda, complex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ctbsv_(char *uplo, char *trans, char *diag, int *n, int *k, complex *a, int *lda, complex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ctpmv_(char *uplo, char *trans, char *diag, int *n, complex *ap, complex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ctpsv_(char *uplo, char *trans, char *diag, int *n, complex *ap, complex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ctrmm_(char *side, char *uplo, char *transa, char *diag, int *m, int *n, complex *alpha, complex *a, int *lda, complex *b, int *ldb, int side_len, int uplo_len, int transa_len, int diag_len);
extern void ctrmv_(char *uplo, char *trans, char *diag, int *n, complex *a, int *lda, complex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ctrsm_(char *side, char *uplo, char *transa, char *diag, int *m, int *n, complex *alpha, complex *a, int *lda, complex *b, int *ldb, int side_len, int uplo_len, int transa_len, int diag_len);
extern void ctrsv_(char *uplo, char *trans, char *diag, int *n, complex *a, int *lda, complex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern double dasum_(int *n, double *x, int *incx);
extern void daxpy_(int *n, double *alpha, double *x, int *incx, double *y, int *incy);
extern double dcabs1_(doublecomplex *z);
extern void dcopy_(int *n, double *x, int *incx, double *y, int *incy);
extern double ddot_(int *n, double *x, int *incx, double *y, int *incy);
extern void dgbmv_(char *trans, int *m, int *n, int *kl, int *ku, double *alpha, double *a, int *lda, double *x, int *incx, double *beta, double *y, int *incy, int trans_len);
extern void dgemm_(char *transa, char *transb, int *m, int *n, int *k, double *alpha, double *a, int *lda, double *b, int *ldb, double *beta, double *c, int *ldc, int transa_len, int transb_len);
extern void dgemv_(char *trans, int *m, int *n, double *alpha, double *a, int *lda, double *x, int *incx, double *beta, double *y, int *incy, int trans_len);
extern void dger_(int *m, int *n, double *alpha, double *x, int *incx, double *y, int *incy, double *a, int *lda);
extern double dnrm2_(int *n, double *x, int *incx);
extern void drot_(int *n, double *x, int *incx, double *y, int *incy, double *c, double *s);
extern void drotg_(double *a, double *b, double *c, double *s);
extern void drotm_(int *n, double *x, int *incx, double *y, int *incy, double *param);
extern void drotmg_(double *d1, double *d2, double *b1, double *b2, double *param);
extern void dsbmv_(char *uplo, int *n, int *k, double *alpha, double *a, int *lda, double *x, int *incx, double *beta, double *y, int *incy, int uplo_len);
extern void dscal_(int *n, double *alpha, double *x, int *incx);
extern double dsdot_(int *n, float *sx, int *incx, float *sy, int *incy);
extern void dspmv_(char *uplo, int *n, double *alpha, double *ap, double *x, int *incx, double *beta, double *y, int *incy, int uplo_len);
extern void dspr_(char *uplo, int *n, double *alpha, double *x, int *incx, double *ap, int uplo_len);
extern void dspr2_(char *uplo, int *n, double *alpha, double *x, int *incx, double *y, int *incy, double *ap, int uplo_len);
extern void dswap_(int *n, double *x, int *incx, double *y, int *incy);
extern void dsymm_(char *side, char *uplo, int *m, int *n, double *alpha, double *a, int *lda, double *b, int *ldb, double *beta, double *c, int *ldc, int side_len, int uplo_len);
extern void dsymv_(char *uplo, int *n, double *alpha, double *a, int *lda, double *x, int *incx, double *beta, double *y, int *incy, int uplo_len);
extern void dsyr_(char *uplo, int *n, double *alpha, double *x, int *incx, double *a, int *lda, int uplo_len);
extern void dsyr2_(char *uplo, int *n, double *alpha, double *x, int *incx, double *y, int *incy, double *a, int *lda, int uplo_len);
extern void dsyr2k_(char *uplo, char *trans, int *n, int *k, double *alpha, double *a, int *lda, double *b, int *ldb, double *beta, double *c, int *ldc, int uplo_len, int trans_len);
extern void dsyrk_(char *uplo, char *trans, int *n, int *k, double *alpha, double *a, int *lda, double *beta, double *c, int *ldc, int uplo_len, int trans_len);
extern void dtbmv_(char *uplo, char *trans, char *diag, int *n, int *k, double *a, int *lda, double *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void dtbsv_(char *uplo, char *trans, char *diag, int *n, int *k, double *a, int *lda, double *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void dtpmv_(char *uplo, char *trans, char *diag, int *n, double *ap, double *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void dtpsv_(char *uplo, char *trans, char *diag, int *n, double *ap, double *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void dtrmm_(char *side, char *uplo, char *transa, char *diag, int *m, int *n, double *alpha, double *a, int *lda, double *b, int *ldb, int side_len, int uplo_len, int transa_len, int diag_len);
extern void dtrmv_(char *uplo, char *trans, char *diag, int *n, double *a, int *lda, double *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void dtrsm_(char *side, char *uplo, char *transa, char *diag, int *m, int *n, double *alpha, double *a, int *lda, double *b, int *ldb, int side_len, int uplo_len, int transa_len, int diag_len);
extern void dtrsv_(char *uplo, char *trans, char *diag, int *n, double *a, int *lda, double *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern double dzasum_(int *n, doublecomplex *x, int *incx);
extern double dznrm2_(int *n, doublecomplex *x, int *incx);
extern int icamax_(int *n, complex *x, int *incx);
extern int idamax_(int *n, double *x, int *incx);
extern int isamax_(int *n, float *x, int *incx);
extern int izamax_(int *n, doublecomplex *x, int *incx);
extern int lsame_(char *ca, char *cb, int ca_len, int cb_len);

#ifdef FORTRAN_FLOAT_FUNCTIONS_RETURN_DOUBLE
extern double sasum_(int *n, float *x, int *incx);
#else
extern float sasum_(int *n, float *x, int *incx);
#endif

extern void saxpy_(int *n, float *alpha, float *x, int *incx, float *y, int *incy);

#ifdef FORTRAN_FLOAT_FUNCTIONS_RETURN_DOUBLE
extern double scasum_(int *n, complex *x, int *incx);
extern double scnrm2_(int *n, complex *x, int *incx);
#else
extern float scasum_(int *n, complex *x, int *incx);
extern float scnrm2_(int *n, complex *x, int *incx);
#endif

extern void scopy_(int *n, float *x, int *incx, float *y, int *incy);

#ifdef FORTRAN_FLOAT_FUNCTIONS_RETURN_DOUBLE
extern double sdot_(int *n, float *x, int *incx, float *y, int *incy);
extern double sdsdot_(int *n, float *b, float *x, int *incx, float *y, int *incy);
#else
extern float sdot_(int *n, float *x, int *incx, float *y, int *incy);
extern float sdsdot_(int *n, float *b, float *x, int *incx, float *y, int *incy);
#endif

extern void sgbmv_(char *trans, int *m, int *n, int *kl, int *ku, float *alpha, float *a, int *lda, float *x, int *incx, float *beta, float *y, int *incy, int trans_len);
extern void sgemm_(char *transa, char *transb, int *m, int *n, int *k, float *alpha, float *a, int *lda, float *b, int *ldb, float *beta, float *c, int *ldc, int transa_len, int transb_len);
extern void sgemv_(char *trans, int *m, int *n, float *alpha, float *a, int *lda, float *x, int *incx, float *beta, float *y, int *incy, int trans_len);
extern void sger_(int *m, int *n, float *alpha, float *x, int *incx, float *y, int *incy, float *a, int *lda);

#ifdef FORTRAN_FLOAT_FUNCTIONS_RETURN_DOUBLE
extern double snrm2_(int *n, float *x, int *incx);
#else
extern float snrm2_(int *n, float *x, int *incx);
#endif

extern void srot_(int *n, float *x, int *incx, float *y, int *incy, float *c, float *s);
extern void srotg_(float *a, float *b, float *c, float *s);
extern void srotm_(int *n, float *x, int *incx, float *y, int *incy, float *param);
extern void srotmg_(float *d1, float *d2, float *b1, float *b2, float *param);
extern void ssbmv_(char *uplo, int *n, int *k, float *alpha, float *a, int *lda, float *x, int *incx, float *beta, float *y, int *incy, int uplo_len);
extern void sscal_(int *n, float *alpha, float *x, int *incx);
extern void sspmv_(char *uplo, int *n, float *alpha, float *ap, float *x, int *incx, float *beta, float *y, int *incy, int uplo_len);
extern void sspr_(char *uplo, int *n, float *alpha, float *x, int *incx, float *ap, int uplo_len);
extern void sspr2_(char *uplo, int *n, float *alpha, float *x, int *incx, float *y, int *incy, float *ap, int uplo_len);
extern void sswap_(int *n, float *x, int *incx, float *y, int *incy);
extern void ssymm_(char *side, char *uplo, int *m, int *n, float *alpha, float *a, int *lda, float *b, int *ldb, float *beta, float *c, int *ldc, int side_len, int uplo_len);
extern void ssymv_(char *uplo, int *n, float *alpha, float *a, int *lda, float *x, int *incx, float *beta, float *y, int *incy, int uplo_len);
extern void ssyr_(char *uplo, int *n, float *alpha, float *x, int *incx, float *a, int *lda, int uplo_len);
extern void ssyr2_(char *uplo, int *n, float *alpha, float *x, int *incx, float *y, int *incy, float *a, int *lda, int uplo_len);
extern void ssyr2k_(char *uplo, char *trans, int *n, int *k, float *alpha, float *a, int *lda, float *b, int *ldb, float *beta, float *c, int *ldc, int uplo_len, int trans_len);
extern void ssyrk_(char *uplo, char *trans, int *n, int *k, float *alpha, float *a, int *lda, float *beta, float *c, int *ldc, int uplo_len, int trans_len);
extern void stbmv_(char *uplo, char *trans, char *diag, int *n, int *k, float *a, int *lda, float *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void stbsv_(char *uplo, char *trans, char *diag, int *n, int *k, float *a, int *lda, float *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void stpmv_(char *uplo, char *trans, char *diag, int *n, float *ap, float *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void stpsv_(char *uplo, char *trans, char *diag, int *n, float *ap, float *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void strmm_(char *side, char *uplo, char *transa, char *diag, int *m, int *n, float *alpha, float *a, int *lda, float *b, int *ldb, int side_len, int uplo_len, int transa_len, int diag_len);
extern void strmv_(char *uplo, char *trans, char *diag, int *n, float *a, int *lda, float *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void strsm_(char *side, char *uplo, char *transa, char *diag, int *m, int *n, float *alpha, float *a, int *lda, float *b, int *ldb, int side_len, int uplo_len, int transa_len, int diag_len);
extern void strsv_(char *uplo, char *trans, char *diag, int *n, float *a, int *lda, float *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void xerbla_(const char *srname, int *info, int srname_len);
extern void zaxpy_(int *n, doublecomplex *alpha, doublecomplex *x, int *incx, doublecomplex *y, int *incy);
extern void zcopy_(int *n, doublecomplex *x, int *incx, doublecomplex *y, int *incy);
extern doublecomplex zdotc_(doublecomplex *ret_val, int *n, doublecomplex *x, int *incx, doublecomplex *y, int *incy);
extern void zdotcsub_(doublecomplex *ret_val, int *n, doublecomplex *x, int *incx, doublecomplex *y, int *incy);
extern doublecomplex zdotu_(doublecomplex *ret_val, int *n, doublecomplex *x, int *incx, doublecomplex *y, int *incy);
extern void zdotusub_(doublecomplex *ret_val, int *n, doublecomplex *x, int *incx, doublecomplex *y, int *incy);
extern void zdrot_(int *n, doublecomplex *x, int *incx, doublecomplex *y, int *incy, double *c, double *s);
extern void zdscal_(int *n, double *alpha, doublecomplex *x, int *incx);
extern void zgbmv_(char *trans, int *m, int *n, int *kl, int *ku, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *x, int *incx, doublecomplex *beta, doublecomplex *y, int *incy, int trans_len);
extern void zgemm_(char *transa, char *transb, int *m, int *n, int *k, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *beta, doublecomplex *c, int *ldc, int transa_len, int transb_len);
extern void zgemv_(char *trans, int *m, int *n, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *x, int *incx, doublecomplex *beta, doublecomplex *y, int *incy, int trans_len);
extern void zgerc_(int *m, int *n, doublecomplex *alpha, doublecomplex *x, int *incx, doublecomplex *y, int *incy, doublecomplex *a, int *lda);
extern void zgeru_(int *m, int *n, doublecomplex *alpha, doublecomplex *x, int *incx, doublecomplex *y, int *incy, doublecomplex *a, int *lda);
extern void zhbmv_(char *uplo, int *n, int *k, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *x, int *incx, doublecomplex *beta, doublecomplex *y, int *incy, int uplo_len);
extern void zhemm_(char *side, char *uplo, int *m, int *n, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *beta, doublecomplex *c, int *ldc, int side_len, int uplo_len);
extern void zhemv_(char *uplo, int *n, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *x, int *incx, doublecomplex *beta, doublecomplex *y, int *incy, int uplo_len);
extern void zher_(char *uplo, int *n, double *alpha, doublecomplex *x, int *incx, doublecomplex *a, int *lda, int uplo_len);
extern void zher2_(char *uplo, int *n, doublecomplex *alpha, doublecomplex *x, int *incx, doublecomplex *y, int *incy, doublecomplex *a, int *lda, int uplo_len);
extern void zher2k_(char *uplo, char *trans, int *n, int *k, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, double *beta, doublecomplex *c, int *ldc, int uplo_len, int trans_len);
extern void zherk_(char *uplo, char *trans, int *n, int *k, double *alpha, doublecomplex *a, int *lda, double *beta, doublecomplex *c, int *ldc, int uplo_len, int trans_len);
extern void zhpmv_(char *uplo, int *n, doublecomplex *alpha, doublecomplex *a, doublecomplex *x, int *incx, doublecomplex *beta, doublecomplex *y, int *incy, int uplo_len);
extern void zhpr_(char *uplo, int *n, double *alpha, doublecomplex *x, int *incx, doublecomplex *a, int uplo_len);
extern void zhpr2_(char *uplo, int *n, doublecomplex *alpha, doublecomplex *x, int *incx, doublecomplex *y, int *incy, doublecomplex *a, int uplo_len);
extern void zrotg_(doublecomplex *a, doublecomplex *b, double *c, doublecomplex *s);
extern void zscal_(int *n, doublecomplex *alpha, doublecomplex *x, int *incx);
extern void zswap_(int *n, doublecomplex *x, int *incx, doublecomplex *y, int *incy);
extern void zsymm_(char *side, char *uplo, int *m, int *n, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *beta, doublecomplex *c, int *ldc, int side_len, int uplo_len);
extern void zsyr2k_(char *uplo, char *trans, int *n, int *k, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *beta, doublecomplex *c, int *ldc, int uplo_len, int trans_len);
extern void zsyrk_(char *uplo, char *trans, int *n, int *k, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *beta, doublecomplex *c, int *ldc, int uplo_len, int trans_len);
extern void ztbmv_(char *uplo, char *trans, char *diag, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ztbsv_(char *uplo, char *trans, char *diag, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ztpmv_(char *uplo, char *trans, char *diag, int *n, doublecomplex *ap, doublecomplex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ztpsv_(char *uplo, char *trans, char *diag, int *n, doublecomplex *ap, doublecomplex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ztrmm_(char *side, char *uplo, char *transa, char *diag, int *m, int *n, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int side_len, int uplo_len, int transa_len, int diag_len);
extern void ztrmv_(char *uplo, char *trans, char *diag, int *n, doublecomplex *a, int *lda, doublecomplex *x, int *incx, int uplo_len, int trans_len, int diag_len);
extern void ztrsm_(char *side, char *uplo, char *transa, char *diag, int *m, int *n, doublecomplex *alpha, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int side_len, int uplo_len, int transa_len, int diag_len);
extern void ztrsv_(char *uplo, char *trans, char *diag, int *n, doublecomplex *a, int *lda, doublecomplex *x, int *incx, int uplo_len, int trans_len, int diag_len);

/* LAPACK routines */
extern int ilaenv_(int *ispec, const char *name, const char *opts, int *n1, int *n2, int *n3, int *n4, int name_len, int opts_len);
extern void ilaenvset_(int *ispec, const char *name, const char *opts, int *n1, int *n2, int *n3, int *n4, int *nvalue, int *info, int name_len, int opts_len);
extern void cbdsqr_(char *uplo, int *n, int *ncvt, int *nru, int *ncc, float *d, float *e, complex *vt, int *ldvt, complex *u, int *ldu, complex *c, int *ldc, float *rwork, int *info, int uplo_len);
extern void cgbbrd_(char *vect, int *m, int *n, int *ncc, int *kl, int *ku, complex *ab, int *ldab, float *d, float *e, complex *q, int *ldq, complex *pt, int *ldpt, complex *c, int *ldc, complex *work, float *rwork, int *info, int vect_len);
extern void cgbcon_(char *norm, int *n, int *kl, int *ku, complex *ab, int *ldab, int *ipiv, float *anorm, float *rcond, complex *work, float *rwork, int *info, int norm_len);
extern void cgbequ_(int *m, int *n, int *kl, int *ku, complex *ab, int *ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax, int *info);
extern void cgbrfs_(char *trans, int *n, int *kl, int *ku, int *nrhs, complex *ab, int *ldab, complex *afb, int *ldafb, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int trans_len);
extern void cgbsv_(int *n, int *kl, int *ku, int *nrhs, complex *ab, int *ldab, int *ipiv, complex *b, int *ldb, int *info);
extern void cgbsvx_(char *fact, char *trans, int *n, int *kl, int *ku, int *nrhs, complex *ab, int *ldab, complex *afb, int *ldafb, int *ipiv, char *equed, float *r, float *c, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, float *rwork, int *info, int fact_len, int trans_len, int equed_len);
extern void cgbtrf_(int *m, int *n, int *kl, int *ku, complex *ab, int *ldab, int *ipiv, int *info);
extern void cgbtrs_(char *trans, int *n, int *kl, int *ku, int *nrhs, complex *ab, int *ldab, int *ipiv, complex *b, int *ldb, int *info, int trans_len);
extern void cgebak_(char *job, char *side, int *n, int *ilo, int *ihi, float *scale, int *m, complex *v, int *ldv, int *info, int job_len, int side_len);
extern void cgebal_(char *job, int *n, complex *a, int *lda, int *ilo, int *ihi, float *scale, int *info, int job_len);
extern void cgebrd_(int *m, int *n, complex *a, int *lda, float *d, float *e, complex *tauq, complex *taup, complex *work, int *lwork, int *info);
extern void cgecon_(char *norm, int *n, complex *a, int *lda, float *anorm, float *rcond, complex *work, float *rwork, int *info, int norm_len);
extern void cgeequ_(int *m, int *n, complex *a, int *lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax, int *info);
extern void cgees_(char *jobvs, char *sort, ACML_CGEES_SELECT select, int *n, complex *a, int *lda, int *sdim, complex *w, complex *vs, int *ldvs, complex *work, int *lwork, float *rwork, int *bwork, int *info, int jobvs_len, int sort_len);
extern void cgeesx_(char *jobvs, char *sort, ACML_CGEESX_SELECT select, char *sense, int *n, complex *a, int *lda, int *sdim, complex *w, complex *vs, int *ldvs, float *rconde, float *rcondv, complex *work, int *lwork, float *rwork, int *bwork, int *info, int jobvs_len, int sort_len, int sense_len);
extern void cgeev_(char *jobvl, char *jobvr, int *n, complex *a, int *lda, complex *w, complex *vl, int *ldvl, complex *vr, int *ldvr, complex *work, int *lwork, float *rwork, int *info, int jobvl_len, int jobvr_len);
extern void cgeevx_(char *balanc, char *jobvl, char *jobvr, char *sense, int *n, complex *a, int *lda, complex *w, complex *vl, int *ldvl, complex *vr, int *ldvr, int *ilo, int *ihi, float *scale, float *abnrm, float *rconde, float *rcondv, complex *work, int *lwork, float *rwork, int *info, int balanc_len, int jobvl_len, int jobvr_len, int sense_len);
extern void cgegs_(char *jobvsl, char *jobvsr, int *n, complex *a, int *lda, complex *b, int *ldb, complex *alpha, complex *beta, complex *vsl, int *ldvsl, complex *vsr, int *ldvsr, complex *work, int *lwork, float *rwork, int *info, int jobvsl_len, int jobvsr_len);
extern void cgegv_(char *jobvl, char *jobvr, int *n, complex *a, int *lda, complex *b, int *ldb, complex *alpha, complex *beta, complex *vl, int *ldvl, complex *vr, int *ldvr, complex *work, int *lwork, float *rwork, int *info, int jobvl_len, int jobvr_len);
extern void cgehrd_(int *n, int *ilo, int *ihi, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cgelqf_(int *m, int *n, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cgels_(char *trans, int *m, int *n, int *nrhs, complex *a, int *lda, complex *b, int *ldb, complex *work, int *lwork, int *info, int trans_len);
extern void cgelsd_(int *m, int *n, int *nrhs, complex *a, int *lda, complex *b, int *ldb, float *s, float *rcond, int *rank, complex *work, int *lwork, float *rwork, int *iwork, int *info);
extern void cgelss_(int *m, int *n, int *nrhs, complex *a, int *lda, complex *b, int *ldb, float *s, float *rcond, int *rank, complex *work, int *lwork, float *rwork, int *info);
extern void cgelsx_(int *m, int *n, int *nrhs, complex *a, int *lda, complex *b, int *ldb, int *jpvt, float *rcond, int *rank, complex *work, float *rwork, int *info);
extern void cgelsy_(int *m, int *n, int *nrhs, complex *a, int *lda, complex *b, int *ldb, int *jpvt, float *rcond, int *rank, complex *work, int *lwork, float *rwork, int *info);
extern void cgeqlf_(int *m, int *n, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cgeqp3_(int *m, int *n, complex *a, int *lda, int *jpvt, complex *tau, complex *work, int *lwork, float *rwork, int *info);
extern void cgeqpf_(int *m, int *n, complex *a, int *lda, int *jpvt, complex *tau, complex *work, float *rwork, int *info);
extern void cgeqrf_(int *m, int *n, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cgerfs_(char *trans, int *n, int *nrhs, complex *a, int *lda, complex *af, int *ldaf, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int trans_len);
extern void cgerqf_(int *m, int *n, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cgesdd_(char *jobz, int *m, int *n, complex *a, int *lda, float *s, complex *u, int *ldu, complex *vt, int *ldvt, complex *work, int *lwork, float *rwork, int *iwork, int *info, int jobz_len);
extern void cgesv_(int *n, int *nrhs, complex *a, int *lda, int *ipiv, complex *b, int *ldb, int *info);
extern void cgesvd_(char *jobu, char *jobvt, int *m, int *n, complex *a, int *lda, float *s, complex *u, int *ldu, complex *vt, int *ldvt, complex *work, int *lwork, float *rwork, int *info, int jobu_len, int jobvt_len);
extern void cgesvx_(char *fact, char *trans, int *n, int *nrhs, complex *a, int *lda, complex *af, int *ldaf, int *ipiv, char *equed, float *r, float *c, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, float *rwork, int *info, int fact_len, int trans_len, int equed_len);
extern void cgetrf_(int *m, int *n, complex *a, int *lda, int *ipiv, int *info);
extern void cgetri_(int *n, complex *a, int *lda, int *ipiv, complex *work, int *lwork, int *info);
extern void cgetrs_(char *trans, int *n, int *nrhs, complex *a, int *lda, int *ipiv, complex *b, int *ldb, int *info, int trans_len);
extern void cggbak_(char *job, char *side, int *n, int *ilo, int *ihi, float *lscale, float *rscale, int *m, complex *v, int *ldv, int *info, int job_len, int side_len);
extern void cggbal_(char *job, int *n, complex *a, int *lda, complex *b, int *ldb, int *ilo, int *ihi, float *lscale, float *rscale, float *work, int *info, int job_len);
extern void cgges_(char *jobvsl, char *jobvsr, char *sort, ACML_CGGES_SELCTG selctg, int *n, complex *a, int *lda, complex *b, int *ldb, int *sdim, complex *alpha, complex *beta, complex *vsl, int *ldvsl, complex *vsr, int *ldvsr, complex *work, int *lwork, float *rwork, int *bwork, int *info, int jobvsl_len, int jobvsr_len, int sort_len);
extern void cggesx_(char *jobvsl, char *jobvsr, char *sort, ACML_CGGESX_SELCTG selctg, char *sense, int *n, complex *a, int *lda, complex *b, int *ldb, int *sdim, complex *alpha, complex *beta, complex *vsl, int *ldvsl, complex *vsr, int *ldvsr, float *rconde, float *rcondv, complex *work, int *lwork, float *rwork, int *iwork, int *liwork, int *bwork, int *info, int jobvsl_len, int jobvsr_len, int sort_len, int sense_len);
extern void cggev_(char *jobvl, char *jobvr, int *n, complex *a, int *lda, complex *b, int *ldb, complex *alpha, complex *beta, complex *vl, int *ldvl, complex *vr, int *ldvr, complex *work, int *lwork, float *rwork, int *info, int jobvl_len, int jobvr_len);
extern void cggevx_(char *balanc, char *jobvl, char *jobvr, char *sense, int *n, complex *a, int *lda, complex *b, int *ldb, complex *alpha, complex *beta, complex *vl, int *ldvl, complex *vr, int *ldvr, int *ilo, int *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv, complex *work, int *lwork, float *rwork, int *iwork, int *bwork, int *info, int balanc_len, int jobvl_len, int jobvr_len, int sense_len);
extern void cggglm_(int *n, int *m, int *p, complex *a, int *lda, complex *b, int *ldb, complex *d, complex *x, complex *y, complex *work, int *lwork, int *info);
extern void cgghrd_(char *compq, char *compz, int *n, int *ilo, int *ihi, complex *a, int *lda, complex *b, int *ldb, complex *q, int *ldq, complex *z, int *ldz, int *info, int compq_len, int compz_len);
extern void cgglse_(int *m, int *n, int *p, complex *a, int *lda, complex *b, int *ldb, complex *c, complex *d, complex *x, complex *work, int *lwork, int *info);
extern void cggqrf_(int *n, int *m, int *p, complex *a, int *lda, complex *taua, complex *b, int *ldb, complex *taub, complex *work, int *lwork, int *info);
extern void cggrqf_(int *m, int *p, int *n, complex *a, int *lda, complex *taua, complex *b, int *ldb, complex *taub, complex *work, int *lwork, int *info);
extern void cggsvd_(char *jobu, char *jobv, char *jobq, int *m, int *n, int *p, int *k, int *l, complex *a, int *lda, complex *b, int *ldb, float *alpha, float *beta, complex *u, int *ldu, complex *v, int *ldv, complex *q, int *ldq, complex *work, float *rwork, int *iwork, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void cggsvp_(char *jobu, char *jobv, char *jobq, int *m, int *p, int *n, complex *a, int *lda, complex *b, int *ldb, float *tola, float *tolb, int *k, int *l, complex *u, int *ldu, complex *v, int *ldv, complex *q, int *ldq, int *iwork, float *rwork, complex *tau, complex *work, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void cgtcon_(char *norm, int *n, complex *dl, complex *d, complex *du, complex *du2, int *ipiv, float *anorm, float *rcond, complex *work, int *info, int norm_len);
extern void cgtrfs_(char *trans, int *n, int *nrhs, complex *dl, complex *d, complex *du, complex *dlf, complex *df, complex *duf, complex *du2, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int trans_len);
extern void cgtsv_(int *n, int *nrhs, complex *dl, complex *d, complex *du, complex *b, int *ldb, int *info);
extern void cgtsvx_(char *fact, char *trans, int *n, int *nrhs, complex *dl, complex *d, complex *du, complex *dlf, complex *df, complex *duf, complex *du2, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, float *rwork, int *info, int fact_len, int trans_len);
extern void cgttrf_(int *n, complex *dl, complex *d, complex *du, complex *du2, int *ipiv, int *info);
extern void cgttrs_(char *trans, int *n, int *nrhs, complex *dl, complex *d, complex *du, complex *du2, int *ipiv, complex *b, int *ldb, int *info, int trans_len);
extern void chbev_(char *jobz, char *uplo, int *n, int *kd, complex *ab, int *ldab, float *w, complex *z, int *ldz, complex *work, float *rwork, int *info, int jobz_len, int uplo_len);
extern void chbevd_(char *jobz, char *uplo, int *n, int *kd, complex *ab, int *ldab, float *w, complex *z, int *ldz, complex *work, int *lwork, float *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void chbevx_(char *jobz, char *range, char *uplo, int *n, int *kd, complex *ab, int *ldab, complex *q, int *ldq, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, complex *z, int *ldz, complex *work, float *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void chbgst_(char *vect, char *uplo, int *n, int *ka, int *kb, complex *ab, int *ldab, complex *bb, int *ldbb, complex *x, int *ldx, complex *work, float *rwork, int *info, int vect_len, int uplo_len);
extern void chbgv_(char *jobz, char *uplo, int *n, int *ka, int *kb, complex *ab, int *ldab, complex *bb, int *ldbb, float *w, complex *z, int *ldz, complex *work, float *rwork, int *info, int jobz_len, int uplo_len);
extern void chbgvd_(char *jobz, char *uplo, int *n, int *ka, int *kb, complex *ab, int *ldab, complex *bb, int *ldbb, float *w, complex *z, int *ldz, complex *work, int *lwork, float *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void chbgvx_(char *jobz, char *range, char *uplo, int *n, int *ka, int *kb, complex *ab, int *ldab, complex *bb, int *ldbb, complex *q, int *ldq, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, complex *z, int *ldz, complex *work, float *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void chbtrd_(char *vect, char *uplo, int *n, int *kd, complex *ab, int *ldab, float *d, float *e, complex *q, int *ldq, complex *work, int *info, int vect_len, int uplo_len);
extern void checon_(char *uplo, int *n, complex *a, int *lda, int *ipiv, float *anorm, float *rcond, complex *work, int *info, int uplo_len);
extern void cheev_(char *jobz, char *uplo, int *n, complex *a, int *lda, float *w, complex *work, int *lwork, float *rwork, int *info, int jobz_len, int uplo_len);
extern void cheevd_(char *jobz, char *uplo, int *n, complex *a, int *lda, float *w, complex *work, int *lwork, float *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void cheevr_(char *jobz, char *range, char *uplo, int *n, complex *a, int *lda, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, complex *z, int *ldz, int *isuppz, complex *work, int *lwork, float *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len, int uplo_len);
extern void cheevx_(char *jobz, char *range, char *uplo, int *n, complex *a, int *lda, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, complex *z, int *ldz, complex *work, int *lwork, float *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void chegst_(int *itype, char *uplo, int *n, complex *a, int *lda, complex *b, int *ldb, int *info, int uplo_len);
extern void chegv_(int *itype, char *jobz, char *uplo, int *n, complex *a, int *lda, complex *b, int *ldb, float *w, complex *work, int *lwork, float *rwork, int *info, int jobz_len, int uplo_len);
extern void chegvd_(int *itype, char *jobz, char *uplo, int *n, complex *a, int *lda, complex *b, int *ldb, float *w, complex *work, int *lwork, float *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void chegvx_(int *itype, char *jobz, char *range, char *uplo, int *n, complex *a, int *lda, complex *b, int *ldb, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, complex *z, int *ldz, complex *work, int *lwork, float *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void cherfs_(char *uplo, int *n, int *nrhs, complex *a, int *lda, complex *af, int *ldaf, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len);
extern void chesv_(char *uplo, int *n, int *nrhs, complex *a, int *lda, int *ipiv, complex *b, int *ldb, complex *work, int *lwork, int *info, int uplo_len);
extern void chesvx_(char *fact, char *uplo, int *n, int *nrhs, complex *a, int *lda, complex *af, int *ldaf, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, int *lwork, float *rwork, int *info, int fact_len, int uplo_len);
extern void chetrd_(char *uplo, int *n, complex *a, int *lda, float *d, float *e, complex *tau, complex *work, int *lwork, int *info, int uplo_len);
extern void chetrf_(char *uplo, int *n, complex *a, int *lda, int *ipiv, complex *work, int *lwork, int *info, int uplo_len);
extern void chetri_(char *uplo, int *n, complex *a, int *lda, int *ipiv, complex *work, int *info, int uplo_len);
extern void chetrs_(char *uplo, int *n, int *nrhs, complex *a, int *lda, int *ipiv, complex *b, int *ldb, int *info, int uplo_len);
extern void chgeqz_(char *job, char *compq, char *compz, int *n, int *ilo, int *ihi, complex *a, int *lda, complex *b, int *ldb, complex *alpha, complex *beta, complex *q, int *ldq, complex *z, int *ldz, complex *work, int *lwork, float *rwork, int *info, int job_len, int compq_len, int compz_len);
extern void chpcon_(char *uplo, int *n, complex *ap, int *ipiv, float *anorm, float *rcond, complex *work, int *info, int uplo_len);
extern void chpev_(char *jobz, char *uplo, int *n, complex *ap, float *w, complex *z, int *ldz, complex *work, float *rwork, int *info, int jobz_len, int uplo_len);
extern void chpevd_(char *jobz, char *uplo, int *n, complex *ap, float *w, complex *z, int *ldz, complex *work, int *lwork, float *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void chpevx_(char *jobz, char *range, char *uplo, int *n, complex *ap, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, complex *z, int *ldz, complex *work, float *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void chpgst_(int *itype, char *uplo, int *n, complex *ap, complex *bp, int *info, int uplo_len);
extern void chpgv_(int *itype, char *jobz, char *uplo, int *n, complex *ap, complex *bp, float *w, complex *z, int *ldz, complex *work, float *rwork, int *info, int jobz_len, int uplo_len);
extern void chpgvd_(int *itype, char *jobz, char *uplo, int *n, complex *ap, complex *bp, float *w, complex *z, int *ldz, complex *work, int *lwork, float *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void chpgvx_(int *itype, char *jobz, char *range, char *uplo, int *n, complex *ap, complex *bp, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, complex *z, int *ldz, complex *work, float *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void chprfs_(char *uplo, int *n, int *nrhs, complex *ap, complex *afp, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len);
extern void chpsv_(char *uplo, int *n, int *nrhs, complex *ap, int *ipiv, complex *b, int *ldb, int *info, int uplo_len);
extern void chpsvx_(char *fact, char *uplo, int *n, int *nrhs, complex *ap, complex *afp, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, float *rwork, int *info, int fact_len, int uplo_len);
extern void chptrd_(char *uplo, int *n, complex *ap, float *d, float *e, complex *tau, int *info, int uplo_len);
extern void chptrf_(char *uplo, int *n, complex *ap, int *ipiv, int *info, int uplo_len);
extern void chptri_(char *uplo, int *n, complex *ap, int *ipiv, complex *work, int *info, int uplo_len);
extern void chptrs_(char *uplo, int *n, int *nrhs, complex *ap, int *ipiv, complex *b, int *ldb, int *info, int uplo_len);
extern void chsein_(char *side, char *eigsrc, char *initv, int *select, int *n, complex *h, int *ldh, complex *w, complex *vl, int *ldvl, complex *vr, int *ldvr, int *mm, int *m, complex *work, float *rwork, int *ifaill, int *ifailr, int *info, int side_len, int eigsrc_len, int initv_len);
extern void chseqr_(char *job, char *compz, int *n, int *ilo, int *ihi, complex *h, int *ldh, complex *w, complex *z, int *ldz, complex *work, int *lwork, int *info, int job_len, int compz_len);
extern void cpbcon_(char *uplo, int *n, int *kd, complex *ab, int *ldab, float *anorm, float *rcond, complex *work, float *rwork, int *info, int uplo_len);
extern void cpbequ_(char *uplo, int *n, int *kd, complex *ab, int *ldab, float *s, float *scond, float *amax, int *info, int uplo_len);
extern void cpbrfs_(char *uplo, int *n, int *kd, int *nrhs, complex *ab, int *ldab, complex *afb, int *ldafb, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len);
extern void cpbstf_(char *uplo, int *n, int *kd, complex *ab, int *ldab, int *info, int uplo_len);
extern void cpbsv_(char *uplo, int *n, int *kd, int *nrhs, complex *ab, int *ldab, complex *b, int *ldb, int *info, int uplo_len);
extern void cpbsvx_(char *fact, char *uplo, int *n, int *kd, int *nrhs, complex *ab, int *ldab, complex *afb, int *ldafb, char *equed, float *s, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, float *rwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void cpbtrf_(char *uplo, int *n, int *kd, complex *ab, int *ldab, int *info, int uplo_len);
extern void cpbtrs_(char *uplo, int *n, int *kd, int *nrhs, complex *ab, int *ldab, complex *b, int *ldb, int *info, int uplo_len);
extern void cpocon_(char *uplo, int *n, complex *a, int *lda, float *anorm, float *rcond, complex *work, float *rwork, int *info, int uplo_len);
extern void cpoequ_(int *n, complex *a, int *lda, float *s, float *scond, float *amax, int *info);
extern void cporfs_(char *uplo, int *n, int *nrhs, complex *a, int *lda, complex *af, int *ldaf, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len);
extern void cposv_(char *uplo, int *n, int *nrhs, complex *a, int *lda, complex *b, int *ldb, int *info, int uplo_len);
extern void cposvx_(char *fact, char *uplo, int *n, int *nrhs, complex *a, int *lda, complex *af, int *ldaf, char *equed, float *s, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, float *rwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void cpotrf_(char *uplo, int *n, complex *a, int *lda, int *info, int uplo_len);
extern void cpotri_(char *uplo, int *n, complex *a, int *lda, int *info, int uplo_len);
extern void cpotrs_(char *uplo, int *n, int *nrhs, complex *a, int *lda, complex *b, int *ldb, int *info, int uplo_len);
extern void cppcon_(char *uplo, int *n, complex *ap, float *anorm, float *rcond, complex *work, float *rwork, int *info, int uplo_len);
extern void cppequ_(char *uplo, int *n, complex *ap, float *s, float *scond, float *amax, int *info, int uplo_len);
extern void cpprfs_(char *uplo, int *n, int *nrhs, complex *ap, complex *afp, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len);
extern void cppsv_(char *uplo, int *n, int *nrhs, complex *ap, complex *b, int *ldb, int *info, int uplo_len);
extern void cppsvx_(char *fact, char *uplo, int *n, int *nrhs, complex *ap, complex *afp, char *equed, float *s, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, float *rwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void cpptrf_(char *uplo, int *n, complex *ap, int *info, int uplo_len);
extern void cpptri_(char *uplo, int *n, complex *ap, int *info, int uplo_len);
extern void cpptrs_(char *uplo, int *n, int *nrhs, complex *ap, complex *b, int *ldb, int *info, int uplo_len);
extern void cptcon_(int *n, float *d, complex *e, float *anorm, float *rcond, float *rwork, int *info);
extern void cpteqr_(char *compz, int *n, float *d, float *e, complex *z, int *ldz, float *work, int *info, int compz_len);
extern void cptrfs_(char *uplo, int *n, int *nrhs, float *d, complex *e, float *df, complex *ef, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len);
extern void cptsv_(int *n, int *nrhs, float *d, complex *e, complex *b, int *ldb, int *info);
extern void cptsvx_(char *fact, int *n, int *nrhs, float *d, complex *e, float *df, complex *ef, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, float *rwork, int *info, int fact_len);
extern void cpttrf_(int *n, float *d, complex *e, int *info);
extern void cpttrs_(char *uplo, int *n, int *nrhs, float *d, complex *e, complex *b, int *ldb, int *info, int uplo_len);
extern void cspcon_(char *uplo, int *n, complex *ap, int *ipiv, float *anorm, float *rcond, complex *work, int *info, int uplo_len);
extern void csprfs_(char *uplo, int *n, int *nrhs, complex *ap, complex *afp, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len);
extern void cspsv_(char *uplo, int *n, int *nrhs, complex *ap, int *ipiv, complex *b, int *ldb, int *info, int uplo_len);
extern void cspsvx_(char *fact, char *uplo, int *n, int *nrhs, complex *ap, complex *afp, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, float *rwork, int *info, int fact_len, int uplo_len);
extern void csptrf_(char *uplo, int *n, complex *ap, int *ipiv, int *info, int uplo_len);
extern void csptri_(char *uplo, int *n, complex *ap, int *ipiv, complex *work, int *info, int uplo_len);
extern void csptrs_(char *uplo, int *n, int *nrhs, complex *ap, int *ipiv, complex *b, int *ldb, int *info, int uplo_len);
extern void cstedc_(char *compz, int *n, float *d, float *e, complex *z, int *ldz, complex *work, int *lwork, float *rwork, int *lrwork, int *iwork, int *liwork, int *info, int compz_len);
extern void cstegr_(char *jobz, char *range, int *n, float *d, float *e, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, complex *z, int *ldz, int *isuppz, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len);
extern void cstein_(int *n, float *d, float *e, int *m, float *w, int *iblock, int *isplit, complex *z, int *ldz, float *work, int *iwork, int *ifail, int *info);
extern void csteqr_(char *compz, int *n, float *d, float *e, complex *z, int *ldz, float *work, int *info, int compz_len);
extern void csycon_(char *uplo, int *n, complex *a, int *lda, int *ipiv, float *anorm, float *rcond, complex *work, int *info, int uplo_len);
extern void csyrfs_(char *uplo, int *n, int *nrhs, complex *a, int *lda, complex *af, int *ldaf, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len);
extern void csysv_(char *uplo, int *n, int *nrhs, complex *a, int *lda, int *ipiv, complex *b, int *ldb, complex *work, int *lwork, int *info, int uplo_len);
extern void csysvx_(char *fact, char *uplo, int *n, int *nrhs, complex *a, int *lda, complex *af, int *ldaf, int *ipiv, complex *b, int *ldb, complex *x, int *ldx, float *rcond, float *ferr, float *berr, complex *work, int *lwork, float *rwork, int *info, int fact_len, int uplo_len);
extern void csytrf_(char *uplo, int *n, complex *a, int *lda, int *ipiv, complex *work, int *lwork, int *info, int uplo_len);
extern void csytri_(char *uplo, int *n, complex *a, int *lda, int *ipiv, complex *work, int *info, int uplo_len);
extern void csytrs_(char *uplo, int *n, int *nrhs, complex *a, int *lda, int *ipiv, complex *b, int *ldb, int *info, int uplo_len);
extern void ctbcon_(char *norm, char *uplo, char *diag, int *n, int *kd, complex *ab, int *ldab, float *rcond, complex *work, float *rwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void ctbrfs_(char *uplo, char *trans, char *diag, int *n, int *kd, int *nrhs, complex *ab, int *ldab, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void ctbtrs_(char *uplo, char *trans, char *diag, int *n, int *kd, int *nrhs, complex *ab, int *ldab, complex *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void ctgevc_(char *side, char *howmny, int *select, int *n, complex *a, int *lda, complex *b, int *ldb, complex *vl, int *ldvl, complex *vr, int *ldvr, int *mm, int *m, complex *work, float *rwork, int *info, int side_len, int howmny_len);
extern void ctgexc_(int *wantq, int *wantz, int *n, complex *a, int *lda, complex *b, int *ldb, complex *q, int *ldq, complex *z, int *ldz, int *ifst, int *ilst, int *info);
extern void ctgsen_(int *ijob, int *wantq, int *wantz, int *select, int *n, complex *a, int *lda, complex *b, int *ldb, complex *alpha, complex *beta, complex *q, int *ldq, complex *z, int *ldz, int *m, float *pl, float *pr, float *dif, complex *work, int *lwork, int *iwork, int *liwork, int *info);
extern void ctgsja_(char *jobu, char *jobv, char *jobq, int *m, int *p, int *n, int *k, int *l, complex *a, int *lda, complex *b, int *ldb, float *tola, float *tolb, float *alpha, float *beta, complex *u, int *ldu, complex *v, int *ldv, complex *q, int *ldq, complex *work, int *ncycle, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void ctgsna_(char *job, char *howmny, int *select, int *n, complex *a, int *lda, complex *b, int *ldb, complex *vl, int *ldvl, complex *vr, int *ldvr, float *s, float *dif, int *mm, int *m, complex *work, int *lwork, int *iwork, int *info, int job_len, int howmny_len);
extern void ctgsyl_(char *trans, int *ijob, int *m, int *n, complex *a, int *lda, complex *b, int *ldb, complex *c, int *ldc, complex *d, int *ldd, complex *e, int *lde, complex *f, int *ldf, float *scale, float *dif, complex *work, int *lwork, int *iwork, int *info, int trans_len);
extern void ctpcon_(char *norm, char *uplo, char *diag, int *n, complex *ap, float *rcond, complex *work, float *rwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void ctprfs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, complex *ap, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void ctptri_(char *uplo, char *diag, int *n, complex *ap, int *info, int uplo_len, int diag_len);
extern void ctptrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, complex *ap, complex *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void ctrcon_(char *norm, char *uplo, char *diag, int *n, complex *a, int *lda, float *rcond, complex *work, float *rwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void ctrevc_(char *side, char *howmny, int *select, int *n, complex *t, int *ldt, complex *vl, int *ldvl, complex *vr, int *ldvr, int *mm, int *m, complex *work, float *rwork, int *info, int side_len, int howmny_len);
extern void ctrexc_(char *compq, int *n, complex *t, int *ldt, complex *q, int *ldq, int *ifst, int *ilst, int *info, int compq_len);
extern void ctrrfs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, complex *a, int *lda, complex *b, int *ldb, complex *x, int *ldx, float *ferr, float *berr, complex *work, float *rwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void ctrsen_(char *job, char *compq, int *select, int *n, complex *t, int *ldt, complex *q, int *ldq, complex *w, int *m, float *s, float *sep, complex *work, int *lwork, int *info, int job_len, int compq_len);
extern void ctrsna_(char *job, char *howmny, int *select, int *n, complex *t, int *ldt, complex *vl, int *ldvl, complex *vr, int *ldvr, float *s, float *sep, int *mm, int *m, complex *work, int *ldwork, float *rwork, int *info, int job_len, int howmny_len);
extern void ctrsyl_(char *trana, char *tranb, int *isgn, int *m, int *n, complex *a, int *lda, complex *b, int *ldb, complex *c, int *ldc, float *scale, int *info, int trana_len, int tranb_len);
extern void ctrtri_(char *uplo, char *diag, int *n, complex *a, int *lda, int *info, int uplo_len, int diag_len);
extern void ctrtrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, complex *a, int *lda, complex *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void ctzrqf_(int *m, int *n, complex *a, int *lda, complex *tau, int *info);
extern void ctzrzf_(int *m, int *n, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cungbr_(char *vect, int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info, int vect_len);
extern void cunghr_(int *n, int *ilo, int *ihi, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cunglq_(int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cungql_(int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cungqr_(int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cungrq_(int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info);
extern void cungtr_(char *uplo, int *n, complex *a, int *lda, complex *tau, complex *work, int *lwork, int *info, int uplo_len);
extern void cunmbr_(char *vect, char *side, char *trans, int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *c, int *ldc, complex *work, int *lwork, int *info, int vect_len, int side_len, int trans_len);
extern void cunmhr_(char *side, char *trans, int *m, int *n, int *ilo, int *ihi, complex *a, int *lda, complex *tau, complex *c, int *ldc, complex *work, int *lwork, int *info, int side_len, int trans_len);
extern void cunmlq_(char *side, char *trans, int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *c, int *ldc, complex *work, int *lwork, int *info, int side_len, int trans_len);
extern void cunmql_(char *side, char *trans, int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *c, int *ldc, complex *work, int *lwork, int *info, int side_len, int trans_len);
extern void cunmqr_(char *side, char *trans, int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *c, int *ldc, complex *work, int *lwork, int *info, int side_len, int trans_len);
extern void cunmr3_(char *side, char *trans, int *m, int *n, int *k, int *l, complex *a, int *lda, complex *tau, complex *c, int *ldc, complex *work, int *info, int side_len, int trans_len);
extern void cunmrq_(char *side, char *trans, int *m, int *n, int *k, complex *a, int *lda, complex *tau, complex *c, int *ldc, complex *work, int *lwork, int *info, int side_len, int trans_len);
extern void cunmrz_(char *side, char *trans, int *m, int *n, int *k, int *l, complex *a, int *lda, complex *tau, complex *c, int *ldc, complex *work, int *lwork, int *info, int side_len, int trans_len);
extern void cunmtr_(char *side, char *uplo, char *trans, int *m, int *n, complex *a, int *lda, complex *tau, complex *c, int *ldc, complex *work, int *lwork, int *info, int side_len, int uplo_len, int trans_len);
extern void cupgtr_(char *uplo, int *n, complex *ap, complex *tau, complex *q, int *ldq, complex *work, int *info, int uplo_len);
extern void cupmtr_(char *side, char *uplo, char *trans, int *m, int *n, complex *ap, complex *tau, complex *c, int *ldc, complex *work, int *info, int side_len, int uplo_len, int trans_len);
extern void dbdsdc_(char *uplo, char *compq, int *n, double *d, double *e, double *u, int *ldu, double *vt, int *ldvt, double *q, int *iq, double *work, int *iwork, int *info, int uplo_len, int compq_len);
extern void dbdsqr_(char *uplo, int *n, int *ncvt, int *nru, int *ncc, double *d, double *e, double *vt, int *ldvt, double *u, int *ldu, double *c, int *ldc, double *work, int *info, int uplo_len);
extern void ddisna_(char *job, int *m, int *n, double *d, double *sep, int *info, int job_len);
extern void dgbbrd_(char *vect, int *m, int *n, int *ncc, int *kl, int *ku, double *ab, int *ldab, double *d, double *e, double *q, int *ldq, double *pt, int *ldpt, double *c, int *ldc, double *work, int *info, int vect_len);
extern void dgbcon_(char *norm, int *n, int *kl, int *ku, double *ab, int *ldab, int *ipiv, double *anorm, double *rcond, double *work, int *iwork, int *info, int norm_len);
extern void dgbequ_(int *m, int *n, int *kl, int *ku, double *ab, int *ldab, double *r, double *c, double *rowcnd, double *colcnd, double *amax, int *info);
extern void dgbrfs_(char *trans, int *n, int *kl, int *ku, int *nrhs, double *ab, int *ldab, double *afb, int *ldafb, int *ipiv, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int trans_len);
extern void dgbsv_(int *n, int *kl, int *ku, int *nrhs, double *ab, int *ldab, int *ipiv, double *b, int *ldb, int *info);
extern void dgbsvx_(char *fact, char *trans, int *n, int *kl, int *ku, int *nrhs, double *ab, int *ldab, double *afb, int *ldafb, int *ipiv, char *equed, double *r, double *c, double *b, int *ldb, double *x, int *ldx, double *rcond, double *ferr, double *berr, double *work, int *iwork, int *info, int fact_len, int trans_len, int equed_len);
extern void dgbtrf_(int *m, int *n, int *kl, int *ku, double *ab, int *ldab, int *ipiv, int *info);
extern void dgbtrs_(char *trans, int *n, int *kl, int *ku, int *nrhs, double *ab, int *ldab, int *ipiv, double *b, int *ldb, int *info, int trans_len);
extern void dgebak_(char *job, char *side, int *n, int *ilo, int *ihi, double *scale, int *m, double *v, int *ldv, int *info, int job_len, int side_len);
extern void dgebal_(char *job, int *n, double *a, int *lda, int *ilo, int *ihi, double *scale, int *info, int job_len);
extern void dgebrd_(int *m, int *n, double *a, int *lda, double *d, double *e, double *tauq, double *taup, double *work, int *lwork, int *info);
extern void dgecon_(char *norm, int *n, double *a, int *lda, double *anorm, double *rcond, double *work, int *iwork, int *info, int norm_len);
extern void dgeequ_(int *m, int *n, double *a, int *lda, double *r, double *c, double *rowcnd, double *colcnd, double *amax, int *info);
extern void dgees_(char *jobvs, char *sort, ACML_DGEES_SELECT select, int *n, double *a, int *lda, int *sdim, double *wr, double *wi, double *vs, int *ldvs, double *work, int *lwork, int *bwork, int *info, int jobvs_len, int sort_len);
extern void dgeesx_(char *jobvs, char *sort, ACML_DGEESX_SELECT select, char *sense, int *n, double *a, int *lda, int *sdim, double *wr, double *wi, double *vs, int *ldvs, double *rconde, double *rcondv, double *work, int *lwork, int *iwork, int *liwork, int *bwork, int *info, int jobvs_len, int sort_len, int sense_len);
extern void dgeev_(char *jobvl, char *jobvr, int *n, double *a, int *lda, double *wr, double *wi, double *vl, int *ldvl, double *vr, int *ldvr, double *work, int *lwork, int *info, int jobvl_len, int jobvr_len);
extern void dgeevx_(char *balanc, char *jobvl, char *jobvr, char *sense, int *n, double *a, int *lda, double *wr, double *wi, double *vl, int *ldvl, double *vr, int *ldvr, int *ilo, int *ihi, double *scale, double *abnrm, double *rconde, double *rcondv, double *work, int *lwork, int *iwork, int *info, int balanc_len, int jobvl_len, int jobvr_len, int sense_len);
extern void dgegs_(char *jobvsl, char *jobvsr, int *n, double *a, int *lda, double *b, int *ldb, double *alphar, double *alphai, double *beta, double *vsl, int *ldvsl, double *vsr, int *ldvsr, double *work, int *lwork, int *info, int jobvsl_len, int jobvsr_len);
extern void dgegv_(char *jobvl, char *jobvr, int *n, double *a, int *lda, double *b, int *ldb, double *alphar, double *alphai, double *beta, double *vl, int *ldvl, double *vr, int *ldvr, double *work, int *lwork, int *info, int jobvl_len, int jobvr_len);
extern void dgehrd_(int *n, int *ilo, int *ihi, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dgelqf_(int *m, int *n, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dgels_(char *trans, int *m, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, double *work, int *lwork, int *info, int trans_len);
extern void dgelsd_(int *m, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, double *s, double *rcond, int *rank, double *work, int *lwork, int *iwork, int *info);
extern void dgelss_(int *m, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, double *s, double *rcond, int *rank, double *work, int *lwork, int *info);
extern void dgelsx_(int *m, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, int *jpvt, double *rcond, int *rank, double *work, int *info);
extern void dgelsy_(int *m, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, int *jpvt, double *rcond, int *rank, double *work, int *lwork, int *info);
extern void dgeqlf_(int *m, int *n, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dgeqp3_(int *m, int *n, double *a, int *lda, int *jpvt, double *tau, double *work, int *lwork, int *info);
extern void dgeqpf_(int *m, int *n, double *a, int *lda, int *jpvt, double *tau, double *work, int *info);
extern void dgeqrf_(int *m, int *n, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dgerfs_(char *trans, int *n, int *nrhs, double *a, int *lda, double *af, int *ldaf, int *ipiv, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int trans_len);
extern void dgerqf_(int *m, int *n, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dgesdd_(char *jobz, int *m, int *n, double *a, int *lda, double *s, double *u, int *ldu, double *vt, int *ldvt, double *work, int *lwork, int *iwork, int *info, int jobz_len);
extern void dgesv_(int *n, int *nrhs, double *a, int *lda, int *ipiv, double *b, int *ldb, int *info);
extern void dgesvd_(char *jobu, char *jobvt, int *m, int *n, double *a, int *lda, double *s, double *u, int *ldu, double *vt, int *ldvt, double *work, int *lwork, int *info, int jobu_len, int jobvt_len);
extern void dgesvx_(char *fact, char *trans, int *n, int *nrhs, double *a, int *lda, double *af, int *ldaf, int *ipiv, char *equed, double *r, double *c, double *b, int *ldb, double *x, int *ldx, double *rcond, double *ferr, double *berr, double *work, int *iwork, int *info, int fact_len, int trans_len, int equed_len);
extern void dgetrf_(int *m, int *n, double *a, int *lda, int *ipiv, int *info);
extern void dgetri_(int *n, double *a, int *lda, int *ipiv, double *work, int *lwork, int *info);
extern void dgetrs_(char *trans, int *n, int *nrhs, double *a, int *lda, int *ipiv, double *b, int *ldb, int *info, int trans_len);
extern void dggbak_(char *job, char *side, int *n, int *ilo, int *ihi, double *lscale, double *rscale, int *m, double *v, int *ldv, int *info, int job_len, int side_len);
extern void dggbal_(char *job, int *n, double *a, int *lda, double *b, int *ldb, int *ilo, int *ihi, double *lscale, double *rscale, double *work, int *info, int job_len);
extern void dgges_(char *jobvsl, char *jobvsr, char *sort, ACML_DGGES_DELCTG delctg, int *n, double *a, int *lda, double *b, int *ldb, int *sdim, double *alphar, double *alphai, double *beta, double *vsl, int *ldvsl, double *vsr, int *ldvsr, double *work, int *lwork, int *bwork, int *info, int jobvsl_len, int jobvsr_len, int sort_len);
extern void dggesx_(char *jobvsl, char *jobvsr, char *sort, ACML_DGGESX_DELCTG delctg, char *sense, int *n, double *a, int *lda, double *b, int *ldb, int *sdim, double *alphar, double *alphai, double *beta, double *vsl, int *ldvsl, double *vsr, int *ldvsr, double *rconde, double *rcondv, double *work, int *lwork, int *iwork, int *liwork, int *bwork, int *info, int jobvsl_len, int jobvsr_len, int sort_len, int sense_len);
extern void dggev_(char *jobvl, char *jobvr, int *n, double *a, int *lda, double *b, int *ldb, double *alphar, double *alphai, double *beta, double *vl, int *ldvl, double *vr, int *ldvr, double *work, int *lwork, int *info, int jobvl_len, int jobvr_len);
extern void dggevx_(char *balanc, char *jobvl, char *jobvr, char *sense, int *n, double *a, int *lda, double *b, int *ldb, double *alphar, double *alphai, double *beta, double *vl, int *ldvl, double *vr, int *ldvr, int *ilo, int *ihi, double *lscale, double *rscale, double *abnrm, double *bbnrm, double *rconde, double *rcondv, double *work, int *lwork, int *iwork, int *bwork, int *info, int balanc_len, int jobvl_len, int jobvr_len, int sense_len);
extern void dggglm_(int *n, int *m, int *p, double *a, int *lda, double *b, int *ldb, double *d, double *x, double *y, double *work, int *lwork, int *info);
extern void dgghrd_(char *compq, char *compz, int *n, int *ilo, int *ihi, double *a, int *lda, double *b, int *ldb, double *q, int *ldq, double *z, int *ldz, int *info, int compq_len, int compz_len);
extern void dgglse_(int *m, int *n, int *p, double *a, int *lda, double *b, int *ldb, double *c, double *d, double *x, double *work, int *lwork, int *info);
extern void dggqrf_(int *n, int *m, int *p, double *a, int *lda, double *taua, double *b, int *ldb, double *taub, double *work, int *lwork, int *info);
extern void dggrqf_(int *m, int *p, int *n, double *a, int *lda, double *taua, double *b, int *ldb, double *taub, double *work, int *lwork, int *info);
extern void dggsvd_(char *jobu, char *jobv, char *jobq, int *m, int *n, int *p, int *k, int *l, double *a, int *lda, double *b, int *ldb, double *alpha, double *beta, double *u, int *ldu, double *v, int *ldv, double *q, int *ldq, double *work, int *iwork, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void dggsvp_(char *jobu, char *jobv, char *jobq, int *m, int *p, int *n, double *a, int *lda, double *b, int *ldb, double *tola, double *tolb, int *k, int *l, double *u, int *ldu, double *v, int *ldv, double *q, int *ldq, int *iwork, double *tau, double *work, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void dgtcon_(char *norm, int *n, double *dl, double *d, double *du, double *du2, int *ipiv, double *anorm, double *rcond, double *work, int *iwork, int *info, int norm_len);
extern void dgtrfs_(char *trans, int *n, int *nrhs, double *dl, double *d, double *du, double *dlf, double *df, double *duf, double *du2, int *ipiv, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int trans_len);
extern void dgtsv_(int *n, int *nrhs, double *dl, double *d, double *du, double *b, int *ldb, int *info);
extern void dgtsvx_(char *fact, char *trans, int *n, int *nrhs, double *dl, double *d, double *du, double *dlf, double *df, double *duf, double *du2, int *ipiv, double *b, int *ldb, double *x, int *ldx, double *rcond, double *ferr, double *berr, double *work, int *iwork, int *info, int fact_len, int trans_len);
extern void dgttrf_(int *n, double *dl, double *d, double *du, double *du2, int *ipiv, int *info);
extern void dgttrs_(char *trans, int *n, int *nrhs, double *dl, double *d, double *du, double *du2, int *ipiv, double *b, int *ldb, int *info, int trans_len);
extern void dhgeqz_(char *job, char *compq, char *compz, int *n, int *ilo, int *ihi, double *a, int *lda, double *b, int *ldb, double *alphar, double *alphai, double *beta, double *q, int *ldq, double *z, int *ldz, double *work, int *lwork, int *info, int job_len, int compq_len, int compz_len);
extern void dhsein_(char *side, char *eigsrc, char *initv, int *select, int *n, double *h, int *ldh, double *wr, double *wi, double *vl, int *ldvl, double *vr, int *ldvr, int *mm, int *m, double *work, int *ifaill, int *ifailr, int *info, int side_len, int eigsrc_len, int initv_len);
extern void dhseqr_(char *job, char *compz, int *n, int *ilo, int *ihi, double *h, int *ldh, double *wr, double *wi, double *z, int *ldz, double *work, int *lwork, int *info, int job_len, int compz_len);
extern double dlamch_(char *cmach, int cmach_len);
extern void dopgtr_(char *uplo, int *n, double *ap, double *tau, double *q, int *ldq, double *work, int *info, int uplo_len);
extern void dopmtr_(char *side, char *uplo, char *trans, int *m, int *n, double *ap, double *tau, double *c, int *ldc, double *work, int *info, int side_len, int uplo_len, int trans_len);
extern void dorgbr_(char *vect, int *m, int *n, int *k, double *a, int *lda, double *tau, double *work, int *lwork, int *info, int vect_len);
extern void dorghr_(int *n, int *ilo, int *ihi, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dorglq_(int *m, int *n, int *k, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dorgql_(int *m, int *n, int *k, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dorgqr_(int *m, int *n, int *k, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dorgrq_(int *m, int *n, int *k, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void dorgtr_(char *uplo, int *n, double *a, int *lda, double *tau, double *work, int *lwork, int *info, int uplo_len);
extern void dormbr_(char *vect, char *side, char *trans, int *m, int *n, int *k, double *a, int *lda, double *tau, double *c, int *ldc, double *work, int *lwork, int *info, int vect_len, int side_len, int trans_len);
extern void dormhr_(char *side, char *trans, int *m, int *n, int *ilo, int *ihi, double *a, int *lda, double *tau, double *c, int *ldc, double *work, int *lwork, int *info, int side_len, int trans_len);
extern void dormlq_(char *side, char *trans, int *m, int *n, int *k, double *a, int *lda, double *tau, double *c, int *ldc, double *work, int *lwork, int *info, int side_len, int trans_len);
extern void dormql_(char *side, char *trans, int *m, int *n, int *k, double *a, int *lda, double *tau, double *c, int *ldc, double *work, int *lwork, int *info, int side_len, int trans_len);
extern void dormqr_(char *side, char *trans, int *m, int *n, int *k, double *a, int *lda, double *tau, double *c, int *ldc, double *work, int *lwork, int *info, int side_len, int trans_len);
extern void dormr3_(char *side, char *trans, int *m, int *n, int *k, int *l, double *a, int *lda, double *tau, double *c, int *ldc, double *work, int *info, int side_len, int trans_len);
extern void dormrq_(char *side, char *trans, int *m, int *n, int *k, double *a, int *lda, double *tau, double *c, int *ldc, double *work, int *lwork, int *info, int side_len, int trans_len);
extern void dormrz_(char *side, char *trans, int *m, int *n, int *k, int *l, double *a, int *lda, double *tau, double *c, int *ldc, double *work, int *lwork, int *info, int side_len, int trans_len);
extern void dormtr_(char *side, char *uplo, char *trans, int *m, int *n, double *a, int *lda, double *tau, double *c, int *ldc, double *work, int *lwork, int *info, int side_len, int uplo_len, int trans_len);
extern void dpbcon_(char *uplo, int *n, int *kd, double *ab, int *ldab, double *anorm, double *rcond, double *work, int *iwork, int *info, int uplo_len);
extern void dpbequ_(char *uplo, int *n, int *kd, double *ab, int *ldab, double *s, double *scond, double *amax, int *info, int uplo_len);
extern void dpbrfs_(char *uplo, int *n, int *kd, int *nrhs, double *ab, int *ldab, double *afb, int *ldafb, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int uplo_len);
extern void dpbstf_(char *uplo, int *n, int *kd, double *ab, int *ldab, int *info, int uplo_len);
extern void dpbsv_(char *uplo, int *n, int *kd, int *nrhs, double *ab, int *ldab, double *b, int *ldb, int *info, int uplo_len);
extern void dpbsvx_(char *fact, char *uplo, int *n, int *kd, int *nrhs, double *ab, int *ldab, double *afb, int *ldafb, char *equed, double *s, double *b, int *ldb, double *x, int *ldx, double *rcond, double *ferr, double *berr, double *work, int *iwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void dpbtrf_(char *uplo, int *n, int *kd, double *ab, int *ldab, int *info, int uplo_len);
extern void dpbtrs_(char *uplo, int *n, int *kd, int *nrhs, double *ab, int *ldab, double *b, int *ldb, int *info, int uplo_len);
extern void dpocon_(char *uplo, int *n, double *a, int *lda, double *anorm, double *rcond, double *work, int *iwork, int *info, int uplo_len);
extern void dpoequ_(int *n, double *a, int *lda, double *s, double *scond, double *amax, int *info);
extern void dporfs_(char *uplo, int *n, int *nrhs, double *a, int *lda, double *af, int *ldaf, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int uplo_len);
extern void dposv_(char *uplo, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, int *info, int uplo_len);
extern void dposvx_(char *fact, char *uplo, int *n, int *nrhs, double *a, int *lda, double *af, int *ldaf, char *equed, double *s, double *b, int *ldb, double *x, int *ldx, double *rcond, double *ferr, double *berr, double *work, int *iwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void dpotrf_(char *uplo, int *n, double *a, int *lda, int *info, int uplo_len);
extern void dpotri_(char *uplo, int *n, double *a, int *lda, int *info, int uplo_len);
extern void dpotrs_(char *uplo, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, int *info, int uplo_len);
extern void dppcon_(char *uplo, int *n, double *ap, double *anorm, double *rcond, double *work, int *iwork, int *info, int uplo_len);
extern void dppequ_(char *uplo, int *n, double *ap, double *s, double *scond, double *amax, int *info, int uplo_len);
extern void dpprfs_(char *uplo, int *n, int *nrhs, double *ap, double *afp, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int uplo_len);
extern void dppsv_(char *uplo, int *n, int *nrhs, double *ap, double *b, int *ldb, int *info, int uplo_len);
extern void dppsvx_(char *fact, char *uplo, int *n, int *nrhs, double *ap, double *afp, char *equed, double *s, double *b, int *ldb, double *x, int *ldx, double *rcond, double *ferr, double *berr, double *work, int *iwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void dpptrf_(char *uplo, int *n, double *ap, int *info, int uplo_len);
extern void dpptri_(char *uplo, int *n, double *ap, int *info, int uplo_len);
extern void dpptrs_(char *uplo, int *n, int *nrhs, double *ap, double *b, int *ldb, int *info, int uplo_len);
extern void dptcon_(int *n, double *d, double *e, double *anorm, double *rcond, double *work, int *info);
extern void dpteqr_(char *compz, int *n, double *d, double *e, double *z, int *ldz, double *work, int *info, int compz_len);
extern void dptrfs_(int *n, int *nrhs, double *d, double *e, double *df, double *ef, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *info);
extern void dptsv_(int *n, int *nrhs, double *d, double *e, double *b, int *ldb, int *info);
extern void dptsvx_(char *fact, int *n, int *nrhs, double *d, double *e, double *df, double *ef, double *b, int *ldb, double *x, int *ldx, double *rcond, double *ferr, double *berr, double *work, int *info, int fact_len);
extern void dpttrf_(int *n, double *d, double *e, int *info);
extern void dpttrs_(int *n, int *nrhs, double *d, double *e, double *b, int *ldb, int *info);
extern void dsbev_(char *jobz, char *uplo, int *n, int *kd, double *ab, int *ldab, double *w, double *z, int *ldz, double *work, int *info, int jobz_len, int uplo_len);
extern void dsbevd_(char *jobz, char *uplo, int *n, int *kd, double *ab, int *ldab, double *w, double *z, int *ldz, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void dsbevx_(char *jobz, char *range, char *uplo, int *n, int *kd, double *ab, int *ldab, double *q, int *ldq, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, double *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void dsbgst_(char *vect, char *uplo, int *n, int *ka, int *kb, double *ab, int *ldab, double *bb, int *ldbb, double *x, int *ldx, double *work, int *info, int vect_len, int uplo_len);
extern void dsbgv_(char *jobz, char *uplo, int *n, int *ka, int *kb, double *ab, int *ldab, double *bb, int *ldbb, double *w, double *z, int *ldz, double *work, int *info, int jobz_len, int uplo_len);
extern void dsbgvd_(char *jobz, char *uplo, int *n, int *ka, int *kb, double *ab, int *ldab, double *bb, int *ldbb, double *w, double *z, int *ldz, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void dsbgvx_(char *jobz, char *range, char *uplo, int *n, int *ka, int *kb, double *ab, int *ldab, double *bb, int *ldbb, double *q, int *ldq, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, double *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void dsbtrd_(char *vect, char *uplo, int *n, int *kd, double *ab, int *ldab, double *d, double *e, double *q, int *ldq, double *work, int *info, int vect_len, int uplo_len);
extern double dsecnd_(void);
extern void dspcon_(char *uplo, int *n, double *ap, int *ipiv, double *anorm, double *rcond, double *work, int *iwork, int *info, int uplo_len);
extern void dspev_(char *jobz, char *uplo, int *n, double *ap, double *w, double *z, int *ldz, double *work, int *info, int jobz_len, int uplo_len);
extern void dspevd_(char *jobz, char *uplo, int *n, double *ap, double *w, double *z, int *ldz, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void dspevx_(char *jobz, char *range, char *uplo, int *n, double *ap, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, double *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void dspgst_(int *itype, char *uplo, int *n, double *ap, double *bp, int *info, int uplo_len);
extern void dspgv_(int *itype, char *jobz, char *uplo, int *n, double *ap, double *bp, double *w, double *z, int *ldz, double *work, int *info, int jobz_len, int uplo_len);
extern void dspgvd_(int *itype, char *jobz, char *uplo, int *n, double *ap, double *bp, double *w, double *z, int *ldz, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void dspgvx_(int *itype, char *jobz, char *range, char *uplo, int *n, double *ap, double *bp, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, double *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void dsprfs_(char *uplo, int *n, int *nrhs, double *ap, double *afp, int *ipiv, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int uplo_len);
extern void dspsv_(char *uplo, int *n, int *nrhs, double *ap, int *ipiv, double *b, int *ldb, int *info, int uplo_len);
extern void dspsvx_(char *fact, char *uplo, int *n, int *nrhs, double *ap, double *afp, int *ipiv, double *b, int *ldb, double *x, int *ldx, double *rcond, double *ferr, double *berr, double *work, int *iwork, int *info, int fact_len, int uplo_len);
extern void dsptrd_(char *uplo, int *n, double *ap, double *d, double *e, double *tau, int *info, int uplo_len);
extern void dsptrf_(char *uplo, int *n, double *ap, int *ipiv, int *info, int uplo_len);
extern void dsptri_(char *uplo, int *n, double *ap, int *ipiv, double *work, int *info, int uplo_len);
extern void dsptrs_(char *uplo, int *n, int *nrhs, double *ap, int *ipiv, double *b, int *ldb, int *info, int uplo_len);
extern void dstebz_(char *range, char *order, int *n, double *vl, double *vu, int *il, int *iu, double *abstol, double *d, double *e, int *m, int *nsplit, double *w, int *iblock, int *isplit, double *work, int *iwork, int *info, int range_len, int order_len);
extern void dstedc_(char *compz, int *n, double *d, double *e, double *z, int *ldz, double *work, int *lwork, int *iwork, int *liwork, int *info, int compz_len);
extern void dstegr_(char *jobz, char *range, int *n, double *d, double *e, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, int *isuppz, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len);
extern void dstein_(int *n, double *d, double *e, int *m, double *w, int *iblock, int *isplit, double *z, int *ldz, double *work, int *iwork, int *ifail, int *info);
extern void dsteqr_(char *compz, int *n, double *d, double *e, double *z, int *ldz, double *work, int *info, int compz_len);
extern void dsterf_(int *n, double *d, double *e, int *info);
extern void dstev_(char *jobz, int *n, double *d, double *e, double *z, int *ldz, double *work, int *info, int jobz_len);
extern void dstevd_(char *jobz, int *n, double *d, double *e, double *z, int *ldz, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len);
extern void dstevr_(char *jobz, char *range, int *n, double *d, double *e, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, int *isuppz, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len);
extern void dstevx_(char *jobz, char *range, int *n, double *d, double *e, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, double *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len);
extern void dsycon_(char *uplo, int *n, double *a, int *lda, int *ipiv, double *anorm, double *rcond, double *work, int *iwork, int *info, int uplo_len);
extern void dsyev_(char *jobz, char *uplo, int *n, double *a, int *lda, double *w, double *work, int *lwork, int *info, int jobz_len, int uplo_len);
extern void dsyevd_(char *jobz, char *uplo, int *n, double *a, int *lda, double *w, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void dsyevr_(char *jobz, char *range, char *uplo, int *n, double *a, int *lda, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, int *isuppz, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len, int uplo_len);
extern void dsyevx_(char *jobz, char *range, char *uplo, int *n, double *a, int *lda, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, double *work, int *lwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void dsygst_(int *itype, char *uplo, int *n, double *a, int *lda, double *b, int *ldb, int *info, int uplo_len);
extern void dsygv_(int *itype, char *jobz, char *uplo, int *n, double *a, int *lda, double *b, int *ldb, double *w, double *work, int *lwork, int *info, int jobz_len, int uplo_len);
extern void dsygvd_(int *itype, char *jobz, char *uplo, int *n, double *a, int *lda, double *b, int *ldb, double *w, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void dsygvx_(int *itype, char *jobz, char *range, char *uplo, int *n, double *a, int *lda, double *b, int *ldb, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, double *z, int *ldz, double *work, int *lwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void dsyrfs_(char *uplo, int *n, int *nrhs, double *a, int *lda, double *af, int *ldaf, int *ipiv, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int uplo_len);
extern void dsysv_(char *uplo, int *n, int *nrhs, double *a, int *lda, int *ipiv, double *b, int *ldb, double *work, int *lwork, int *info, int uplo_len);
extern void dsysvx_(char *fact, char *uplo, int *n, int *nrhs, double *a, int *lda, double *af, int *ldaf, int *ipiv, double *b, int *ldb, double *x, int *ldx, double *rcond, double *ferr, double *berr, double *work, int *lwork, int *iwork, int *info, int fact_len, int uplo_len);
extern void dsytrd_(char *uplo, int *n, double *a, int *lda, double *d, double *e, double *tau, double *work, int *lwork, int *info, int uplo_len);
extern void dsytrf_(char *uplo, int *n, double *a, int *lda, int *ipiv, double *work, int *lwork, int *info, int uplo_len);
extern void dsytri_(char *uplo, int *n, double *a, int *lda, int *ipiv, double *work, int *info, int uplo_len);
extern void dsytrs_(char *uplo, int *n, int *nrhs, double *a, int *lda, int *ipiv, double *b, int *ldb, int *info, int uplo_len);
extern void dtbcon_(char *norm, char *uplo, char *diag, int *n, int *kd, double *ab, int *ldab, double *rcond, double *work, int *iwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void dtbrfs_(char *uplo, char *trans, char *diag, int *n, int *kd, int *nrhs, double *ab, int *ldab, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void dtbtrs_(char *uplo, char *trans, char *diag, int *n, int *kd, int *nrhs, double *ab, int *ldab, double *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void dtgevc_(char *side, char *howmny, int *select, int *n, double *a, int *lda, double *b, int *ldb, double *vl, int *ldvl, double *vr, int *ldvr, int *mm, int *m, double *work, int *info, int side_len, int howmny_len);
extern void dtgexc_(int *wantq, int *wantz, int *n, double *a, int *lda, double *b, int *ldb, double *q, int *ldq, double *z, int *ldz, int *ifst, int *ilst, double *work, int *lwork, int *info);
extern void dtgsen_(int *ijob, int *wantq, int *wantz, int *select, int *n, double *a, int *lda, double *b, int *ldb, double *alphar, double *alphai, double *beta, double *q, int *ldq, double *z, int *ldz, int *m, double *pl, double *pr, double *dif, double *work, int *lwork, int *iwork, int *liwork, int *info);
extern void dtgsja_(char *jobu, char *jobv, char *jobq, int *m, int *p, int *n, int *k, int *l, double *a, int *lda, double *b, int *ldb, double *tola, double *tolb, double *alpha, double *beta, double *u, int *ldu, double *v, int *ldv, double *q, int *ldq, double *work, int *ncycle, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void dtgsna_(char *job, char *howmny, int *select, int *n, double *a, int *lda, double *b, int *ldb, double *vl, int *ldvl, double *vr, int *ldvr, double *s, double *dif, int *mm, int *m, double *work, int *lwork, int *iwork, int *info, int job_len, int howmny_len);
extern void dtgsyl_(char *trans, int *ijob, int *m, int *n, double *a, int *lda, double *b, int *ldb, double *c, int *ldc, double *d, int *ldd, double *e, int *lde, double *f, int *ldf, double *scale, double *dif, double *work, int *lwork, int *iwork, int *info, int trans_len);
extern void dtpcon_(char *norm, char *uplo, char *diag, int *n, double *ap, double *rcond, double *work, int *iwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void dtprfs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, double *ap, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void dtptri_(char *uplo, char *diag, int *n, double *ap, int *info, int uplo_len, int diag_len);
extern void dtptrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, double *ap, double *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void dtrcon_(char *norm, char *uplo, char *diag, int *n, double *a, int *lda, double *rcond, double *work, int *iwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void dtrevc_(char *side, char *howmny, int *select, int *n, double *t, int *ldt, double *vl, int *ldvl, double *vr, int *ldvr, int *mm, int *m, double *work, int *info, int side_len, int howmny_len);
extern void dtrexc_(char *compq, int *n, double *t, int *ldt, double *q, int *ldq, int *ifst, int *ilst, double *work, int *info, int compq_len);
extern void dtrrfs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, double *x, int *ldx, double *ferr, double *berr, double *work, int *iwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void dtrsen_(char *job, char *compq, int *select, int *n, double *t, int *ldt, double *q, int *ldq, double *wr, double *wi, int *m, double *s, double *sep, double *work, int *lwork, int *iwork, int *liwork, int *info, int job_len, int compq_len);
extern void dtrsna_(char *job, char *howmny, int *select, int *n, double *t, int *ldt, double *vl, int *ldvl, double *vr, int *ldvr, double *s, double *sep, int *mm, int *m, double *work, int *ldwork, int *iwork, int *info, int job_len, int howmny_len);
extern void dtrsyl_(char *trana, char *tranb, int *isgn, int *m, int *n, double *a, int *lda, double *b, int *ldb, double *c, int *ldc, double *scale, int *info, int trana_len, int tranb_len);
extern void dtrtri_(char *uplo, char *diag, int *n, double *a, int *lda, int *info, int uplo_len, int diag_len);
extern void dtrtrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void dtzrqf_(int *m, int *n, double *a, int *lda, double *tau, int *info);
extern void dtzrzf_(int *m, int *n, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern void sbdsdc_(char *uplo, char *compq, int *n, float *d, float *e, float *u, int *ldu, float *vt, int *ldvt, float *q, int *iq, float *work, int *iwork, int *info, int uplo_len, int compq_len);
extern void sbdsqr_(char *uplo, int *n, int *ncvt, int *nru, int *ncc, float *d, float *e, float *vt, int *ldvt, float *u, int *ldu, float *c, int *ldc, float *work, int *info, int uplo_len);
extern void sdisna_(char *job, int *m, int *n, float *d, float *sep, int *info, int job_len);

#ifdef FORTRAN_FLOAT_FUNCTIONS_RETURN_DOUBLE
extern double second_(void);
#else
extern float second_(void);
#endif

extern void sgbbrd_(char *vect, int *m, int *n, int *ncc, int *kl, int *ku, float *ab, int *ldab, float *d, float *e, float *q, int *ldq, float *pt, int *ldpt, float *c, int *ldc, float *work, int *info, int vect_len);
extern void sgbcon_(char *norm, int *n, int *kl, int *ku, float *ab, int *ldab, int *ipiv, float *anorm, float *rcond, float *work, int *iwork, int *info, int norm_len);
extern void sgbequ_(int *m, int *n, int *kl, int *ku, float *ab, int *ldab, float *r, float *c, float *rowcnd, float *colcnd, float *amax, int *info);
extern void sgbrfs_(char *trans, int *n, int *kl, int *ku, int *nrhs, float *ab, int *ldab, float *afb, int *ldafb, int *ipiv, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int trans_len);
extern void sgbsv_(int *n, int *kl, int *ku, int *nrhs, float *ab, int *ldab, int *ipiv, float *b, int *ldb, int *info);
extern void sgbsvx_(char *fact, char *trans, int *n, int *kl, int *ku, int *nrhs, float *ab, int *ldab, float *afb, int *ldafb, int *ipiv, char *equed, float *r, float *c, float *b, int *ldb, float *x, int *ldx, float *rcond, float *ferr, float *berr, float *work, int *iwork, int *info, int fact_len, int trans_len, int equed_len);
extern void sgbtrf_(int *m, int *n, int *kl, int *ku, float *ab, int *ldab, int *ipiv, int *info);
extern void sgbtrs_(char *trans, int *n, int *kl, int *ku, int *nrhs, float *ab, int *ldab, int *ipiv, float *b, int *ldb, int *info, int trans_len);
extern void sgebak_(char *job, char *side, int *n, int *ilo, int *ihi, float *scale, int *m, float *v, int *ldv, int *info, int job_len, int side_len);
extern void sgebal_(char *job, int *n, float *a, int *lda, int *ilo, int *ihi, float *scale, int *info, int job_len);
extern void sgebrd_(int *m, int *n, float *a, int *lda, float *d, float *e, float *tauq, float *taup, float *work, int *lwork, int *info);
extern void sgecon_(char *norm, int *n, float *a, int *lda, float *anorm, float *rcond, float *work, int *iwork, int *info, int norm_len);
extern void sgeequ_(int *m, int *n, float *a, int *lda, float *r, float *c, float *rowcnd, float *colcnd, float *amax, int *info);
extern void sgees_(char *jobvs, char *sort, ACML_SGEES_SELECT select, int *n, float *a, int *lda, int *sdim, float *wr, float *wi, float *vs, int *ldvs, float *work, int *lwork, int *bwork, int *info, int jobvs_len, int sort_len);
extern void sgeesx_(char *jobvs, char *sort, ACML_SGEESX_SELECT select, char *sense, int *n, float *a, int *lda, int *sdim, float *wr, float *wi, float *vs, int *ldvs, float *rconde, float *rcondv, float *work, int *lwork, int *iwork, int *liwork, int *bwork, int *info, int jobvs_len, int sort_len, int sense_len);
extern void sgeev_(char *jobvl, char *jobvr, int *n, float *a, int *lda, float *wr, float *wi, float *vl, int *ldvl, float *vr, int *ldvr, float *work, int *lwork, int *info, int jobvl_len, int jobvr_len);
extern void sgeevx_(char *balanc, char *jobvl, char *jobvr, char *sense, int *n, float *a, int *lda, float *wr, float *wi, float *vl, int *ldvl, float *vr, int *ldvr, int *ilo, int *ihi, float *scale, float *abnrm, float *rconde, float *rcondv, float *work, int *lwork, int *iwork, int *info, int balanc_len, int jobvl_len, int jobvr_len, int sense_len);
extern void sgegs_(char *jobvsl, char *jobvsr, int *n, float *a, int *lda, float *b, int *ldb, float *alphar, float *alphai, float *beta, float *vsl, int *ldvsl, float *vsr, int *ldvsr, float *work, int *lwork, int *info, int jobvsl_len, int jobvsr_len);
extern void sgegv_(char *jobvl, char *jobvr, int *n, float *a, int *lda, float *b, int *ldb, float *alphar, float *alphai, float *beta, float *vl, int *ldvl, float *vr, int *ldvr, float *work, int *lwork, int *info, int jobvl_len, int jobvr_len);
extern void sgehrd_(int *n, int *ilo, int *ihi, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sgelqf_(int *m, int *n, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sgels_(char *trans, int *m, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, float *work, int *lwork, int *info, int trans_len);
extern void sgelsd_(int *m, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, float *s, float *rcond, int *rank, float *work, int *lwork, int *iwork, int *info);
extern void sgelss_(int *m, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, float *s, float *rcond, int *rank, float *work, int *lwork, int *info);
extern void sgelsx_(int *m, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, int *jpvt, float *rcond, int *rank, float *work, int *info);
extern void sgelsy_(int *m, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, int *jpvt, float *rcond, int *rank, float *work, int *lwork, int *info);
extern void sgeqlf_(int *m, int *n, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sgeqp3_(int *m, int *n, float *a, int *lda, int *jpvt, float *tau, float *work, int *lwork, int *info);
extern void sgeqpf_(int *m, int *n, float *a, int *lda, int *jpvt, float *tau, float *work, int *info);
extern void sgeqrf_(int *m, int *n, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sgerfs_(char *trans, int *n, int *nrhs, float *a, int *lda, float *af, int *ldaf, int *ipiv, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int trans_len);
extern void sgerqf_(int *m, int *n, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sgesdd_(char *jobz, int *m, int *n, float *a, int *lda, float *s, float *u, int *ldu, float *vt, int *ldvt, float *work, int *lwork, int *iwork, int *info, int jobz_len);
extern void sgesv_(int *n, int *nrhs, float *a, int *lda, int *ipiv, float *b, int *ldb, int *info);
extern void sgesvd_(char *jobu, char *jobvt, int *m, int *n, float *a, int *lda, float *s, float *u, int *ldu, float *vt, int *ldvt, float *work, int *lwork, int *info, int jobu_len, int jobvt_len);
extern void sgesvx_(char *fact, char *trans, int *n, int *nrhs, float *a, int *lda, float *af, int *ldaf, int *ipiv, char *equed, float *r, float *c, float *b, int *ldb, float *x, int *ldx, float *rcond, float *ferr, float *berr, float *work, int *iwork, int *info, int fact_len, int trans_len, int equed_len);
extern void sgetrf_(int *m, int *n, float *a, int *lda, int *ipiv, int *info);
extern void sgetri_(int *n, float *a, int *lda, int *ipiv, float *work, int *lwork, int *info);
extern void sgetrs_(char *trans, int *n, int *nrhs, float *a, int *lda, int *ipiv, float *b, int *ldb, int *info, int trans_len);
extern void sggbak_(char *job, char *side, int *n, int *ilo, int *ihi, float *lscale, float *rscale, int *m, float *v, int *ldv, int *info, int job_len, int side_len);
extern void sggbal_(char *job, int *n, float *a, int *lda, float *b, int *ldb, int *ilo, int *ihi, float *lscale, float *rscale, float *work, int *info, int job_len);
extern void sgges_(char *jobvsl, char *jobvsr, char *sort, ACML_SGGES_SELCTG selctg, int *n, float *a, int *lda, float *b, int *ldb, int *sdim, float *alphar, float *alphai, float *beta, float *vsl, int *ldvsl, float *vsr, int *ldvsr, float *work, int *lwork, int *bwork, int *info, int jobvsl_len, int jobvsr_len, int sort_len);
extern void sggesx_(char *jobvsl, char *jobvsr, char *sort, ACML_SGGESX_SELCTG selctg, char *sense, int *n, float *a, int *lda, float *b, int *ldb, int *sdim, float *alphar, float *alphai, float *beta, float *vsl, int *ldvsl, float *vsr, int *ldvsr, float *rconde, float *rcondv, float *work, int *lwork, int *iwork, int *liwork, int *bwork, int *info, int jobvsl_len, int jobvsr_len, int sort_len, int sense_len);
extern void sggev_(char *jobvl, char *jobvr, int *n, float *a, int *lda, float *b, int *ldb, float *alphar, float *alphai, float *beta, float *vl, int *ldvl, float *vr, int *ldvr, float *work, int *lwork, int *info, int jobvl_len, int jobvr_len);
extern void sggevx_(char *balanc, char *jobvl, char *jobvr, char *sense, int *n, float *a, int *lda, float *b, int *ldb, float *alphar, float *alphai, float *beta, float *vl, int *ldvl, float *vr, int *ldvr, int *ilo, int *ihi, float *lscale, float *rscale, float *abnrm, float *bbnrm, float *rconde, float *rcondv, float *work, int *lwork, int *iwork, int *bwork, int *info, int balanc_len, int jobvl_len, int jobvr_len, int sense_len);
extern void sggglm_(int *n, int *m, int *p, float *a, int *lda, float *b, int *ldb, float *d, float *x, float *y, float *work, int *lwork, int *info);
extern void sgghrd_(char *compq, char *compz, int *n, int *ilo, int *ihi, float *a, int *lda, float *b, int *ldb, float *q, int *ldq, float *z, int *ldz, int *info, int compq_len, int compz_len);
extern void sgglse_(int *m, int *n, int *p, float *a, int *lda, float *b, int *ldb, float *c, float *d, float *x, float *work, int *lwork, int *info);
extern void sggqrf_(int *n, int *m, int *p, float *a, int *lda, float *taua, float *b, int *ldb, float *taub, float *work, int *lwork, int *info);
extern void sggrqf_(int *m, int *p, int *n, float *a, int *lda, float *taua, float *b, int *ldb, float *taub, float *work, int *lwork, int *info);
extern void sggsvd_(char *jobu, char *jobv, char *jobq, int *m, int *n, int *p, int *k, int *l, float *a, int *lda, float *b, int *ldb, float *alpha, float *beta, float *u, int *ldu, float *v, int *ldv, float *q, int *ldq, float *work, int *iwork, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void sggsvp_(char *jobu, char *jobv, char *jobq, int *m, int *p, int *n, float *a, int *lda, float *b, int *ldb, float *tola, float *tolb, int *k, int *l, float *u, int *ldu, float *v, int *ldv, float *q, int *ldq, int *iwork, float *tau, float *work, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void sgtcon_(char *norm, int *n, float *dl, float *d, float *du, float *du2, int *ipiv, float *anorm, float *rcond, float *work, int *iwork, int *info, int norm_len);
extern void sgtrfs_(char *trans, int *n, int *nrhs, float *dl, float *d, float *du, float *dlf, float *df, float *duf, float *du2, int *ipiv, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int trans_len);
extern void sgtsv_(int *n, int *nrhs, float *dl, float *d, float *du, float *b, int *ldb, int *info);
extern void sgtsvx_(char *fact, char *trans, int *n, int *nrhs, float *dl, float *d, float *du, float *dlf, float *df, float *duf, float *du2, int *ipiv, float *b, int *ldb, float *x, int *ldx, float *rcond, float *ferr, float *berr, float *work, int *iwork, int *info, int fact_len, int trans_len);
extern void sgttrf_(int *n, float *dl, float *d, float *du, float *du2, int *ipiv, int *info);
extern void sgttrs_(char *trans, int *n, int *nrhs, float *dl, float *d, float *du, float *du2, int *ipiv, float *b, int *ldb, int *info, int trans_len);
extern void shgeqz_(char *job, char *compq, char *compz, int *n, int *ilo, int *ihi, float *a, int *lda, float *b, int *ldb, float *alphar, float *alphai, float *beta, float *q, int *ldq, float *z, int *ldz, float *work, int *lwork, int *info, int job_len, int compq_len, int compz_len);
extern void shsein_(char *side, char *eigsrc, char *initv, int *select, int *n, float *h, int *ldh, float *wr, float *wi, float *vl, int *ldvl, float *vr, int *ldvr, int *mm, int *m, float *work, int *ifaill, int *ifailr, int *info, int side_len, int eigsrc_len, int initv_len);
extern void shseqr_(char *job, char *compz, int *n, int *ilo, int *ihi, float *h, int *ldh, float *wr, float *wi, float *z, int *ldz, float *work, int *lwork, int *info, int job_len, int compz_len);

#ifdef FORTRAN_FLOAT_FUNCTIONS_RETURN_DOUBLE
extern double slamch_(char *cmach, int cmach_len);
#else
extern float slamch_(char *cmach, int cmach_len);
#endif

extern void sopgtr_(char *uplo, int *n, float *ap, float *tau, float *q, int *ldq, float *work, int *info, int uplo_len);
extern void sopmtr_(char *side, char *uplo, char *trans, int *m, int *n, float *ap, float *tau, float *c, int *ldc, float *work, int *info, int side_len, int uplo_len, int trans_len);
extern void sorgbr_(char *vect, int *m, int *n, int *k, float *a, int *lda, float *tau, float *work, int *lwork, int *info, int vect_len);
extern void sorghr_(int *n, int *ilo, int *ihi, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sorglq_(int *m, int *n, int *k, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sorgql_(int *m, int *n, int *k, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sorgqr_(int *m, int *n, int *k, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sorgrq_(int *m, int *n, int *k, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void sorgtr_(char *uplo, int *n, float *a, int *lda, float *tau, float *work, int *lwork, int *info, int uplo_len);
extern void sormbr_(char *vect, char *side, char *trans, int *m, int *n, int *k, float *a, int *lda, float *tau, float *c, int *ldc, float *work, int *lwork, int *info, int vect_len, int side_len, int trans_len);
extern void sormhr_(char *side, char *trans, int *m, int *n, int *ilo, int *ihi, float *a, int *lda, float *tau, float *c, int *ldc, float *work, int *lwork, int *info, int side_len, int trans_len);
extern void sormlq_(char *side, char *trans, int *m, int *n, int *k, float *a, int *lda, float *tau, float *c, int *ldc, float *work, int *lwork, int *info, int side_len, int trans_len);
extern void sormql_(char *side, char *trans, int *m, int *n, int *k, float *a, int *lda, float *tau, float *c, int *ldc, float *work, int *lwork, int *info, int side_len, int trans_len);
extern void sormqr_(char *side, char *trans, int *m, int *n, int *k, float *a, int *lda, float *tau, float *c, int *ldc, float *work, int *lwork, int *info, int side_len, int trans_len);
extern void sormr3_(char *side, char *trans, int *m, int *n, int *k, int *l, float *a, int *lda, float *tau, float *c, int *ldc, float *work, int *info, int side_len, int trans_len);
extern void sormrq_(char *side, char *trans, int *m, int *n, int *k, float *a, int *lda, float *tau, float *c, int *ldc, float *work, int *lwork, int *info, int side_len, int trans_len);
extern void sormrz_(char *side, char *trans, int *m, int *n, int *k, int *l, float *a, int *lda, float *tau, float *c, int *ldc, float *work, int *lwork, int *info, int side_len, int trans_len);
extern void sormtr_(char *side, char *uplo, char *trans, int *m, int *n, float *a, int *lda, float *tau, float *c, int *ldc, float *work, int *lwork, int *info, int side_len, int uplo_len, int trans_len);
extern void spbcon_(char *uplo, int *n, int *kd, float *ab, int *ldab, float *anorm, float *rcond, float *work, int *iwork, int *info, int uplo_len);
extern void spbequ_(char *uplo, int *n, int *kd, float *ab, int *ldab, float *s, float *scond, float *amax, int *info, int uplo_len);
extern void spbrfs_(char *uplo, int *n, int *kd, int *nrhs, float *ab, int *ldab, float *afb, int *ldafb, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int uplo_len);
extern void spbstf_(char *uplo, int *n, int *kd, float *ab, int *ldab, int *info, int uplo_len);
extern void spbsv_(char *uplo, int *n, int *kd, int *nrhs, float *ab, int *ldab, float *b, int *ldb, int *info, int uplo_len);
extern void spbsvx_(char *fact, char *uplo, int *n, int *kd, int *nrhs, float *ab, int *ldab, float *afb, int *ldafb, char *equed, float *s, float *b, int *ldb, float *x, int *ldx, float *rcond, float *ferr, float *berr, float *work, int *iwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void spbtrf_(char *uplo, int *n, int *kd, float *ab, int *ldab, int *info, int uplo_len);
extern void spbtrs_(char *uplo, int *n, int *kd, int *nrhs, float *ab, int *ldab, float *b, int *ldb, int *info, int uplo_len);
extern void spocon_(char *uplo, int *n, float *a, int *lda, float *anorm, float *rcond, float *work, int *iwork, int *info, int uplo_len);
extern void spoequ_(int *n, float *a, int *lda, float *s, float *scond, float *amax, int *info);
extern void sporfs_(char *uplo, int *n, int *nrhs, float *a, int *lda, float *af, int *ldaf, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int uplo_len);
extern void sposv_(char *uplo, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, int *info, int uplo_len);
extern void sposvx_(char *fact, char *uplo, int *n, int *nrhs, float *a, int *lda, float *af, int *ldaf, char *equed, float *s, float *b, int *ldb, float *x, int *ldx, float *rcond, float *ferr, float *berr, float *work, int *iwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void spotrf_(char *uplo, int *n, float *a, int *lda, int *info, int uplo_len);
extern void spotri_(char *uplo, int *n, float *a, int *lda, int *info, int uplo_len);
extern void spotrs_(char *uplo, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, int *info, int uplo_len);
extern void sppcon_(char *uplo, int *n, float *ap, float *anorm, float *rcond, float *work, int *iwork, int *info, int uplo_len);
extern void sppequ_(char *uplo, int *n, float *ap, float *s, float *scond, float *amax, int *info, int uplo_len);
extern void spprfs_(char *uplo, int *n, int *nrhs, float *ap, float *afp, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int uplo_len);
extern void sppsv_(char *uplo, int *n, int *nrhs, float *ap, float *b, int *ldb, int *info, int uplo_len);
extern void sppsvx_(char *fact, char *uplo, int *n, int *nrhs, float *ap, float *afp, char *equed, float *s, float *b, int *ldb, float *x, int *ldx, float *rcond, float *ferr, float *berr, float *work, int *iwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void spptrf_(char *uplo, int *n, float *ap, int *info, int uplo_len);
extern void spptri_(char *uplo, int *n, float *ap, int *info, int uplo_len);
extern void spptrs_(char *uplo, int *n, int *nrhs, float *ap, float *b, int *ldb, int *info, int uplo_len);
extern void sptcon_(int *n, float *d, float *e, float *anorm, float *rcond, float *work, int *info);
extern void spteqr_(char *compz, int *n, float *d, float *e, float *z, int *ldz, float *work, int *info, int compz_len);
extern void sptrfs_(int *n, int *nrhs, float *d, float *e, float *df, float *ef, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *info);
extern void sptsv_(int *n, int *nrhs, float *d, float *e, float *b, int *ldb, int *info);
extern void sptsvx_(char *fact, int *n, int *nrhs, float *d, float *e, float *df, float *ef, float *b, int *ldb, float *x, int *ldx, float *rcond, float *ferr, float *berr, float *work, int *info, int fact_len);
extern void spttrf_(int *n, float *d, float *e, int *info);
extern void spttrs_(int *n, int *nrhs, float *d, float *e, float *b, int *ldb, int *info);
extern void ssbev_(char *jobz, char *uplo, int *n, int *kd, float *ab, int *ldab, float *w, float *z, int *ldz, float *work, int *info, int jobz_len, int uplo_len);
extern void ssbevd_(char *jobz, char *uplo, int *n, int *kd, float *ab, int *ldab, float *w, float *z, int *ldz, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void ssbevx_(char *jobz, char *range, char *uplo, int *n, int *kd, float *ab, int *ldab, float *q, int *ldq, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, float *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void ssbgst_(char *vect, char *uplo, int *n, int *ka, int *kb, float *ab, int *ldab, float *bb, int *ldbb, float *x, int *ldx, float *work, int *info, int vect_len, int uplo_len);
extern void ssbgv_(char *jobz, char *uplo, int *n, int *ka, int *kb, float *ab, int *ldab, float *bb, int *ldbb, float *w, float *z, int *ldz, float *work, int *info, int jobz_len, int uplo_len);
extern void ssbgvd_(char *jobz, char *uplo, int *n, int *ka, int *kb, float *ab, int *ldab, float *bb, int *ldbb, float *w, float *z, int *ldz, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void ssbgvx_(char *jobz, char *range, char *uplo, int *n, int *ka, int *kb, float *ab, int *ldab, float *bb, int *ldbb, float *q, int *ldq, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, float *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void ssbtrd_(char *vect, char *uplo, int *n, int *kd, float *ab, int *ldab, float *d, float *e, float *q, int *ldq, float *work, int *info, int vect_len, int uplo_len);
extern void sspcon_(char *uplo, int *n, float *ap, int *ipiv, float *anorm, float *rcond, float *work, int *iwork, int *info, int uplo_len);
extern void sspev_(char *jobz, char *uplo, int *n, float *ap, float *w, float *z, int *ldz, float *work, int *info, int jobz_len, int uplo_len);
extern void sspevd_(char *jobz, char *uplo, int *n, float *ap, float *w, float *z, int *ldz, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void sspevx_(char *jobz, char *range, char *uplo, int *n, float *ap, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, float *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void sspgst_(int *itype, char *uplo, int *n, float *ap, float *bp, int *info, int uplo_len);
extern void sspgv_(int *itype, char *jobz, char *uplo, int *n, float *ap, float *bp, float *w, float *z, int *ldz, float *work, int *info, int jobz_len, int uplo_len);
extern void sspgvd_(int *itype, char *jobz, char *uplo, int *n, float *ap, float *bp, float *w, float *z, int *ldz, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void sspgvx_(int *itype, char *jobz, char *range, char *uplo, int *n, float *ap, float *bp, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, float *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void ssprfs_(char *uplo, int *n, int *nrhs, float *ap, float *afp, int *ipiv, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int uplo_len);
extern void sspsv_(char *uplo, int *n, int *nrhs, float *ap, int *ipiv, float *b, int *ldb, int *info, int uplo_len);
extern void sspsvx_(char *fact, char *uplo, int *n, int *nrhs, float *ap, float *afp, int *ipiv, float *b, int *ldb, float *x, int *ldx, float *rcond, float *ferr, float *berr, float *work, int *iwork, int *info, int fact_len, int uplo_len);
extern void ssptrd_(char *uplo, int *n, float *ap, float *d, float *e, float *tau, int *info, int uplo_len);
extern void ssptrf_(char *uplo, int *n, float *ap, int *ipiv, int *info, int uplo_len);
extern void ssptri_(char *uplo, int *n, float *ap, int *ipiv, float *work, int *info, int uplo_len);
extern void ssptrs_(char *uplo, int *n, int *nrhs, float *ap, int *ipiv, float *b, int *ldb, int *info, int uplo_len);
extern void sstebz_(char *range, char *order, int *n, float *vl, float *vu, int *il, int *iu, float *abstol, float *d, float *e, int *m, int *nsplit, float *w, int *iblock, int *isplit, float *work, int *iwork, int *info, int range_len, int order_len);
extern void sstedc_(char *compz, int *n, float *d, float *e, float *z, int *ldz, float *work, int *lwork, int *iwork, int *liwork, int *info, int compz_len);
extern void sstegr_(char *jobz, char *range, int *n, float *d, float *e, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, int *isuppz, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len);
extern void sstein_(int *n, float *d, float *e, int *m, float *w, int *iblock, int *isplit, float *z, int *ldz, float *work, int *iwork, int *ifail, int *info);
extern void ssteqr_(char *compz, int *n, float *d, float *e, float *z, int *ldz, float *work, int *info, int compz_len);
extern void ssterf_(int *n, float *d, float *e, int *info);
extern void sstev_(char *jobz, int *n, float *d, float *e, float *z, int *ldz, float *work, int *info, int jobz_len);
extern void sstevd_(char *jobz, int *n, float *d, float *e, float *z, int *ldz, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len);
extern void sstevr_(char *jobz, char *range, int *n, float *d, float *e, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, int *isuppz, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len);
extern void sstevx_(char *jobz, char *range, int *n, float *d, float *e, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, float *work, int *iwork, int *ifail, int *info, int jobz_len, int range_len);
extern void ssycon_(char *uplo, int *n, float *a, int *lda, int *ipiv, float *anorm, float *rcond, float *work, int *iwork, int *info, int uplo_len);
extern void ssyev_(char *jobz, char *uplo, int *n, float *a, int *lda, float *w, float *work, int *lwork, int *info, int jobz_len, int uplo_len);
extern void ssyevd_(char *jobz, char *uplo, int *n, float *a, int *lda, float *w, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void ssyevr_(char *jobz, char *range, char *uplo, int *n, float *a, int *lda, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, int *isuppz, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len, int uplo_len);
extern void ssyevx_(char *jobz, char *range, char *uplo, int *n, float *a, int *lda, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, float *work, int *lwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void ssygst_(int *itype, char *uplo, int *n, float *a, int *lda, float *b, int *ldb, int *info, int uplo_len);
extern void ssygv_(int *itype, char *jobz, char *uplo, int *n, float *a, int *lda, float *b, int *ldb, float *w, float *work, int *lwork, int *info, int jobz_len, int uplo_len);
extern void ssygvd_(int *itype, char *jobz, char *uplo, int *n, float *a, int *lda, float *b, int *ldb, float *w, float *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void ssygvx_(int *itype, char *jobz, char *range, char *uplo, int *n, float *a, int *lda, float *b, int *ldb, float *vl, float *vu, int *il, int *iu, float *abstol, int *m, float *w, float *z, int *ldz, float *work, int *lwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void ssyrfs_(char *uplo, int *n, int *nrhs, float *a, int *lda, float *af, int *ldaf, int *ipiv, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int uplo_len);
extern void ssysv_(char *uplo, int *n, int *nrhs, float *a, int *lda, int *ipiv, float *b, int *ldb, float *work, int *lwork, int *info, int uplo_len);
extern void ssysvx_(char *fact, char *uplo, int *n, int *nrhs, float *a, int *lda, float *af, int *ldaf, int *ipiv, float *b, int *ldb, float *x, int *ldx, float *rcond, float *ferr, float *berr, float *work, int *lwork, int *iwork, int *info, int fact_len, int uplo_len);
extern void ssytrd_(char *uplo, int *n, float *a, int *lda, float *d, float *e, float *tau, float *work, int *lwork, int *info, int uplo_len);
extern void ssytrf_(char *uplo, int *n, float *a, int *lda, int *ipiv, float *work, int *lwork, int *info, int uplo_len);
extern void ssytri_(char *uplo, int *n, float *a, int *lda, int *ipiv, float *work, int *info, int uplo_len);
extern void ssytrs_(char *uplo, int *n, int *nrhs, float *a, int *lda, int *ipiv, float *b, int *ldb, int *info, int uplo_len);
extern void stbcon_(char *norm, char *uplo, char *diag, int *n, int *kd, float *ab, int *ldab, float *rcond, float *work, int *iwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void stbrfs_(char *uplo, char *trans, char *diag, int *n, int *kd, int *nrhs, float *ab, int *ldab, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void stbtrs_(char *uplo, char *trans, char *diag, int *n, int *kd, int *nrhs, float *ab, int *ldab, float *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void stgevc_(char *side, char *howmny, int *select, int *n, float *a, int *lda, float *b, int *ldb, float *vl, int *ldvl, float *vr, int *ldvr, int *mm, int *m, float *work, int *info, int side_len, int howmny_len);
extern void stgexc_(int *wantq, int *wantz, int *n, float *a, int *lda, float *b, int *ldb, float *q, int *ldq, float *z, int *ldz, int *ifst, int *ilst, float *work, int *lwork, int *info);
extern void stgsen_(int *ijob, int *wantq, int *wantz, int *select, int *n, float *a, int *lda, float *b, int *ldb, float *alphar, float *alphai, float *beta, float *q, int *ldq, float *z, int *ldz, int *m, float *pl, float *pr, float *dif, float *work, int *lwork, int *iwork, int *liwork, int *info);
extern void stgsja_(char *jobu, char *jobv, char *jobq, int *m, int *p, int *n, int *k, int *l, float *a, int *lda, float *b, int *ldb, float *tola, float *tolb, float *alpha, float *beta, float *u, int *ldu, float *v, int *ldv, float *q, int *ldq, float *work, int *ncycle, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void stgsna_(char *job, char *howmny, int *select, int *n, float *a, int *lda, float *b, int *ldb, float *vl, int *ldvl, float *vr, int *ldvr, float *s, float *dif, int *mm, int *m, float *work, int *lwork, int *iwork, int *info, int job_len, int howmny_len);
extern void stgsyl_(char *trans, int *ijob, int *m, int *n, float *a, int *lda, float *b, int *ldb, float *c, int *ldc, float *d, int *ldd, float *e, int *lde, float *f, int *ldf, float *scale, float *dif, float *work, int *lwork, int *iwork, int *info, int trans_len);
extern void stpcon_(char *norm, char *uplo, char *diag, int *n, float *ap, float *rcond, float *work, int *iwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void stprfs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, float *ap, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void stptri_(char *uplo, char *diag, int *n, float *ap, int *info, int uplo_len, int diag_len);
extern void stptrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, float *ap, float *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void strcon_(char *norm, char *uplo, char *diag, int *n, float *a, int *lda, float *rcond, float *work, int *iwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void strevc_(char *side, char *howmny, int *select, int *n, float *t, int *ldt, float *vl, int *ldvl, float *vr, int *ldvr, int *mm, int *m, float *work, int *info, int side_len, int howmny_len);
extern void strexc_(char *compq, int *n, float *t, int *ldt, float *q, int *ldq, int *ifst, int *ilst, float *work, int *info, int compq_len);
extern void strrfs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, float *x, int *ldx, float *ferr, float *berr, float *work, int *iwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void strsen_(char *job, char *compq, int *select, int *n, float *t, int *ldt, float *q, int *ldq, float *wr, float *wi, int *m, float *s, float *sep, float *work, int *lwork, int *iwork, int *liwork, int *info, int job_len, int compq_len);
extern void strsna_(char *job, char *howmny, int *select, int *n, float *t, int *ldt, float *vl, int *ldvl, float *vr, int *ldvr, float *s, float *sep, int *mm, int *m, float *work, int *ldwork, int *iwork, int *info, int job_len, int howmny_len);
extern void strsyl_(char *trana, char *tranb, int *isgn, int *m, int *n, float *a, int *lda, float *b, int *ldb, float *c, int *ldc, float *scale, int *info, int trana_len, int tranb_len);
extern void strtri_(char *uplo, char *diag, int *n, float *a, int *lda, int *info, int uplo_len, int diag_len);
extern void strtrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void stzrqf_(int *m, int *n, float *a, int *lda, float *tau, int *info);
extern void stzrzf_(int *m, int *n, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern void zbdsqr_(char *uplo, int *n, int *ncvt, int *nru, int *ncc, double *d, double *e, doublecomplex *vt, int *ldvt, doublecomplex *u, int *ldu, doublecomplex *c, int *ldc, double *rwork, int *info, int uplo_len);
extern void zgbbrd_(char *vect, int *m, int *n, int *ncc, int *kl, int *ku, doublecomplex *ab, int *ldab, double *d, double *e, doublecomplex *q, int *ldq, doublecomplex *pt, int *ldpt, doublecomplex *c, int *ldc, doublecomplex *work, double *rwork, int *info, int vect_len);
extern void zgbcon_(char *norm, int *n, int *kl, int *ku, doublecomplex *ab, int *ldab, int *ipiv, double *anorm, double *rcond, doublecomplex *work, double *rwork, int *info, int norm_len);
extern void zgbequ_(int *m, int *n, int *kl, int *ku, doublecomplex *ab, int *ldab, double *r, double *c, double *rowcnd, double *colcnd, double *amax, int *info);
extern void zgbrfs_(char *trans, int *n, int *kl, int *ku, int *nrhs, doublecomplex *ab, int *ldab, doublecomplex *afb, int *ldafb, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int trans_len);
extern void zgbsv_(int *n, int *kl, int *ku, int *nrhs, doublecomplex *ab, int *ldab, int *ipiv, doublecomplex *b, int *ldb, int *info);
extern void zgbsvx_(char *fact, char *trans, int *n, int *kl, int *ku, int *nrhs, doublecomplex *ab, int *ldab, doublecomplex *afb, int *ldafb, int *ipiv, char *equed, double *r, double *c, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int fact_len, int trans_len, int equed_len);
extern void zgbtrf_(int *m, int *n, int *kl, int *ku, doublecomplex *ab, int *ldab, int *ipiv, int *info);
extern void zgbtrs_(char *trans, int *n, int *kl, int *ku, int *nrhs, doublecomplex *ab, int *ldab, int *ipiv, doublecomplex *b, int *ldb, int *info, int trans_len);
extern void zgebak_(char *job, char *side, int *n, int *ilo, int *ihi, double *scale, int *m, doublecomplex *v, int *ldv, int *info, int job_len, int side_len);
extern void zgebal_(char *job, int *n, doublecomplex *a, int *lda, int *ilo, int *ihi, double *scale, int *info, int job_len);
extern void zgebrd_(int *m, int *n, doublecomplex *a, int *lda, double *d, double *e, doublecomplex *tauq, doublecomplex *taup, doublecomplex *work, int *lwork, int *info);
extern void zgecon_(char *norm, int *n, doublecomplex *a, int *lda, double *anorm, double *rcond, doublecomplex *work, double *rwork, int *info, int norm_len);
extern void zgeequ_(int *m, int *n, doublecomplex *a, int *lda, double *r, double *c, double *rowcnd, double *colcnd, double *amax, int *info);
extern void zgees_(char *jobvs, char *sort, ACML_ZGEES_SELECT select, int *n, doublecomplex *a, int *lda, int *sdim, doublecomplex *w, doublecomplex *vs, int *ldvs, doublecomplex *work, int *lwork, double *rwork, int *bwork, int *info, int jobvs_len, int sort_len);
extern void zgeesx_(char *jobvs, char *sort, ACML_ZGEESX_SELECT select, char *sense, int *n, doublecomplex *a, int *lda, int *sdim, doublecomplex *w, doublecomplex *vs, int *ldvs, double *rconde, double *rcondv, doublecomplex *work, int *lwork, double *rwork, int *bwork, int *info, int jobvs_len, int sort_len, int sense_len);
extern void zgeev_(char *jobvl, char *jobvr, int *n, doublecomplex *a, int *lda, doublecomplex *w, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, doublecomplex *work, int *lwork, double *rwork, int *info, int jobvl_len, int jobvr_len);
extern void zgeevx_(char *balanc, char *jobvl, char *jobvr, char *sense, int *n, doublecomplex *a, int *lda, doublecomplex *w, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, int *ilo, int *ihi, double *scale, double *abnrm, double *rconde, double *rcondv, doublecomplex *work, int *lwork, double *rwork, int *info, int balanc_len, int jobvl_len, int jobvr_len, int sense_len);
extern void zgegs_(char *jobvsl, char *jobvsr, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vsl, int *ldvsl, doublecomplex *vsr, int *ldvsr, doublecomplex *work, int *lwork, double *rwork, int *info, int jobvsl_len, int jobvsr_len);
extern void zgegv_(char *jobvl, char *jobvr, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, doublecomplex *work, int *lwork, double *rwork, int *info, int jobvl_len, int jobvr_len);
extern void zgehrd_(int *n, int *ilo, int *ihi, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zgelqf_(int *m, int *n, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zgels_(char *trans, int *m, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *work, int *lwork, int *info, int trans_len);
extern void zgelsd_(int *m, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, double *s, double *rcond, int *rank, doublecomplex *work, int *lwork, double *rwork, int *iwork, int *info);
extern void zgelss_(int *m, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, double *s, double *rcond, int *rank, doublecomplex *work, int *lwork, double *rwork, int *info);
extern void zgelsx_(int *m, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int *jpvt, double *rcond, int *rank, doublecomplex *work, double *rwork, int *info);
extern void zgelsy_(int *m, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int *jpvt, double *rcond, int *rank, doublecomplex *work, int *lwork, double *rwork, int *info);
extern void zgeqlf_(int *m, int *n, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zgeqp3_(int *m, int *n, doublecomplex *a, int *lda, int *jpvt, doublecomplex *tau, doublecomplex *work, int *lwork, double *rwork, int *info);
extern void zgeqpf_(int *m, int *n, doublecomplex *a, int *lda, int *jpvt, doublecomplex *tau, doublecomplex *work, double *rwork, int *info);
extern void zgeqrf_(int *m, int *n, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zgerfs_(char *trans, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *af, int *ldaf, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int trans_len);
extern void zgerqf_(int *m, int *n, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zgesdd_(char *jobz, int *m, int *n, doublecomplex *a, int *lda, double *s, doublecomplex *u, int *ldu, doublecomplex *vt, int *ldvt, doublecomplex *work, int *lwork, double *rwork, int *iwork, int *info, int jobz_len);
extern void zgesv_(int *n, int *nrhs, doublecomplex *a, int *lda, int *ipiv, doublecomplex *b, int *ldb, int *info);
extern void zgesvd_(char *jobu, char *jobvt, int *m, int *n, doublecomplex *a, int *lda, double *s, doublecomplex *u, int *ldu, doublecomplex *vt, int *ldvt, doublecomplex *work, int *lwork, double *rwork, int *info, int jobu_len, int jobvt_len);
extern void zgesvx_(char *fact, char *trans, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *af, int *ldaf, int *ipiv, char *equed, double *r, double *c, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int fact_len, int trans_len, int equed_len);
extern void zgetrf_(int *m, int *n, doublecomplex *a, int *lda, int *ipiv, int *info);
extern void zgetri_(int *n, doublecomplex *a, int *lda, int *ipiv, doublecomplex *work, int *lwork, int *info);
extern void zgetrs_(char *trans, int *n, int *nrhs, doublecomplex *a, int *lda, int *ipiv, doublecomplex *b, int *ldb, int *info, int trans_len);
extern void zggbak_(char *job, char *side, int *n, int *ilo, int *ihi, double *lscale, double *rscale, int *m, doublecomplex *v, int *ldv, int *info, int job_len, int side_len);
extern void zggbal_(char *job, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int *ilo, int *ihi, double *lscale, double *rscale, double *work, int *info, int job_len);
extern void zgges_(char *jobvsl, char *jobvsr, char *sort, ACML_ZGGES_DELCTG delctg, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int *sdim, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vsl, int *ldvsl, doublecomplex *vsr, int *ldvsr, doublecomplex *work, int *lwork, double *rwork, int *bwork, int *info, int jobvsl_len, int jobvsr_len, int sort_len);
extern void zggesx_(char *jobvsl, char *jobvsr, char *sort, ACML_ZGGESX_DELCTG delctg, char *sense, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int *sdim, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vsl, int *ldvsl, doublecomplex *vsr, int *ldvsr, double *rconde, double *rcondv, doublecomplex *work, int *lwork, double *rwork, int *iwork, int *liwork, int *bwork, int *info, int jobvsl_len, int jobvsr_len, int sort_len, int sense_len);
extern void zggev_(char *jobvl, char *jobvr, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, doublecomplex *work, int *lwork, double *rwork, int *info, int jobvl_len, int jobvr_len);
extern void zggevx_(char *balanc, char *jobvl, char *jobvr, char *sense, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, int *ilo, int *ihi, double *lscale, double *rscale, double *abnrm, double *bbnrm, double *rconde, double *rcondv, doublecomplex *work, int *lwork, double *rwork, int *iwork, int *bwork, int *info, int balanc_len, int jobvl_len, int jobvr_len, int sense_len);
extern void zggglm_(int *n, int *m, int *p, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *d, doublecomplex *x, doublecomplex *y, doublecomplex *work, int *lwork, int *info);
extern void zgghrd_(char *compq, char *compz, int *n, int *ilo, int *ihi, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *q, int *ldq, doublecomplex *z, int *ldz, int *info, int compq_len, int compz_len);
extern void zgglse_(int *m, int *n, int *p, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *c, doublecomplex *d, doublecomplex *x, doublecomplex *work, int *lwork, int *info);
extern void zggqrf_(int *n, int *m, int *p, doublecomplex *a, int *lda, doublecomplex *taua, doublecomplex *b, int *ldb, doublecomplex *taub, doublecomplex *work, int *lwork, int *info);
extern void zggrqf_(int *m, int *p, int *n, doublecomplex *a, int *lda, doublecomplex *taua, doublecomplex *b, int *ldb, doublecomplex *taub, doublecomplex *work, int *lwork, int *info);
extern void zggsvd_(char *jobu, char *jobv, char *jobq, int *m, int *n, int *p, int *k, int *l, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, double *alpha, double *beta, doublecomplex *u, int *ldu, doublecomplex *v, int *ldv, doublecomplex *q, int *ldq, doublecomplex *work, double *rwork, int *iwork, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void zggsvp_(char *jobu, char *jobv, char *jobq, int *m, int *p, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, double *tola, double *tolb, int *k, int *l, doublecomplex *u, int *ldu, doublecomplex *v, int *ldv, doublecomplex *q, int *ldq, int *iwork, double *rwork, doublecomplex *tau, doublecomplex *work, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void zgtcon_(char *norm, int *n, doublecomplex *dl, doublecomplex *d, doublecomplex *du, doublecomplex *du2, int *ipiv, double *anorm, double *rcond, doublecomplex *work, int *info, int norm_len);
extern void zgtrfs_(char *trans, int *n, int *nrhs, doublecomplex *dl, doublecomplex *d, doublecomplex *du, doublecomplex *dlf, doublecomplex *df, doublecomplex *duf, doublecomplex *du2, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int trans_len);
extern void zgtsv_(int *n, int *nrhs, doublecomplex *dl, doublecomplex *d, doublecomplex *du, doublecomplex *b, int *ldb, int *info);
extern void zgtsvx_(char *fact, char *trans, int *n, int *nrhs, doublecomplex *dl, doublecomplex *d, doublecomplex *du, doublecomplex *dlf, doublecomplex *df, doublecomplex *duf, doublecomplex *du2, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int fact_len, int trans_len);
extern void zgttrf_(int *n, doublecomplex *dl, doublecomplex *d, doublecomplex *du, doublecomplex *du2, int *ipiv, int *info);
extern void zgttrs_(char *trans, int *n, int *nrhs, doublecomplex *dl, doublecomplex *d, doublecomplex *du, doublecomplex *du2, int *ipiv, doublecomplex *b, int *ldb, int *info, int trans_len);
extern void zhbev_(char *jobz, char *uplo, int *n, int *kd, doublecomplex *ab, int *ldab, double *w, doublecomplex *z, int *ldz, doublecomplex *work, double *rwork, int *info, int jobz_len, int uplo_len);
extern void zhbevd_(char *jobz, char *uplo, int *n, int *kd, doublecomplex *ab, int *ldab, double *w, doublecomplex *z, int *ldz, doublecomplex *work, int *lwork, double *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void zhbevx_(char *jobz, char *range, char *uplo, int *n, int *kd, doublecomplex *ab, int *ldab, doublecomplex *q, int *ldq, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, doublecomplex *z, int *ldz, doublecomplex *work, double *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void zhbgst_(char *vect, char *uplo, int *n, int *ka, int *kb, doublecomplex *ab, int *ldab, doublecomplex *bb, int *ldbb, doublecomplex *x, int *ldx, doublecomplex *work, double *rwork, int *info, int vect_len, int uplo_len);
extern void zhbgv_(char *jobz, char *uplo, int *n, int *ka, int *kb, doublecomplex *ab, int *ldab, doublecomplex *bb, int *ldbb, double *w, doublecomplex *z, int *ldz, doublecomplex *work, double *rwork, int *info, int jobz_len, int uplo_len);
extern void zhbgvd_(char *jobz, char *uplo, int *n, int *ka, int *kb, doublecomplex *ab, int *ldab, doublecomplex *bb, int *ldbb, double *w, doublecomplex *z, int *ldz, doublecomplex *work, int *lwork, double *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void zhbgvx_(char *jobz, char *range, char *uplo, int *n, int *ka, int *kb, doublecomplex *ab, int *ldab, doublecomplex *bb, int *ldbb, doublecomplex *q, int *ldq, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, doublecomplex *z, int *ldz, doublecomplex *work, double *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void zhbtrd_(char *vect, char *uplo, int *n, int *kd, doublecomplex *ab, int *ldab, double *d, double *e, doublecomplex *q, int *ldq, doublecomplex *work, int *info, int vect_len, int uplo_len);
extern void zhecon_(char *uplo, int *n, doublecomplex *a, int *lda, int *ipiv, double *anorm, double *rcond, doublecomplex *work, int *info, int uplo_len);
extern void zheev_(char *jobz, char *uplo, int *n, doublecomplex *a, int *lda, double *w, doublecomplex *work, int *lwork, double *rwork, int *info, int jobz_len, int uplo_len);
extern void zheevd_(char *jobz, char *uplo, int *n, doublecomplex *a, int *lda, double *w, doublecomplex *work, int *lwork, double *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void zheevr_(char *jobz, char *range, char *uplo, int *n, doublecomplex *a, int *lda, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, doublecomplex *z, int *ldz, int *isuppz, doublecomplex *work, int *lwork, double *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len, int uplo_len);
extern void zheevx_(char *jobz, char *range, char *uplo, int *n, doublecomplex *a, int *lda, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, doublecomplex *z, int *ldz, doublecomplex *work, int *lwork, double *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void zhegst_(int *itype, char *uplo, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zhegv_(int *itype, char *jobz, char *uplo, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, double *w, doublecomplex *work, int *lwork, double *rwork, int *info, int jobz_len, int uplo_len);
extern void zhegvd_(int *itype, char *jobz, char *uplo, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, double *w, doublecomplex *work, int *lwork, double *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void zhegvx_(int *itype, char *jobz, char *range, char *uplo, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, doublecomplex *z, int *ldz, doublecomplex *work, int *lwork, double *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void zherfs_(char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *af, int *ldaf, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zhesv_(char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *work, int *lwork, int *info, int uplo_len);
extern void zhesvx_(char *fact, char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *af, int *ldaf, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, int *lwork, double *rwork, int *info, int fact_len, int uplo_len);
extern void zhetrd_(char *uplo, int *n, doublecomplex *a, int *lda, double *d, double *e, doublecomplex *tau, doublecomplex *work, int *lwork, int *info, int uplo_len);
extern void zhetrf_(char *uplo, int *n, doublecomplex *a, int *lda, int *ipiv, doublecomplex *work, int *lwork, int *info, int uplo_len);
extern void zhetri_(char *uplo, int *n, doublecomplex *a, int *lda, int *ipiv, doublecomplex *work, int *info, int uplo_len);
extern void zhetrs_(char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, int *ipiv, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zhgeqz_(char *job, char *compq, char *compz, int *n, int *ilo, int *ihi, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *q, int *ldq, doublecomplex *z, int *ldz, doublecomplex *work, int *lwork, double *rwork, int *info, int job_len, int compq_len, int compz_len);
extern void zhpcon_(char *uplo, int *n, doublecomplex *ap, int *ipiv, double *anorm, double *rcond, doublecomplex *work, int *info, int uplo_len);
extern void zhpev_(char *jobz, char *uplo, int *n, doublecomplex *ap, double *w, doublecomplex *z, int *ldz, doublecomplex *work, double *rwork, int *info, int jobz_len, int uplo_len);
extern void zhpevd_(char *jobz, char *uplo, int *n, doublecomplex *ap, double *w, doublecomplex *z, int *ldz, doublecomplex *work, int *lwork, double *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void zhpevx_(char *jobz, char *range, char *uplo, int *n, doublecomplex *ap, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, doublecomplex *z, int *ldz, doublecomplex *work, double *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void zhpgst_(int *itype, char *uplo, int *n, doublecomplex *ap, doublecomplex *bp, int *info, int uplo_len);
extern void zhpgv_(int *itype, char *jobz, char *uplo, int *n, doublecomplex *ap, doublecomplex *bp, double *w, doublecomplex *z, int *ldz, doublecomplex *work, double *rwork, int *info, int jobz_len, int uplo_len);
extern void zhpgvd_(int *itype, char *jobz, char *uplo, int *n, doublecomplex *ap, doublecomplex *bp, double *w, doublecomplex *z, int *ldz, doublecomplex *work, int *lwork, double *rwork, int *lrwork, int *iwork, int *liwork, int *info, int jobz_len, int uplo_len);
extern void zhpgvx_(int *itype, char *jobz, char *range, char *uplo, int *n, doublecomplex *ap, doublecomplex *bp, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, doublecomplex *z, int *ldz, doublecomplex *work, double *rwork, int *iwork, int *ifail, int *info, int jobz_len, int range_len, int uplo_len);
extern void zhprfs_(char *uplo, int *n, int *nrhs, doublecomplex *ap, doublecomplex *afp, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zhpsv_(char *uplo, int *n, int *nrhs, doublecomplex *ap, int *ipiv, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zhpsvx_(char *fact, char *uplo, int *n, int *nrhs, doublecomplex *ap, doublecomplex *afp, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int fact_len, int uplo_len);
extern void zhptrd_(char *uplo, int *n, doublecomplex *ap, double *d, double *e, doublecomplex *tau, int *info, int uplo_len);
extern void zhptrf_(char *uplo, int *n, doublecomplex *ap, int *ipiv, int *info, int uplo_len);
extern void zhptri_(char *uplo, int *n, doublecomplex *ap, int *ipiv, doublecomplex *work, int *info, int uplo_len);
extern void zhptrs_(char *uplo, int *n, int *nrhs, doublecomplex *ap, int *ipiv, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zhsein_(char *side, char *eigsrc, char *initv, int *select, int *n, doublecomplex *h, int *ldh, doublecomplex *w, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, int *mm, int *m, doublecomplex *work, double *rwork, int *ifaill, int *ifailr, int *info, int side_len, int eigsrc_len, int initv_len);
extern void zhseqr_(char *job, char *compz, int *n, int *ilo, int *ihi, doublecomplex *h, int *ldh, doublecomplex *w, doublecomplex *z, int *ldz, doublecomplex *work, int *lwork, int *info, int job_len, int compz_len);
extern void zpbcon_(char *uplo, int *n, int *kd, doublecomplex *ab, int *ldab, double *anorm, double *rcond, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zpbequ_(char *uplo, int *n, int *kd, doublecomplex *ab, int *ldab, double *s, double *scond, double *amax, int *info, int uplo_len);
extern void zpbrfs_(char *uplo, int *n, int *kd, int *nrhs, doublecomplex *ab, int *ldab, doublecomplex *afb, int *ldafb, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zpbstf_(char *uplo, int *n, int *kd, doublecomplex *ab, int *ldab, int *info, int uplo_len);
extern void zpbsv_(char *uplo, int *n, int *kd, int *nrhs, doublecomplex *ab, int *ldab, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zpbsvx_(char *fact, char *uplo, int *n, int *kd, int *nrhs, doublecomplex *ab, int *ldab, doublecomplex *afb, int *ldafb, char *equed, double *s, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void zpbtrf_(char *uplo, int *n, int *kd, doublecomplex *ab, int *ldab, int *info, int uplo_len);
extern void zpbtrs_(char *uplo, int *n, int *kd, int *nrhs, doublecomplex *ab, int *ldab, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zpocon_(char *uplo, int *n, doublecomplex *a, int *lda, double *anorm, double *rcond, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zpoequ_(int *n, doublecomplex *a, int *lda, double *s, double *scond, double *amax, int *info);
extern void zporfs_(char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *af, int *ldaf, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zposv_(char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zposvx_(char *fact, char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *af, int *ldaf, char *equed, double *s, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void zpotrf_(char *uplo, int *n, doublecomplex *a, int *lda, int *info, int uplo_len);
extern void zpotri_(char *uplo, int *n, doublecomplex *a, int *lda, int *info, int uplo_len);
extern void zpotrs_(char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zppcon_(char *uplo, int *n, doublecomplex *ap, double *anorm, double *rcond, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zppequ_(char *uplo, int *n, doublecomplex *ap, double *s, double *scond, double *amax, int *info, int uplo_len);
extern void zpprfs_(char *uplo, int *n, int *nrhs, doublecomplex *ap, doublecomplex *afp, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zppsv_(char *uplo, int *n, int *nrhs, doublecomplex *ap, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zppsvx_(char *fact, char *uplo, int *n, int *nrhs, doublecomplex *ap, doublecomplex *afp, char *equed, double *s, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int fact_len, int uplo_len, int equed_len);
extern void zpptrf_(char *uplo, int *n, doublecomplex *ap, int *info, int uplo_len);
extern void zpptri_(char *uplo, int *n, doublecomplex *ap, int *info, int uplo_len);
extern void zpptrs_(char *uplo, int *n, int *nrhs, doublecomplex *ap, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zptcon_(int *n, double *d, doublecomplex *e, double *anorm, double *rcond, double *rwork, int *info);
extern void zpteqr_(char *compz, int *n, double *d, double *e, doublecomplex *z, int *ldz, double *work, int *info, int compz_len);
extern void zptrfs_(char *uplo, int *n, int *nrhs, double *d, doublecomplex *e, double *df, doublecomplex *ef, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zptsv_(int *n, int *nrhs, double *d, doublecomplex *e, doublecomplex *b, int *ldb, int *info);
extern void zptsvx_(char *fact, int *n, int *nrhs, double *d, doublecomplex *e, double *df, doublecomplex *ef, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int fact_len);
extern void zpttrf_(int *n, double *d, doublecomplex *e, int *info);
extern void zpttrs_(char *uplo, int *n, int *nrhs, double *d, doublecomplex *e, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zspcon_(char *uplo, int *n, doublecomplex *ap, int *ipiv, double *anorm, double *rcond, doublecomplex *work, int *info, int uplo_len);
extern void zsprfs_(char *uplo, int *n, int *nrhs, doublecomplex *ap, doublecomplex *afp, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zspsv_(char *uplo, int *n, int *nrhs, doublecomplex *ap, int *ipiv, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zspsvx_(char *fact, char *uplo, int *n, int *nrhs, doublecomplex *ap, doublecomplex *afp, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int fact_len, int uplo_len);
extern void zsptrf_(char *uplo, int *n, doublecomplex *ap, int *ipiv, int *info, int uplo_len);
extern void zsptri_(char *uplo, int *n, doublecomplex *ap, int *ipiv, doublecomplex *work, int *info, int uplo_len);
extern void zsptrs_(char *uplo, int *n, int *nrhs, doublecomplex *ap, int *ipiv, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void zstedc_(char *compz, int *n, double *d, double *e, doublecomplex *z, int *ldz, doublecomplex *work, int *lwork, double *rwork, int *lrwork, int *iwork, int *liwork, int *info, int compz_len);
extern void zstegr_(char *jobz, char *range, int *n, double *d, double *e, double *vl, double *vu, int *il, int *iu, double *abstol, int *m, double *w, doublecomplex *z, int *ldz, int *isuppz, double *work, int *lwork, int *iwork, int *liwork, int *info, int jobz_len, int range_len);
extern void zstein_(int *n, double *d, double *e, int *m, double *w, int *iblock, int *isplit, doublecomplex *z, int *ldz, double *work, int *iwork, int *ifail, int *info);
extern void zsteqr_(char *compz, int *n, double *d, double *e, doublecomplex *z, int *ldz, double *work, int *info, int compz_len);
extern void zsycon_(char *uplo, int *n, doublecomplex *a, int *lda, int *ipiv, double *anorm, double *rcond, doublecomplex *work, int *info, int uplo_len);
extern void zsyrfs_(char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *af, int *ldaf, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len);
extern void zsysv_(char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *work, int *lwork, int *info, int uplo_len);
extern void zsysvx_(char *fact, char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *af, int *ldaf, int *ipiv, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *rcond, double *ferr, double *berr, doublecomplex *work, int *lwork, double *rwork, int *info, int fact_len, int uplo_len);
extern void zsytrf_(char *uplo, int *n, doublecomplex *a, int *lda, int *ipiv, doublecomplex *work, int *lwork, int *info, int uplo_len);
extern void zsytri_(char *uplo, int *n, doublecomplex *a, int *lda, int *ipiv, doublecomplex *work, int *info, int uplo_len);
extern void zsytrs_(char *uplo, int *n, int *nrhs, doublecomplex *a, int *lda, int *ipiv, doublecomplex *b, int *ldb, int *info, int uplo_len);
extern void ztbcon_(char *norm, char *uplo, char *diag, int *n, int *kd, doublecomplex *ab, int *ldab, double *rcond, doublecomplex *work, double *rwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void ztbrfs_(char *uplo, char *trans, char *diag, int *n, int *kd, int *nrhs, doublecomplex *ab, int *ldab, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void ztbtrs_(char *uplo, char *trans, char *diag, int *n, int *kd, int *nrhs, doublecomplex *ab, int *ldab, doublecomplex *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void ztgevc_(char *side, char *howmny, int *select, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, int *mm, int *m, doublecomplex *work, double *rwork, int *info, int side_len, int howmny_len);
extern void ztgexc_(int *wantq, int *wantz, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *q, int *ldq, doublecomplex *z, int *ldz, int *ifst, int *ilst, int *info);
extern void ztgsen_(int *ijob, int *wantq, int *wantz, int *select, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *alpha, doublecomplex *beta, doublecomplex *q, int *ldq, doublecomplex *z, int *ldz, int *m, double *pl, double *pr, double *dif, doublecomplex *work, int *lwork, int *iwork, int *liwork, int *info);
extern void ztgsja_(char *jobu, char *jobv, char *jobq, int *m, int *p, int *n, int *k, int *l, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, double *tola, double *tolb, double *alpha, double *beta, doublecomplex *u, int *ldu, doublecomplex *v, int *ldv, doublecomplex *q, int *ldq, doublecomplex *work, int *ncycle, int *info, int jobu_len, int jobv_len, int jobq_len);
extern void ztgsna_(char *job, char *howmny, int *select, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, double *s, double *dif, int *mm, int *m, doublecomplex *work, int *lwork, int *iwork, int *info, int job_len, int howmny_len);
extern void ztgsyl_(char *trans, int *ijob, int *m, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *c, int *ldc, doublecomplex *d, int *ldd, doublecomplex *e, int *lde, doublecomplex *f, int *ldf, double *scale, double *dif, doublecomplex *work, int *lwork, int *iwork, int *info, int trans_len);
extern void ztpcon_(char *norm, char *uplo, char *diag, int *n, doublecomplex *ap, double *rcond, doublecomplex *work, double *rwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void ztprfs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, doublecomplex *ap, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void ztptri_(char *uplo, char *diag, int *n, doublecomplex *ap, int *info, int uplo_len, int diag_len);
extern void ztptrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, doublecomplex *ap, doublecomplex *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void ztrcon_(char *norm, char *uplo, char *diag, int *n, doublecomplex *a, int *lda, double *rcond, doublecomplex *work, double *rwork, int *info, int norm_len, int uplo_len, int diag_len);
extern void ztrevc_(char *side, char *howmny, int *select, int *n, doublecomplex *t, int *ldt, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, int *mm, int *m, doublecomplex *work, double *rwork, int *info, int side_len, int howmny_len);
extern void ztrexc_(char *compq, int *n, doublecomplex *t, int *ldt, doublecomplex *q, int *ldq, int *ifst, int *ilst, int *info, int compq_len);
extern void ztrrfs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *x, int *ldx, double *ferr, double *berr, doublecomplex *work, double *rwork, int *info, int uplo_len, int trans_len, int diag_len);
extern void ztrsen_(char *job, char *compq, int *select, int *n, doublecomplex *t, int *ldt, doublecomplex *q, int *ldq, doublecomplex *w, int *m, double *s, double *sep, doublecomplex *work, int *lwork, int *info, int job_len, int compq_len);
extern void ztrsna_(char *job, char *howmny, int *select, int *n, doublecomplex *t, int *ldt, doublecomplex *vl, int *ldvl, doublecomplex *vr, int *ldvr, double *s, double *sep, int *mm, int *m, doublecomplex *work, int *ldwork, double *rwork, int *info, int job_len, int howmny_len);
extern void ztrsyl_(char *trana, char *tranb, int *isgn, int *m, int *n, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, doublecomplex *c, int *ldc, double *scale, int *info, int trana_len, int tranb_len);
extern void ztrtri_(char *uplo, char *diag, int *n, doublecomplex *a, int *lda, int *info, int uplo_len, int diag_len);
extern void ztrtrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, doublecomplex *a, int *lda, doublecomplex *b, int *ldb, int *info, int uplo_len, int trans_len, int diag_len);
extern void ztzrqf_(int *m, int *n, doublecomplex *a, int *lda, doublecomplex *tau, int *info);
extern void ztzrzf_(int *m, int *n, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zungbr_(char *vect, int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info, int vect_len);
extern void zunghr_(int *n, int *ilo, int *ihi, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zunglq_(int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zungql_(int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zungqr_(int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zungrq_(int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info);
extern void zungtr_(char *uplo, int *n, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *work, int *lwork, int *info, int uplo_len);
extern void zunmbr_(char *vect, char *side, char *trans, int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *lwork, int *info, int vect_len, int side_len, int trans_len);
extern void zunmhr_(char *side, char *trans, int *m, int *n, int *ilo, int *ihi, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *lwork, int *info, int side_len, int trans_len);
extern void zunmlq_(char *side, char *trans, int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *lwork, int *info, int side_len, int trans_len);
extern void zunmql_(char *side, char *trans, int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *lwork, int *info, int side_len, int trans_len);
extern void zunmqr_(char *side, char *trans, int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *lwork, int *info, int side_len, int trans_len);
extern void zunmr3_(char *side, char *trans, int *m, int *n, int *k, int *l, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *info, int side_len, int trans_len);
extern void zunmrq_(char *side, char *trans, int *m, int *n, int *k, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *lwork, int *info, int side_len, int trans_len);
extern void zunmrz_(char *side, char *trans, int *m, int *n, int *k, int *l, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *lwork, int *info, int side_len, int trans_len);
extern void zunmtr_(char *side, char *uplo, char *trans, int *m, int *n, doublecomplex *a, int *lda, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *lwork, int *info, int side_len, int uplo_len, int trans_len);
extern void zupgtr_(char *uplo, int *n, doublecomplex *ap, doublecomplex *tau, doublecomplex *q, int *ldq, doublecomplex *work, int *info, int uplo_len);
extern void zupmtr_(char *side, char *uplo, char *trans, int *m, int *n, doublecomplex *ap, doublecomplex *tau, doublecomplex *c, int *ldc, doublecomplex *work, int *info, int side_len, int uplo_len, int trans_len);

/* FFT routines */
extern void cfft1d_(int *mode, int *n, complex *x, complex *comm, int *info);
extern void cfft1dx_(int *mode, float *scale, int *inpl, int *n, complex *x, int *incx, complex *y, int *incy, complex *comm, int *info);
extern void cfft1m_(int *mode, int *nseq, int *n, complex *x, complex *comm, int *info);
extern void cfft1mx_(int *mode, float *scale, int *inpl, int *nseq, int *n, complex *x, int *incx1, int *incx2, complex *y, int *incy1, int *incy2, complex *comm, int *info);
extern void cfft2d_(int *mode, int *m, int *n, complex *x, complex *comm, int *info);
extern void cfft2dx_(int *mode, float *scale, int *ltrans, int *inpl, int *m, int *n, complex *x, int *incx1, int *incx2, complex *y, int *incy1, int *incy2, complex *comm, int *info);
extern void cfft3d_(int *mode, int *l, int *m, int *n, complex *x, complex *comm, int *info);
extern void cfft3dx_(int *mode, float *scale, int *ltrans, int *inpl, int *l, int *m, int *n, complex *x, complex *y, complex *comm, int *info);
extern void cfft3dy_(int *mode, float *scale, int *inpl, int *l, int *m, int *n, complex *x, int *incx1, int *incx2, int *incx3, complex *y, int *incy1, int *incy2, int *incy3, complex *comm, int *lcomm, int *info);
extern void csfft_(int *mode, int *n, float *x, float *comm, int *info);
extern void csfftm_(int *nseq, int *n, float *x, float *comm, int *info);
extern void dzfft_(int *mode, int *n, double *x, double *comm, int *info);
extern void dzfftm_(int *nseq, int *n, double *x, double *comm, int *info);
extern void scfft_(int *mode, int *n, float *x, float *comm, int *info);
extern void scfftm_(int *nseq, int *n, float *x, float *comm, int *info);
extern void zdfft_(int *mode, int *n, double *x, double *comm, int *info);
extern void zdfftm_(int *nseq, int *n, double *x, double *comm, int *info);
extern void zfft1d_(int *mode, int *n, doublecomplex *x, doublecomplex *comm, int *info);
extern void zfft1dx_(int *mode, double *scale, int *inpl, int *n, doublecomplex *x, int *incx, doublecomplex *y, int *incy, doublecomplex *comm, int *info);
extern void zfft1m_(int *mode, int *nseq, int *n, doublecomplex *x, doublecomplex *comm, int *info);
extern void zfft1mx_(int *mode, double *scale, int *inpl, int *nseq, int *n, doublecomplex *x, int *incx1, int *incx2, doublecomplex *y, int *incy1, int *incy2, doublecomplex *comm, int *info);
extern void zfft2d_(int *mode, int *m, int *n, doublecomplex *x, doublecomplex *comm, int *info);
extern void zfft2dx_(int *mode, double *scale, int *ltrans, int *inpl, int *m, int *n, doublecomplex *x, int *incx1, int *incx2, doublecomplex *y, int *incy1, int *incy2, doublecomplex *comm, int *info);
extern void zfft3d_(int *mode, int *l, int *m, int *n, doublecomplex *x, doublecomplex *comm, int *info);
extern void zfft3dx_(int *mode, double *scale, int *ltrans, int *inpl, int *l, int *m, int *n, doublecomplex *x, doublecomplex *y, doublecomplex *comm, int *info);
extern void zfft3dy_(int *mode, double *scale, int *inpl, int *l, int *m, int *n, doublecomplex *x, int *incx1, int *incx2, int *incx3, doublecomplex *y, int *incy1, int *incy2, int *incy3, doublecomplex *comm, int *lcomm, int *info);

/* Sparse BLAS routines */
extern void caxpyi_(int *nz, complex *a, complex *x, int *indx, complex *y);
extern complex cdotci_(complex *ret_val, int *nz, complex *x, int *indx, complex *y);
extern void cdotcisub_(complex *ret_val, int *nz, complex *x, int *indx, complex *y);
extern complex cdotui_(complex *ret_val, int *nz, complex *x, int *indx, complex *y);
extern void cdotuisub_(complex *ret_val, int *nz, complex *x, int *indx, complex *y);
extern void cgthr_(int *nz, complex *y, complex *x, int *indx);
extern void cgthrz_(int *nz, complex *y, complex *x, int *indx);
extern void csctr_(int *nz, complex *x, int *indx, complex *y);
extern void daxpyi_(int *nz, double *a, double *x, int *indx, double *y);
extern double ddoti_(int *nz, double *x, int *indx, double *y);
extern void dgthr_(int *nz, double *y, double *x, int *indx);
extern void dgthrz_(int *nz, double *y, double *x, int *indx);
extern void droti_(int *nz, double *x, int *indx, double *y, double *c, double *s);
extern void dsctr_(int *nz, double *x, int *indx, double *y);
extern void saxpyi_(int *nz, float *a, float *x, int *indx, float *y);

#ifdef FORTRAN_FLOAT_FUNCTIONS_RETURN_DOUBLE
extern double sdoti_(int *nz, float *x, int *indx, float *y);
#else
extern float sdoti_(int *nz, float *x, int *indx, float *y);
#endif

extern void sgthr_(int *nz, float *y, float *x, int *indx);
extern void sgthrz_(int *nz, float *y, float *x, int *indx);
extern void sroti_(int *nz, float *x, int *indx, float *y, float *c, float *s);
extern void ssctr_(int *nz, float *x, int *indx, float *y);
extern void zaxpyi_(int *nz, doublecomplex *a, doublecomplex *x, int *indx, doublecomplex *y);
extern doublecomplex zdotci_(doublecomplex *ret_val, int *nz, doublecomplex *x, int *indx, doublecomplex *y);
extern void zdotcisub_(doublecomplex *ret_val, int *nz, doublecomplex *x, int *indx, doublecomplex *y);
extern doublecomplex zdotui_(doublecomplex *ret_val, int *nz, doublecomplex *x, int *indx, doublecomplex *y);
extern void zdotuisub_(doublecomplex *ret_val, int *nz, doublecomplex *x, int *indx, doublecomplex *y);
extern void zgthr_(int *nz, doublecomplex *y, doublecomplex *x, int *indx);
extern void zgthrz_(int *nz, doublecomplex *y, doublecomplex *x, int *indx);
extern void zsctr_(int *nz, doublecomplex *x, int *indx, doublecomplex *y);

/* Random number generators */
extern void drandbeta_(int *n, double *a, double *b, int *state, double *x, int *info);
extern void drandbinomial_(int *n, int *m, double *p, int *state, int *x, int *info);
extern void drandbinomialreference_(int *m, double *p, double *ref, int *lref, int *info);
extern void drandblumblumshub_(int *n, int *state, double *x, int *info);
extern void drandcauchy_(int *n, double *a, double *b, int *state, double *x, int *info);
extern void drandchisquared_(int *n, int *df, int *state, double *x, int *info);
extern void dranddiscreteuniform_(int *n, int *a, int *b, int *state, int *x, int *info);
extern void drandexponential_(int *n, double *a, int *state, double *x, int *info);
extern void drandf_(int *n, int *df1, int *df2, int *state, double *x, int *info);
extern void drandgamma_(int *n, double *a, double *b, int *state, double *x, int *info);
extern void drandgaussian_(int *n, double *xmu, double *var, int *state, double *x, int *info);
extern void drandgeneraldiscrete_(int *n, double *ref, int *state, int *x, int *info);
extern void drandgeometric_(int *n, double *p, int *state, int *x, int *info);
extern void drandgeometricreference_(double *p, double *ref, int *lref, int *info);
extern void drandhypergeometric_(int *n, int *np, int *ns, int *m, int *state, int *x, int *info);
extern void drandhypergeometricreference_(int *np, int *ns, int *m, double *ref, int *lref, int *info);
extern void drandinitialize_(int *genid, int *subid, int *seed, int *lseed, int *state, int *lstate, int *info);
extern void drandinitializebbs_(int *nbits, int *lp, int *p, int *lq, int *q, int *ls, int *s, int *state, int *lstate, int *info);
extern void drandinitializeuser_(ACML_DRANDINITIALIZEUSER_UINI uini, ACML_DRANDINITIALIZEUSER_UGEN ugen, int genid, int subid, int *seed, int *lseed, int *state, int *lstate, int *info);
extern void drandleapfrog_(int *n, int *k, int *state, int *info);
extern void drandlogistic_(int *n, double *a, double *b, int *state, double *x, int *info);
extern void drandlognormal_(int *n, double *xmu, double *var, int *state, double *x, int *info);
extern void drandmultinomial_(int *n, int *m, double *p, int *k, int *state, int *x, int *ldx, int *info);
extern void drandmultinormalreference_(int *m, double *xmu, double *c, int *ldc, double *ref, int *lref, int *info);
extern void drandmultinormalr_(int *n, double *ref, int *state, double *x, int *ldx, int *info);
extern void drandmultistudentstreference_(int *m, int *df, double *xmu, double *c, int *ldc, double *ref, int *lref, int *info);
extern void drandmultistudentstr_(int *n, double *ref, int *state, double *x, int *ldx, int *info);
extern void drandnegativebinomial_(int *n, int *m, double *p, int *state, int *x, int *info);
extern void drandnegativebinomialreference_(int *m, double *p, double *ref, int *lref, int *info);
extern void drandpoisson_(int *n, double *lambda, int *state, int *x, int *info);
extern void drandpoissonreference_(double *lambda, double *ref, int *lref, int *info);
extern void drandskipahead_(int *n, int *state, int *info);
extern void drandstudentst_(int *n, int *df, int *state, double *x, int *info);
extern void drandtriangular_(int *n, double *xmin, double *xmed, double *xmax, int *state, double *x, int *info);
extern void dranduniform_(int *n, double *a, double *b, int *state, double *x, int *info);
extern void drandvonmises_(int *n, double *vk, int *state, double *x, int *info);
extern void drandweibull_(int *n, double *a, double *b, int *state, double *x, int *info);
extern void drandmultinormal_(int *n, int *m, double *xmu, double *c, int *ldc, int *state, double *x, int *ldx, int *info);
extern void drandmultistudentst_(int *n, int *m, int *df, double *xmu, double *c, int *ldc, int *state, double *x, int *ldx, int *info);
extern void srandbeta_(int *n, float *a, float *b, int *state, float *x, int *info);
extern void srandblumblumshub_(int *n, int *state, float *x, int *info);
extern void srandbinomial_(int *n, int *m, float *p, int *state, int *x, int *info);
extern void srandbinomialreference_(int *m, float *p, float *ref, int *lref, int *info);
extern void srandcauchy_(int *n, float *a, float *b, int *state, float *x, int *info);
extern void srandchisquared_(int *n, int *df, int *state, float *x, int *info);
extern void sranddiscreteuniform_(int *n, int *a, int *b, int *state, int *x, int *info);
extern void srandexponential_(int *n, float *a, int *state, float *x, int *info);
extern void srandf_(int *n, int *df1, int *df2, int *state, float *x, int *info);
extern void srandgamma_(int *n, float *a, float *b, int *state, float *x, int *info);
extern void srandgaussian_(int *n, float *xmu, float *var, int *state, float *x, int *info);
extern void srandgeneraldiscrete_(int *n, float *ref, int *state, int *x, int *info);
extern void srandgeometric_(int *n, float *p, int *state, int *x, int *info);
extern void srandgeometricreference_(float *p, float *ref, int *lref, int *info);
extern void srandhypergeometric_(int *n, int *np, int *ns, int *m, int *state, int *x, int *info);
extern void srandhypergeometricreference_(int *np, int *ns, int *m, float *ref, int *lref, int *info);
extern void srandinitialize_(int *genid, int *subid, int *seed, int *lseed, int *state, int *lstate, int *info);
extern void srandinitializebbs_(int *nbits, int *lp, int *p, int *lq, int *q, int *ls, int *s, int *state, int *lstate, int *info);
extern void srandinitializeuser_(ACML_SRANDINITIALIZEUSER_UINI uini, ACML_SRANDINITIALIZEUSER_UGEN ugen, int genid, int subid, int *seed, int *lseed, int *state, int *lstate, int *info);
extern void srandleapfrog_(int *n, int *k, int *state, int *info);
extern void srandlogistic_(int *n, float *a, float *b, int *state, float *x, int *info);
extern void srandlognormal_(int *n, float *xmu, float *var, int *state, float *x, int *info);
extern void srandmultinomial_(int *n, int *m, float *p, int *k, int *state, int *x, int *ldx, int *info);
extern void srandmultinormalreference_(int *m, float *xmu, float *c, int *ldc, float *ref, int *lref, int *info);
extern void srandmultinormalr_(int *n, float *ref, int *state, float *x, int *ldx, int *info);
extern void srandmultistudentstreference_(int *m, int *df, float *xmu, float *c, int *ldc, float *ref, int *lref, int *info);
extern void srandmultistudentstr_(int *n, float *ref, int *state, float *x, int *ldx, int *info);
extern void srandnegativebinomial_(int *n, int *m, float *p, int *state, int *x, int *info);
extern void srandnegativebinomialreference_(int *m, float *p, float *ref, int *lref, int *info);
extern void srandpoisson_(int *n, float *lambda, int *state, int *x, int *info);
extern void srandpoissonreference_(float *lambda, float *ref, int *lref, int *info);
extern void srandskipahead_(int *n, int *state, int *info);
extern void srandstudentst_(int *n, int *df, int *state, float *x, int *info);
extern void srandtriangular_(int *n, float *xmin, float *xmed, float *xmax, int *state, float *x, int *info);
extern void sranduniform_(int *n, float *a, float *b, int *state, float *x, int *info);
extern void srandvonmises_(int *n, float *vk, int *state, float *x, int *info);
extern void srandweibull_(int *n, float *a, float *b, int *state, float *x, int *info);
extern void srandmultinormal_(int *n, int *m, float *xmu, float *c, int *ldc, int *state, float *x, int *ldx, int *info);
extern void srandmultistudentst_(int *n, int *m, int *df, float *xmu, float *c, int *ldc, int *state, float *x, int *ldx, int *info);

/* ACML version information */
extern void acmlversion_(int *major, int *minor, int *patch);
extern void acmlinfo_(void);

/* OMP interfaces */
extern int acmlgetmaxthreads_(void);
extern int acmlgetnumthreads_(void);
extern void acmlsetnumthreads_(int *numthreads);
extern int acmlgetnumprocs_(void);

/* Putenv and getenv utilities */
extern void acmlputenv_(char *name, char *value, int name_len, int value_len);
extern void acmlgetenv_(char *name, char *value, int name_len, int value_len);

/* ------------------------------------------------------------------ */

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* !defined(_ACML_H) */
