/*
  \file
  \brief Models for synchronous electric drive using IT++ and BDM
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#include <itpp/itbase.h>
#include <estim/libKF.h>
#include <estim/libPF.h>
#include <stat/libFN.h>

#include "pmsm.h"
#include "simulator.h"

#include "iopom.h"

using namespace itpp;
//!Extended Kalman filter with unknown \c Q
class EKF_unQ : public EKFCh , public BMcond {
public:
	//! Default constructor
	EKF_unQ ( RV rx, RV ry,RV ru,RV rQ ) :EKFCh ( rx,ry,ru ),BMcond ( rQ ) {};
	void condition ( const vec &Q0 ) {
		Q.setD ( Q0,0 );
		//from EKF
		preA.set_submatrix ( dimy+dimx,dimy,Q._Ch() );
	};
};

void set_simulator_t(double &Ww) {

        if (t>0.2) x[8]=1.2;    // 1A //0.2ZP
        if (t>0.4) x[8]=10.8;   // 9A
        if (t>0.6) x[8]=25.2;  // 21A

        if (t>0.7) Ww=2.*M_PI*10.;
        if (t>1.0) x[8]=1.2;    // 1A
        if (t>1.2) x[8]=10.8;   // 9A
        if (t>1.4) x[8]=25.2;  // 21A

        if (t>1.6) Ww=2.*M_PI*50.;
        if (t>1.9) x[8]=1.2;    // 1A
        if (t>2.1) x[8]=10.8;   // 9A
        if (t>2.3) x[8]=25.2;  // 21A

        if (t>2.5) Ww=2.*M_PI*100;
        if (t>2.8) x[8]=1.2;    // 1A
        if (t>3.0) x[8]=10.8;   // 9A
        if (t>3.2) x[8]=25.2;  // 21A

        if (t>3.4) Ww=2.*M_PI*150;
        if (t>3.7) x[8]=1.2;    // 1A
        if (t>3.9) x[8]=10.8;   // 9A
        if (t>4.1) x[8]=25.2;  // 21A

        if (t>4.3) Ww=2.*M_PI*0;
        if (t>4.8) x[8]=-1.2;    // 1A
        if (t>5.0) x[8]=-10.8;   // 9A
        if (t>5.2) x[8]=-25.2;  // 21A

        if (t>5.4) Ww=2.*M_PI*(-10.);
        if (t>5.7) x[8]=-1.2;    // 1A
        if (t>5.9) x[8]=-10.8;   // 9A
        if (t>6.1) x[8]=-25.2;  // 21A

        if (t>6.3) Ww=2.*M_PI*(-50.);
        if (t>6.7) x[8]=-1.2;    // 1A
        if (t>6.9) x[8]=-10.8;   // 9A
        if (t>7.1) x[8]=-25.2;  // 21A

        if (t>7.3) Ww=2.*M_PI*(-100.);
        if (t>7.7) x[8]=-1.2;    // 1A
        if (t>7.9) x[8]=-10.8;   // 9A
        if (t>8.1) x[8]=-25.2;  // 21A
        if (t>8.3) x[8]=10.8;   // 9A
        if (t>8.5) x[8]=25.2;  // 21A
        
//        x[8]=0.0;
}

int main() {
	// Kalman filter
	int Ndat = 90000;
	double h = 1e-6;
	int Nsimstep = 125;
	int Npart = 100;
	
	//StrSim:06:
	vec SSAT(Ndat);

	// internal model
	IMpmsm fxu;
	//                  Rs    Ls        dt           Fmag(Ypm) kp   p    J     Bf(Mz)
	fxu.set_parameters ( 0.28, 0.003465, Nsimstep*h, 0.1989,   1.5 ,4.0, 0.04, 0.0 );
	// observation model
	OMpmsm hxu;

	vec mu0= "0.0 0.0 0.0 0.0";
//	vec Qdiag ( "0.01 0.01 0.01 0.0001" ); //zdenek: 0.01 0.01 0.0001 0.0001
	vec Qdiag ( "18 18 157.9 0.0001" ); //zdenek: 0.01 0.01 0.0001 0.0001
	vec Rdiag ( "90 90" ); //var(diff(xth)) = "0.034 0.034"
	chmat Q ( Qdiag );
	chmat R ( Rdiag );
	EKFCh KFE ( rx,ry,ru );
	KFE.set_est ( mu0, ( 1*eye ( 4 ) ) );
	KFE.set_parameters ( &fxu,&hxu,Qdiag,Rdiag);

	RV rQ ( "100","{Q}","4","0" );
	EKF_unQ KFEp ( rx,ry,ru,rQ );
	KFEp.set_est ( mu0, chmat ( 1*ones ( 4 ) ) );
	KFEp.set_parameters ( &fxu,&hxu,Q,R );

	mgamma_fix evolQ ( rQ,rQ );
	MPF<EKF_unQ> M ( rx,rQ,evolQ,evolQ,Npart,KFEp );
	// initialize
	evolQ.set_parameters ( 1000.0 ,Qdiag, 0.5); //sigma = 1/10 mu
	evolQ.condition ( Qdiag ); //Zdenek default
	epdf& pfinit=evolQ._epdf();
	M.set_est ( pfinit );
	evolQ.set_parameters ( 5000.0, Qdiag, 0.9999 );

	//

	epdf& KFEep = KFE._epdf();
	epdf& Mep = M._epdf();

	mat Xt=zeros ( Ndat ,9 ); //true state from simulator
	mat Dt=zeros ( Ndat,4+2 ); //observation
	mat XtE=zeros ( Ndat, 4 );
	mat XtM=zeros ( Ndat,4+4 ); //Q + x

	// SET SIMULATOR
	pmsmsim_set_parameters ( 0.28,0.003465,0.1989,0.0,4,1.5,0.04, 200., 3e-6, h );
	double Ww=0.0;
	vec dt ( 2 );
	vec ut ( 2 );

	for ( int tK=1;tK<Ndat;tK++ ) {
		//Number of steps of a simulator for one step of Kalman
		for ( int ii=0; ii<Nsimstep;ii++ ) {
			//simulator
			set_simulator_t(Ww);
			pmsmsim_step ( Ww );
		};
		// collect data
		ut ( 0 ) = KalmanObs[0];
		ut ( 1 ) = KalmanObs[1];
		dt ( 0 ) = KalmanObs[2];
		dt ( 1 ) = KalmanObs[3];

		//estimator
		KFE.bayes ( concat ( dt,ut ) );
		M.bayes ( concat ( dt,ut ) );
		SSAT(tK) = M.SSAT;
		
		Xt.set_row ( tK,vec ( x,9 ) ); //vec from C-array
		Dt.set_row ( tK, concat ( dt,ut,vec_1(sqrt(pow(ut(0),2)+pow(ut(1),2))), vec_1(sqrt(pow(dt(0),2)+pow(dt(1),2))) ) );
		XtE.set_row ( tK,KFEep.mean() );
		XtM.set_row ( tK,Mep.mean() );
	}

	it_file fou ( "pmsm_sim.it" );

	fou << Name ( "xth" ) << Xt;
	fou << Name ( "Dt" ) << Dt;
	fou << Name ( "xthE" ) << XtE;
	fou << Name ( "xthM" ) << XtM;
	fou << Name ( "SSAT" ) << SSAT;
	//Exit program:

	char tmpstr[200];
	sprintf(tmpstr,"%s/%s","here/","format");
	ofstream  form(tmpstr);
	form << "# Experimetal header file"<< endl;
	dirfile_write(form,"here/",Xt,"X","{isa isb om th }" );
	dirfile_write ( form,"here",XtM,"XtM","{q1 q2 q3 q4 isa isb om th }" );
	dirfile_write ( form,"here",XtE,"XE","{isa isb om th }" );
	dirfile_write ( form,"here",Dt,"Dt","{isa isb ua ub }" );

	////////////////
	// Just Testing
/*	NcFile nc ( "pmsm_sim2.nc", NcFile::Replace ); // Create and leave in define mode
	if ( ! nc.is_valid() ) {	std::cerr << "creation of NCFile failed."<<endl;}

	write_to_nc ( nc,Xt,"X","{isa isb om th }" );
	write_to_nc ( nc,XtM,"XtM","{q1 q2 q3 q4 isa isb om th }" );
	write_to_nc ( nc,XtE,"XE","{isa isb om th }" );
	write_to_nc ( nc,Dt,"Dt","{isa isb ua ub }" );*/
	return 0;
}

