/*!
 * \file
 * \brief Matrices in decomposed forms (LDL', LU, UDU', etc).
 * \author Vaclav Smidl.
 *
 * -----------------------------------
 * BDM++ - C++ library for Bayesian Decision Making under Uncertainty
 *
 * Using IT++ for numerical operations
 * -----------------------------------
 */

#ifndef CHMAT_H
#define CHMAT_H

#include "../bdmerror.h"
#include "square_mat.h"

namespace bdm
{

/*! \brief Symmetric matrix stored in square root decomposition using upper cholesky

This matrix represent \f$A=Ch' Ch\f$ where only the upper triangle \f$Ch\f$ is stored;

*/
class chmat : public sqmat {
protected:
//! Upper triangle of the cholesky matrix
	mat Ch;
public:

	void opupdt ( const vec &v, double w );
	mat to_mat() const;
	void mult_sym ( const mat &C );
	//! mult_sym with return value in parameter \c U
	void mult_sym ( const mat &C , chmat &U ) const;
	void mult_sym_t ( const mat &C );
	//! mult_sym with return value in parameter \c U
	void mult_sym_t ( const mat &C, chmat &U ) const;
	double logdet() const;
	vec sqrt_mult ( const vec &v ) const;
	double qform ( const vec &v ) const;
	double invqform ( const vec &v ) const;
	void clear();
	//! add another chmat \c A2 with weight \c w.
	void add ( const chmat &A2, double w = 1.0 ) {
		bdm_assert_debug ( dim == A2.dim, "Matrices of unequal dimension" );
		mat pre = concat_vertical ( Ch, sqrt ( w ) * A2.Ch );
		mat post = zeros ( pre.rows(), pre.cols() );
		if ( !qr ( pre, post ) ) {
			bdm_warning ( "Unstable QR in chmat add" );
		}
		Ch = post ( 0, dim - 1, 0, dim - 1 );
	};
	//!Inversion in the same form, i.e. cholesky
	void inv ( chmat &Inv ) const	{
		( Inv.Ch ) = itpp::inv ( Ch ).T();
	}; //Fixme: can be more efficient
	;
//	void inv ( mat &Inv );

	//! Destructor for future use;
	virtual ~chmat() {};
	//!
	chmat ( ) : sqmat (), Ch ( ) {};
	//! Default constructor
	chmat ( const int dim0 ) : sqmat ( dim0 ), Ch ( dim0, dim0 ) {};
	//! Default constructor
	chmat ( const vec &v ) : sqmat ( v.length() ), Ch ( diag ( sqrt ( v ) ) ) {};
	//! Copy constructor
	chmat ( const chmat &Ch0 ) : sqmat ( Ch0.dim ), Ch ( Ch0.dim, Ch0.dim ) {
		Ch = Ch0.Ch;
	}

	//! Default constructor (m3k:cholform)
	chmat ( const mat &M ) : sqmat ( M.rows() ), Ch ( M.rows(), M.cols() ) {
		mat Q;
		bdm_assert_debug ( M.rows() == M.cols(), "chmat:: input matrix must be square!" );
		Ch = chol ( M );
	}

	/*!
	  Some templates require this constructor to compile, but
	  it shouldn't actually be called.
	*/
	chmat ( const chmat &M, const ivec &perm ) {
		bdm_error ( "not implemented" );
	}

	//! Access function
	mat & _Ch() {
		return Ch;
	}
	//! Access function
	const mat & _Ch() const {
		return Ch;
	}
	//! Access functions
	void setD ( const vec &nD ) {
		Ch = diag ( sqrt ( nD ) );
	}
	//! Access functions
	void setCh ( const vec &chQ ) {
		bdm_assert_debug ( chQ.length() == dim * dim, "wrong length" );
		copy_vector ( dim*dim, chQ._data(), Ch._data() );
	}

	//! Access functions
	void setD ( const vec &nD, int i ) {
		for ( int j = i; j < nD.length(); j++ ) {
			Ch ( j, j ) = sqrt ( nD ( j - i ) );    //Fixme can be more general
		}
	}

	//! Operator 
	chmat& operator += ( const chmat &A2 );
	//! Operator 
	chmat& operator -= ( const chmat &A2 );
	//! Operator 
	chmat& operator * ( const double &d ) {
		Ch*sqrt ( d );
		return *this;
	};
	//! Operator 
	chmat& operator = ( const chmat &A2 ) {
		Ch = A2.Ch;
		dim = A2.dim;
		return *this;
	}
	//! Operator 
	chmat& operator *= ( double x ) {
		Ch *= sqrt ( x );
		return *this;
	};
};


//////// Operations:
//!mapping of add operation to operators
inline chmat& chmat::operator += ( const chmat & A2 )  {
	this->add ( A2 );
	return *this;
}
//!mapping of negative add operation to operators
inline chmat& chmat::operator -= ( const chmat & A2 )  {
	this->add ( A2, -1.0 );
	return *this;
}

}

#endif // CHMAT_H
