//
// C++ Interface: userinfo
//
// Description:
//
//
// Author: smidl <smidl@utia.cas.cz>, (C) 2008
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include <itpp/itbase.h>
#include <itpp/base/mat.h>

using std::cout;
using std::endl;

/*!
@brief User Info base class

This class is used to store information about parameters of an object. It support loading and saving of the information and, potentially, interaction with the user.
*/
class uibase {
protected:
	std::string comment;
	std::string help;
	uibase* parent;

public:
//!Default constructor
	uibase ( std::string com = "Abstract class, please ignore!") :comment ( com ) {parent=NULL;}
//!Default constructor
	uibase ( std::string com,uibase *par ) :comment ( com ),parent ( par ) {}

//! returns a summary of its contents (used in dialogs)
	virtual void getsummary ( std::string &S ) {};

//! interaction with the user
	virtual void askuser(){};

//! test if the info is valid
	virtual bool isvalid() {return true;}
	//! for future use
	virtual ~uibase(){};
};

//! User info for scalars
template<class T>
class uiscalar : public uibase {
protected:
	T N;
public:
//!Default constructor
	uiscalar ( std::string com,uibase* par ) :uibase ( com,par ) {N=T ( 0 );};
	uiscalar ( std::string com) :uibase ( com ) {N=T ( 0 );};

	void getsummary ( std::string &S ) {S="Scalar";};

	void askuser (){};

//! saving the info
template<class T2>
	friend std::ostream &operator<< ( std::ostream &os, const uiscalar<T2> &ui );

//! saving the info
template<class T2>
	friend std::istream &operator>> ( std::istream &is, const uiscalar<T2> &ui );
	~uiscalar(){};
};

//! User info for strings
class uistring : public uibase {
protected:
	std::string S;
public:
	void getsummary ( std::string &S ) {S="String";};

//!Default constructor
	uistring ( std::string com ) :uibase ( com ) {}
	
	void askuser (){};
//! saving the info
	friend std::ostream &operator<< ( std::ostream &os, const uistring &ui );

//! saving the info
	friend std::istream &operator>> ( std::istream &is, const uistring &ui );

	~uistring(){};
};

//! User info for vectors
template<class T>
class uivector : public uibase {
protected:	
	itpp::Vec<T> V;
public:
	void getsummary ( std::string &S ) { S="Vector of length "+ V.length();};

//!Default constructor
	uivector ( std::string com ) :uibase ( com ) {};

	void askuser (){};
	//! saving the info
template<class T2>
	friend std::ostream &operator<< ( std::ostream &os, const uivector<T2> &ui );

//! saving the info
template<class T2>
	friend std::istream &operator>> ( std::istream &is, const uivector<T2> &ui );

};

//! User info for matrices
template<class T>
class uimatrix : public uibase {
	itpp::Mat<T> M;
	
	//!Default constructor
	uimatrix ( std::string com ) :uibase ( com ) {}

	void getsummary ( std::string &S ) {sprintf ( S,"Matrix %dx%d",M.rows(),M.cols() );};

	void askuser (){};
//! saving the info
template<class T2>
	friend std::ostream &operator<< ( std::ostream &os, const uimatrix<T2> &ui );

//! saving the info
template<class T2>
	friend std::istream &operator>> ( std::istream &is, const uimatrix<T2> &ui );

};

//!Compound user info
class uicompound: public uibase {
protected:
	uibase** elems;
public:
//!Default constructor
	uicompound ( const int n0 ) : elems(new uibase*[n0]) {};
	~uicompound(){delete elems;}
};

typedef uimatrix<double> uimat;
typedef uimatrix<int> uiimat;

typedef uivector<double> uivec;
typedef uivector<int> uiivec;


template<class T>
std::istream &operator>> ( std::istream &is, const uiscalar<T> &ui ) {is>>ui.N;return is;};
template<class T>
std::ostream &operator<< ( std::ostream &os, const uiscalar<T> &ui ) {os<<ui.N;return os;};

template<class T>
std::istream &operator>> ( std::istream &is, const uivector<T> &ui ) {is>>ui.V;return is;};

template<class T>
std::ostream &operator<< ( std::ostream &os, const uivector<T> &ui ) {os<<ui.V; return os;};

template<class T>
std::istream &operator>> ( std::istream &is, const uimatrix<T> &ui ) {is>>ui.M;return is;};

template<class T>
std::ostream &operator<< ( std::ostream &os, const uimatrix<T> &ui ) {os<<ui.M;return os;};

