/*!
  \file
  \brief Bayesian Models (bm) that use Bayes rule to learn from observations
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef BM_H
#define BM_H

#include <itpp/itbase.h>
//#include <std>

using namespace itpp;

/*!
* \brief Class representing variables, most often random variables

* More?...
*/
class RV {
	int len;
	ivec ids;
	ivec sizes;
	ivec times;
	ivec obs;
	Array<std::string> names;

private:
	void init ( ivec in_ids, Array<std::string> in_names, ivec in_sizes, ivec in_times, ivec in_obs );
public:
	//! Full constructor which is called by the others
	RV ( ivec in_ids, Array<std::string> in_names, ivec in_sizes, ivec in_times, ivec in_obs );
	//! default constructor
	RV ( ivec ids );
	//! Printing output e.g. for debugging.
	friend std::ostream &operator<< ( std::ostream &os, const RV &rv );

	//! Find indexes of another rv in self
	ivec rvfind(RV rv2);
	//! Add (concat) another variable to the current one
	RV rvadd(RV rv2);
	//! Subtract  another variable from the current one
	RV rvsubt(RV rv2);
	//! Select only variables at indeces ind
	RV rvsubselect(ivec ind);
	//! Select only variables at indeces ind
	RV operator()(ivec ind);
};




//! Class representing function of variables
class fnc {
	RV rv;
};

//! Bayesian Model of the world, i.e. all uncertainty is modeled by probabilities.
class BM {
public:
	//!Logarithm of marginalized data likelihood.
	double ll;

	/*! \brief Incremental Bayes rule
	@param dt vector of input data
	@param evall If true, the filter will compute likelihood of the data record and store it in \c ll
	*/
	virtual void bayes ( const vec &dt, bool evall=true ) = 0;
	//! Batch Bayes rule (columns of Dt are observations)
	void bayes ( mat Dt );
};

//! Probability density function with numerical statistics, e.g. posterior density.
class epdf {
	RV rv;
public:
	//! Returns the required moment of the epdf
	virtual vec moment ( const int order = 1 );
};

//! Conditional probability density, e.g. modeling some dependencies.
class mpdf {
	//! modeled random variable
	RV rv;
	//! random variable in condition
	RV rvc;
public:

	//! Returns the required moment of the epdf
	virtual fnc moment ( const int order = 1 );
};

#endif // BM_H
