#include "discrete.h"

namespace bdm {

void rectangular_support::set_parameters ( const Array<vec> &ranges0, const ivec &gridsize0 ) {
	ranges = ranges0;
	gridsizes = gridsize0;
	initialize();
}


void rectangular_support::initialize() {
	dim = ranges.length();
	bdm_assert ( gridsizes.length() == dim, "Incompatible dimensions of input" );
	Npoints = prod ( gridsizes );
	bdm_assert ( Npoints > 0, "Wrong input parameters" );

	//precompute steps
	steps.set_size ( dim );
	for ( int j = 0; j < dim; j++ ) {
		steps ( j ) = ( ranges ( j ) ( 1 ) - ranges ( j ) ( 0 ) ) / gridsizes ( j );
	}
	actvec.set_size ( dim );
	actvec_ind.set_size ( dim );
}

vec rectangular_support::get_vec ( const ivec &inds ) {
	vec v ( dim );
	for ( int j = 0; j < dim; j++ ) {
		bdm_assert_debug ( inds ( j ) < gridsizes ( j ), "Index out of bounds" );
		v ( j ) = ranges ( j ) ( 0 ) + ( 0.5 + inds ( j ) ) * steps ( j );
	}
	return v;
}

long rectangular_support::linear_index ( const ivec inds ) {
	long ind = 0;
	bdm_assert_debug ( inds.length() == dim, "Improper indices inds" );
	int dim_skips = 1; // skips in active dimension, in the first dimension, the skips are 1 index per value

	for ( int j = 0; j < dim; j++ ) {
		ind += dim_skips * ( inds ( j ) ); // add shift in linear index caused by this dimension
		dim_skips *= gridsizes ( j );  // indeces in the next dimension are repeated with period gridsizes(j) times greater that in this dimesion
	}
	return ind;
}
const vec& rectangular_support::first_vec() {
	for ( int j = 0; j < dim; j++ ) {
		actvec ( j ) = ranges ( j ) ( 0 ) + 0.5 * steps ( j );
		actvec_ind ( j ) = 0;
	}
	return actvec;
}
const vec& rectangular_support::next_vec() {
	// go through all dimensions
	int j = 0;
	while ( j < dim ) {
		if ( actvec_ind ( j ) == gridsizes ( j ) - 1 ) { //j-th index is full
			actvec_ind ( j ) = 0; //shift back
			actvec ( j ) = ranges ( j ) ( 0 ) + 0.5 * steps ( j );
			j++;
		} else {
			actvec_ind ( j ) ++;
			actvec ( j ) += steps ( j );
			break;
		}
	}
	return actvec;
}


ivec rectangular_support::nearest_point ( const vec &val ) {
	ivec inds;
	inds.set_size ( dim );
	for ( int j = 0; j < dim; j++ ) {
		if ( val ( j ) < ranges ( j ) ( 0 ) )
			inds ( j ) = 0;
		else {
			if ( val ( j ) > ranges ( j ) ( 1 ) )
				//! \todo GRIDSIZES prejmenovat..
				inds ( j ) = gridsizes ( j ) - 1;
			else {
				inds ( j ) = (int) ::round ( val ( j ) - ranges ( j ) ( 0 ) / steps ( j ) );
			}
		}
	}
	return inds;
}

void rectangular_support::from_setting ( const Setting &set ) {
	UI::get ( ranges , set, "ranges", UI::compulsory );
	UI::get ( gridsizes, set, "gridsizes", UI::compulsory );
	initialize();
}



void discrete_support::from_setting ( const Setting &set ) {
	UI::get ( Spoints, set, "points", UI::optional );
	if ( points() < 1 ) {
		int npoints;
		shared_ptr<epdf> ep = UI::build<epdf> ( set, "epdf", UI::compulsory );
		if ( !UI::get ( npoints, set, "npoints", UI::optional ) ) {
			npoints = 100;
		}

		//sample
		Spoints.set_size ( npoints );
		for ( int i = 0; i < points(); i++ ) {
			Spoints ( i ) = ep->sample();
		}
	}
}

void grid_fnc::set_values ( double ( *evalptr ) ( const vec& ) ) {
	if ( sup.points() > 0 ) {
		values ( 0 ) = ( *evalptr ) ( sup.first_vec() );
		for ( int j = 1; j < sup.points(); j++ ) {
			values ( j ) = ( *evalptr ) ( sup.next_vec() );
		}
	}
}

void grid_fnc::set_values ( const epdf &ep ) {
	if ( sup.points() > 0 ) {
		values ( 0 ) =  exp ( ep.evallog ( sup.first_vec() ) );
		for ( int j = 1; j < sup.points(); j++ ) {
			values ( j ) = exp ( ep.evallog ( sup.next_vec() ) );
		}
	}
}

}
