/*!
  \file
  \brief Bayesian Models (bm) that use Bayes rule to learn from observations
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef BM_H
#define BM_H

#include <itpp/itbase.h>
//#include <std>

using namespace itpp;

/*!
* \brief Class representing variables, most often random variables

* More?...
*/

class RV {
protected:
	//! size = sum of sizes
	int tsize;
	//! len = number of individual rvs
	int len;
	//! Vector of unique IDs 
	ivec ids;
	//! Vector of sizes
	ivec sizes;
	//! Vector of shifts from current time
	ivec times;
	//! Array of names
	Array<std::string> names;

private:
	//! auxiliary function used in constructor
	void init ( ivec in_ids, Array<std::string> in_names, ivec in_sizes, ivec in_times);
public:
	//! Full constructor which is called by the others
	RV ( ivec in_ids, Array<std::string> in_names, ivec in_sizes, ivec in_times);
	//! default constructor
	RV ( ivec ids );
	//! Empty constructor will be set later
	RV ();

	//! Printing output e.g. for debugging.
	friend std::ostream &operator<< ( std::ostream &os, const RV &rv );

	//! Return number of scalars in the RV.
	int count() const {return tsize;} ;
	//! Return length (number of entries) of the RV.
	int length() const {return len;} ;

	//TODO why not inline and later??

	//! Find indexes of another rv in self
	ivec find ( RV rv2 );
	//! Add (concat) another variable to the current one
 	void add (const RV &rv2 );
	//! Add (concat) another variable to the current one
	friend RV concat (const RV &rv1, const RV &rv2 );
	//! Subtract  another variable from the current one
	RV subt ( RV rv2 );
	//! Select only variables at indeces ind
	RV subselect ( ivec ind );
	//! Select only variables at indeces ind
	RV operator() ( ivec ind );
	//! Generate new \c RV with \c time shifted by delta.
	void t ( int delta );
	//! generate a list of indeces, i.e. which
	ivec indexlist();

	//!access function
	Array<std::string>& _names(){return names;};

	//!access function
	int id(int at){return ids(at);};
	//!access function
	int size(int at){return sizes(at);};
	//!access function
	int time(int at){return times(at);};
	//!access function
	std::string name(int at){return names(at);};
};


//! Class representing function \f$f(x)\f$ of variable \f$x\f$ represented by \c rv

class fnc {
protected:
	//! Length of the output vector
	int dimy;
public:
	//!default constructor
	fnc(int dy):dimy(dy){};
	//! function evaluates numerical value of \f$f(x)\f$ at \f$x=\f$ \c cond
	virtual vec eval ( const vec &cond ) {
		return vec ( 0 );
	}; 

	//! access function
	int _dimy() const{return dimy;}

	//! Destructor for future use;
	virtual ~fnc() {};
};


//! Probability density function with numerical statistics, e.g. posterior density.

class epdf {
protected:
	//! Identified of the random variable
	RV rv;
public:
	//!default constructor
	epdf() :rv ( ivec ( 0 ) ) {};

	//!default constructor
	epdf ( const RV &rv0 ) :rv ( rv0 ) {};

	//! Returns the required moment of the epdf
//	virtual vec moment ( const int order = 1 );
	//! Returns a sample from the density, \f$x \sim epdf(rv)\f$
	virtual vec sample () const =0;
	//! Compute probability of argument \c val
	virtual double eval ( const vec &val ) const {return exp(this->evalpdflog(val));};

	//! Compute log-probability of argument \c val
	virtual double evalpdflog ( const vec &val ) const =0;

	//! return expected value 
	virtual vec mean() const =0;
	
	//! Destructor for future use;
	virtual ~epdf() {};
	//! access function
	RV _rv() const {return rv;}
};


//! Conditional probability density, e.g. modeling some dependencies.
//TODO Samplecond can be generalized

class mpdf {
protected:
	//! modeled random variable
	RV rv;
	//! random variable in condition
	RV rvc;
	//! pointer to internal epdf
	epdf* ep;
public:

	//! Returns the required moment of the epdf
//	virtual fnc moment ( const int order = 1 );
	//! Returns a sample from the density conditioned on \c cond, \f$x \sim epdf(rv|cond)\f$. \param cond is numeric value of \c rv \param ll is a return value of log-likelihood of the sample.
	virtual vec samplecond ( vec &cond, double &ll ) {this->condition(cond);vec temp= ep->sample();ll=ep->evalpdflog(temp);return temp;};
	//! Update \c ep so that it represents this mpdf conditioned on \c rvc = cond
	virtual void condition ( const vec &cond ) {};
	
	//! Shortcut for conditioning and evaluation of the internal epdf. In some cases,  this operation can be implemented efficiently.
	virtual double evalcond (const vec &dt, const vec &cond ) {this->condition(cond);return ep->eval(dt);};

	//! Destructor for future use;
	virtual ~mpdf() {};

	//! Default constructor
	mpdf ( const RV &rv0, const RV &rvc0 ) :rv ( rv0 ),rvc ( rvc0 ) {};
	//! access function
	RV _rvc(){return rvc;}
	//!access function
	epdf& _epdf(){return *ep;}
};

/*! \brief Abstract class for discrete-time sources of data.

The class abstracts operations of: (i) data aquisition, (ii) data-preprocessing, (iii) scaling of data, and (iv) data resampling from the task of estimation and control.
Moreover, for controlled systems, it is able to receive the desired control action and perform it in the next step. (Or as soon as possible).

*/

class DS {
protected:
	//!Observed variables, returned by \c getdata().
	RV Drv;
	//!Action variables, accepted by \c write().
	RV Urv; //
public:
	//! Returns full vector of observed data
	void getdata ( vec &dt );
	//! Returns data records at indeces.
	void getdata ( vec &dt, ivec &indeces );
	//! Accepts action variable and schedule it for application.
	void write ( vec &ut );
	//! Accepts action variables at specific indeces
	void write ( vec &ut, ivec &indeces );
	/*! \brief Method that assigns random variables to the datasource.
	Typically, the datasource will be constructed without knowledge of random variables. This method will associate existing variables with RVs.

	(Inherited from m3k, may be deprecated soon).
	*/
	void linkrvs ( RV &drv, RV &urv );

	//! Moves from \f$t\f$ to \f$t+1\f$, i.e. perfroms the actions and reads response of the system.
	void step();

};

/*! \brief Bayesian Model of the world, i.e. all uncertainty is modeled by probabilities.

*/

class BM {
protected:
	//!Random variable of the posterior
	RV rv;
	//!Logarithm of marginalized data likelihood.
 double ll;
	//!  If true, the filter will compute likelihood of the data record and store it in \c ll . Set to false if you want to save time.
	bool evalll;
public:

	//!Default constructor
	BM(const RV &rv0) :rv(rv0), ll ( 0 ),evalll ( true ) {//Fixme: test rv 
	};

	/*! \brief Incremental Bayes rule
	@param dt vector of input data
	*/
	virtual void bayes ( const vec &dt ) = 0;
	//! Batch Bayes rule (columns of Dt are observations)
	void bayes ( mat Dt );
	//! Returns a pointer to the epdf representing posterior density on parameters. Use with care!
	virtual epdf& _epdf()=0;

	//! Destructor for future use;
	virtual ~BM() {};
	//!access function
	const RV& _rv() const {return rv;}
	//!access function
	double _ll() const {return ll;}
};

/*!
\brief Conditional Bayesian Filter

Evaluates conditional filtering density \f$f(rv|rvc,data)\f$ for a given \c rvc which is specified in each step by calling function \c condition.

This is an interface class used to assure that certain BM has operation \c condition .

*/

class BMcond {
protected:
	//! Identificator of the conditioning variable
	RV rvc;
public:
	//! Substitute \c val for \c rvc.
	virtual void condition ( const vec &val ) =0;
	//! Default constructor
	BMcond(RV &rv0):rvc(rv0){};
	//! Destructor for future use
	virtual ~BMcond(){};
	//! access function
	const RV& _rvc() const {return rvc;}
};

#endif // BM_H
