/*!
\file
\brief Application Estimator

The general task of estimation is defined on the following scheme:
\dot
digraph estimation{
	node [shape=box];
	subgraph cl0 {
	"Data Source" -> "Controller" [label="observations"];
	"Controller" -> "Data Source" [label="actions"];
	}
	{rank="same"; "Controller"; "Result Logger"}
	"Controller" -> "Result Logger" [label="internals"];
	"Data Source" -> "Result Logger" [label="Simulated\n data"];
}
\enddot

Here,
\li Data Source is an object (class DS) providing sequential data, \f$ [d_1, d_2, \ldots d_t] \f$.
\li Bayesian Model is an object (class BM) performing Bayesian filtering,
\li Result Logger is an object (class logger) dedicated to storing important data from the experiment.

\section  cmd Command-line usage
Execute command:
\code
$> estimator config_file.cfg
\endcode

Full description of the experiment is in the file config_file.cfg which is expected to have the following structure:
\code
system = {type = "DS_offspring", ...};      // definition of a data source
estimator = {type = "BM_offspring", ...};   // definition of an estimator
logger = {type = "logger_type",...};        // definition of a logger
experiment = {ndat = 11000; };              // definition of number of data records
\endcode

The above description must be specialized to specific classes. See, \subpage arx_ui how to do it for estimation of an ARX model.

\section ex Matlab usage
Execute command:
\code
>> estimator('config_file.cfg');
\endcode
when using loggers storing results on hard drives, and
\code
>> Res=estimator('config_file.cfg');
\endcode
when using logger of the type \c "mex_logger". The results will be stored in structure \c M.

 */

#include <estim/arx.h>
#include <stat/emix.h>
#include <base/datasources.h>
#include <base/loggers.h>
#include <design/arx_ctrl.h>

//#include "mex_datasource.h"

using namespace bdm;

#ifdef MEX
#include <itpp/itmex.h>
#include <mex/mex_BM.h>
#include <mex/mex_logger.h>
#include <mex/mex_datasource.h>
#include <mex/mex_function.h>

void mexFunction ( int n_output, mxArray *output[], int n_input, const mxArray *input[] ) {
	// Check the number of inputs and output arguments
	if ( n_input<2 ) mexErrMsgTxt ( "Usage:\n"
		                                "result=controlloop(system, controllers, experiment, logger)\n"
		                                "  system     = struct('class','datasource',...);  % Estimated system\n"
		                                "  controllers= {struct('class','controller',...),  % Controllers\n"
		                                "                struct('class','controller',...),...} \n"
		                                "  === optional ==="
		                                "  experiment = struct('ndat',100,...               % number of data in experiment, full length of finite datasources, 100 otherwise \n"
		                                "               'burnin',10,...                    % initial time with different control\n"
		                                "               'burn_pdf', struct('class','epdf_offspring') );\n"
		                                "                                                  % sampler of the initial control\n"
		                                "  logger     = struct('class','mexlogger');       % How to store results, default=mexlog, i.e. matlab structure\n\n"
		                                "see documentation of classes datasource, BM, and mexlogger and their offsprings in BDM." );

	RV::clear_all();
	//CONFIG
	UImxArray Cfg;
	try {
		Cfg.addGroup ( input[0],"system" );
		Cfg.addList ( input[1],"controllers" );
		if ( n_input>2 ) {
			Cfg.addGroup ( input[2],"experiment" );
		}
		if ( n_input>3 ) {
			Cfg.addGroup ( input[3],"logger" );
		}
	} catch ( SettingException e ) {
		it_error ( "error: "+string ( e.getPath() ) );
	}

	//DBG
// 	Cfg.writeFile ( "controlloop.cfg" );

#else
int main ( int argc, char* argv[] ) {
	const char *fname;
	if ( argc>1 ) {
		fname = argv[1];
	} else {
		fname="controlloop.cfg";
	}
	UIFile Cfg ( fname );
#endif

	RNG_randomize();
	
	shared_ptr<DS> Ds = UI::build<DS> ( Cfg, "system" );
	Array<shared_ptr<Controller> > Cs;
	UI::get ( Cs,Cfg, "controllers" );
	int Ndat=100;
	int burnin=0;
	shared_ptr<epdf> burn_pdf; 
	
	if ( Cfg.exists ( "experiment" ) ) {
		Setting &exper=Cfg.getRoot()["experiment"];
		if (UI::get(Ndat, exper, "Ndat", UI::optional ) ) {
			bdm_assert ( Ndat<=Ds->max_length(), "Data source has less data then required" );
		};
		if (UI::get(burnin, exper, "burnin",UI::optional )){
			burn_pdf = UI::build<epdf>(exper,"burn_pdf", UI::compulsory);
			if (burn_pdf){
				bdm_assert(burn_pdf->dimension()==Ds->_urv()._dsize(),"Given burn_pdf does not match the DataSource");
			} else {
				bdm_error("burn_pdf not specified!");
			}
			
		}
	} else {
		if ( Ds->max_length() < std::numeric_limits< int >::max() ) {
			Ndat=Ds->max_length();
		}
		;// else Ndat=10;
	}
	shared_ptr<logger> L = UI::build<logger> ( Cfg, "logger",UI::optional );
	if ( !L ) {
#ifdef MEX
		//mex logger has only from_setting constructor - we  have to fill it...
		L=new mexlog ( Ndat );
#else
		L=new stdlog();
#endif
	}

	Ds->log_register ( *L, "DS" );
	bdm_assert((Ds->_urv()._dsize() > 0), "Given DataSource is not controllable");
	string Cname;
	Setting &S=Cfg;
	for ( int i=0; i<Cs.length(); i++ ) {
		if (!UI::get ( Cname, S["controllers"][i], "name" ,UI::optional)){
			Cname="Ctrl"+num2str ( i );
		}
		
		Cs ( i )->log_register ( *L,Cname ); // estimate
	}
	L->init();

	vec dt=zeros ( Ds->_drv()._dsize() );   //data variable
	Array<datalink_part*> Dlsu ( Cs.length() );
	Array<datalink*> Dlsc ( Cs.length() );
	Array<datalink_buffered*> Dls_buf (0);
	for ( int i=0; i<Cs.length(); i++ ) {
		//connect actual data
		Dlsu ( i ) = new datalink_part;
		Dlsu(i)->set_connection( Ds->_urv(), Cs ( i )->_rv()); //datalink controller -> datasource
		//connect data in condition: datasource -> controller
		if (Cs ( i )->_rvc().mint()<0){ 
			//delayed values are required
			
			//create delayed dl
			int ith_buf=Dls_buf.size();
			Dls_buf.set_size( ith_buf + 1, true);
			Dls_buf(ith_buf) = new datalink_buffered(); 
			//add dl to list of buffered DS
			Dlsc(i) = Dls_buf(ith_buf);
			Dlsc(i)->set_connection ( Cs ( i )->_rvc(),Ds->_drv() ); //datalink between a datasource and estimator
			
		} else {
			Dlsc ( i ) = new datalink ( Cs ( i )->_rvc(),Ds->_drv() ); //datalink between a datasource and estimator
		}
	}

	vec ut(Ds->_urv()._dsize());
	for ( int tK=0;tK<Ndat;tK++ ) {
		Ds->getdata ( dt );					// read data
		Ds->log_write ( );

		for ( int i=0; i<Cs.length(); i++ ) {
			if (tK + Cs ( i )->_rvc().mint() > 0 ) {
				Cs(i) -> redesign();
				Cs(i) -> adapt( Dlsc(i) ->pushdown(dt));
				if (tK >= burnin){
					vec uti=Cs ( i )->ctrlaction ( Dlsc(i) ->pushdown(dt) );		// update estimates
					Dlsu(i)->filldown(uti, ut);
				}
			}
			if(tK<burnin) {
				ut = burn_pdf->sample();
			}
			
			Cs ( i )->log_write ();
		}
		Ds->write(ut);
		
		L->step();
		Ds->step();							// simulator step
		//update buffered fdat links
		for (int i=0; i<Dls_buf.length(); i++){
			Dls_buf(i)->store_data(dt);
		}
			
	}

	L->finalize();
	// ------------------ End of routine -----------------------------

#ifdef MEX
	mexlog* mL=dynamic_cast<mexlog*> ( L.get() );

	if ( mL ) { // user wants output!!
		if ( n_output<1 ) mexErrMsgTxt ( "Wrong number of output variables!" );
		output[0] = mL->toCell();
	}
#endif
	for (int i=0;i<Dlsu.length(); i++){delete Dlsu(i); delete Dlsc(i);}
}
