//
// C++ Interface: userinfo
//
// Description:
//
//
// Author: smidl <smidl@utia.cas.cz>, (C) 2008
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include <itpp/itbase.h>
#include <itpp/base/mat.h>

using std::cout;
using std::endl;
using std::string;

/*!
@brief User Info base class

This class is used to store information about parameters of an object. It support loading and saving of the information and, potentially, interaction with the user.
*/
class uibase {
protected:
	//! String identifier of a field
	string name;
	//! Explanation for a user what the field means
	string help;
	//! Possible parent of the userinfo
	uibase* parent;
	//! Indentation level, i.e. number of parents
	int ilevel;
public:
//!Default constructor
	uibase ( string com = "Abstract class, please ignore!", uibase* par=NULL ) :name ( com ),help ( "" ),parent ( par ) {
		if ( parent!=NULL ) {ilevel=parent->get_level() +1;}
		else {ilevel =0;}
	}

//! returns a summary of its contents (used in dialogs)
	virtual void getsummary ( std::string &S ) {};

//! interaction with the user
	virtual void askuser() {};

//! test if the info is valid
	virtual bool isvalid() {return true;}
	//! for future use
	virtual ~uibase() {};

	virtual void save ( std::ostream &os ) {
		os.width(ilevel);
		os.fill(' ');
		os<<"#"<<help<<endl;
		os.width(ilevel);
		os.fill(' ');
		os<<name<<" = ";
	};
	virtual void load ( std::istream &is ) {
		char tmp[200];
		is.ignore ( ilevel+1,'#' ); // +1 is for #
		is.getline ( tmp,200 );help=tmp;
		is.ignore ( ilevel,'\0' );
		is.getline ( tmp,200,'=' ); name=tmp;
	};
	
	//!access function
	int get_level(){return ilevel;}
};

//! User info for scalars
template<class T>
class uiscalar : public uibase {
protected:
	T N;
public:
//!Default constructor
	uiscalar ( std::string com, uibase* par =NULL) :uibase ( com,par ) {N=T ( 0 );};

	void getsummary ( std::string &S ) {S="Scalar";};

	void askuser () {};

	void save ( std::ostream &os ) {uibase::save ( os );os<<N<<endl;}
	void load ( std::istream &is ) {uibase::load ( is );is>>N;}
	//! for future use
	~uiscalar() {};

	//! access function
	void set_value ( T N0 ) {N=N0;}
};

//! User info for strings
class uistring : public uibase {
protected:
	std::string S;
public:
	void getsummary ( std::string &S ) {S="String";};

//!Default constructor
	uistring ( std::string com , uibase* par=NULL) :uibase ( com,par ) {}

	void askuser () {};

	void save ( std::ostream &os ) {uibase::save ( os );os<<S<<endl;}
	void load ( std::istream &is ) {uibase::load ( is );is>>S;}

	~uistring() {};
	//! access function
	void set_value ( std::string S0 ) {S=S0;}
};

//! User info for vectors
template<class T>
class uivector : public uibase {
protected:
	itpp::Vec<T> V;
public:
	void getsummary ( std::string &S ) { S="Vector of length "+ V.length();};

//!Default constructor
	uivector ( std::string com, uibase* par=NULL ) :uibase ( com,par ) {};

	void askuser () {};

	void save ( std::ostream &os ) {uibase::save ( os );os<<V<<endl;;}
	void load ( std::istream &is ) {uibase::load ( is );is>>V;}

	//! access function
	void set_value ( itpp::Vec<T> V0 ) {V=V0;}

};

//! User info for matrices
template<class T>
class uimatrix : public uibase {
protected:
	itpp::Mat<T> M;

public:
	//!Default constructor
	uimatrix ( std::string com, uibase* par=NULL ) :uibase ( com,par ) {}

	void getsummary ( std::string &S ) { S="Matrix ";};

	void askuser () {};

	void save ( std::ostream &os ) {uibase::save ( os );os<<M<<endl;}
	void load ( std::istream &is ) {uibase::load ( is );is>>M;}

	//! access function
	void set_value ( itpp::Mat<T> M0 ) {M=M0;}
};


typedef uimatrix<double> uimat;
typedef uimatrix<int> uiimat;

typedef uivector<double> uivec;
typedef uivector<int> uiivec;

