/*!
  \file
  \brief Loggers for storing results of experiments
  \author Vaclav Smidl.

  -----------------------------------
  BDM++ - C++ library for Bayesian Decision Making under Uncertainty

  Using IT++ for numerical operations
  -----------------------------------
*/

#ifndef LGR_H
#define LGR_H

#include <itpp/itbase.h>
#include "libBM.h"
#include "../itpp_ext.h"

using namespace itpp;
using std::string;

/*!
@brief Class for storing results (and semi-results) of an experiment

This class abstracts logging of results from implementation. This class replaces direct logging of results (e.g. to files or to global variables) by calling methods of a logger. Specializations of this abstract class for specific storage method are designed.
*/
class logger {
protected:
	//! RVs of all logged variables. 
	Array<RV> entries;
	//! Names of logged quantities, e.g. names of algorithm variants
	Array<string> names;
public:
//!Default constructor
	logger ( ) : entries(0),names ( 0 ) {}

//! returns an identifier which will be later needed for calling the log() function
	virtual int add ( RV rv, string name="" ) {
		int id=entries.length();
		names=concat ( names, name ); // diff
		entries.set_length(id+1,true);
		entries(id)= rv;
		return id; // identifier of the last entry
	}

//! log this vector
	virtual void logit ( int id, vec v ) =0;

//! consider this record to be complete
	virtual void step(bool final=false) =0;

	//! for future use
	virtual ~logger() {};
};


/*!
* \brief Logging into matrices in data format in memory

* More?...
*/

class memlog : public logger {

protected:
	//! Maximum length of vectors stored in memory
	int maxlen;
	//! Currect record to be written
	int ind;
	//! Storage
	Array<mat> vectors;
	//!
public:
	//!Default constructor
	memlog ( int maxlen0 ) : logger(),maxlen ( maxlen0 ),ind ( 0 ),vectors ( 0 ) {}
	//! Initialize storage
	void init() {
		int i; int n =entries.length();
		vectors.set_size ( n ); 
		for ( i=0;i<n;i++ ) {vectors(i).set_size (maxlen,entries(i).count() );}
		;
	}
	void step(bool final=false) {if ( ind<maxlen ) ind++; else it_error ( "memlog::ind is too high;" );}
	void logit ( int id, vec v ) {vectors ( id ).set_row ( ind,v );}
};

/*!
* \brief Logging into dirfile with buffer in memory

* Dirfile is a special format used by the kst program. See documentation of kst for description.
*
* This format is used to store scalars, hence multivariate RVs must be separated.
*/

class dirfilelog : public memlog {

protected:
	//!name of the directory
	string dirname;
	//! Automatically generated 
	Array<string> scalarnames;
public:
	/*!\brief Default constructor 
	@param dirname0 name of the directory in which to store the results
	@param maxlen0 length of the memory buffers, when full the buffers will be dumped to HDD and returned to the beginning. */
	dirfilelog ( std::string dirname0, int maxlen0 ) : memlog ( maxlen0 ), dirname ( dirname0 ), scalarnames ( 0 ) {}
	//! Initialize storage
	void init();
	void step(bool final=false);
	/*! \brief Write memory storage to disk. 
	@param Len length of buffer to be written, if 0 the file is truncated at 0.
	*/
	void write_buffers ( int Len );
};


#endif // LGR_H
